/****************************************
*********** Author: Burryaga ************
*****************************************/

/*
 * For Use in Mod Builder Version 2.1.2
 */

using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System.IO;
using System.Text;

using nms			= libMBIN.NMS;
using globals		= libMBIN.NMS.Globals;
using components	= libMBIN.NMS.GameComponents;
using toolkit		= libMBIN.NMS.Toolkit;

//=============================================================================

namespace cmk.NMS.Scripts.Mod
{
	using CoverageTypeEnum = components.GcSpawnDensity.CoverageTypeEnum;
	using PlacementPriorityEnum = components.GcObjectSpawnData.PlacementPriorityEnum;
	using OverlapStyleEnum = components.GcObjectSpawnData.OverlapStyleEnum;
	using SkyString = nms.NMSString0x10;

	public class MorePlants : cmk.NMS.ModScript
	{

		// If the forests are too much for you, set this to false.
		protected bool IAmTheLorax = true;

		// Setting this to true will resize many objects to be larger. Different
		// objects are resized differently because some things scale better than others.
		protected bool ModifySize = true;
		
		// Setting these values to true will cause plants on each planet
		// to spawn as a variant instead (all plants of a type will change
		// to the same variant). THIS CAUSES NEW SPECIES OF FLORA TO APPEAR
		// ON YOUR PLANETS. THEIR EXISTENCE IN YOUR DISCOVERIES WILL REMAIN
		// EVEN AFTER DEACTIVATING MOD, THUS PREVENTING THEM FROM SPAWNING.
		// YOU HAVE BEEN WARNED. But I enjoy enabling these. :)
		protected bool ModifyColors = true;

		// These strings are used to check the Biome and determine whether
		// to mod coverage or density of specific objects. Only bubble presence
		// is reduced using these flags (simply to save frames on bubble worlds 
		// for better stuff. Sometimes, density is reduced so that coverage can increase, but the
		// overall presence of all flora will be much greater.
		protected string Biome = "NONE";
		protected string Grass = "GRASS";
		
		// Begins true because detail objects get augmented first. Tells
		// GenerateMore whether to use the SmallShrooms or LargeShrooms method.
		protected bool Detail = true;

		protected override void Execute()
		{
			PropagateMoreLife();
		}
		
		
		// The big one. Stores all OBJECTS**.MBIN locations in strings,
		// then puts them in a list and loops through many biome object sets,
		// though perhaps not all of them.
		protected void PropagateMoreLife()
		{

			var pathsForGcExternalObjectLists = GetPathsFor("GcExternalObjectList");

			foreach (var objectListPath in pathsForGcExternalObjectLists)
			{
				Biome = BiomeFromPath(objectListPath);
				GenerateMoreAwesome(objectListPath);
			}

		}

		// Landmarks, Objedcts, and DetailObjects are target separately because
		// they contain different kinds of similarly tagged objects, and it's easier
		// to produce differing effects on different classes of objects this way.
		// I could have also combined the objects from Landmarks, Objects, and DetailObjects
		// into a superlist and called one version of GenerateMore, and I will probably
		// update this code to work that way in the future.
		protected void GenerateMoreAwesome(string objectListPath)
		{
			var objectTable = Mbin<components.GcExternalObjectList>(objectListPath).Objects;
			
			var objects = new List<components.GcObjectSpawnData>();
			var details = objectTable.DetailObjects;
			
			objects.AddRange(objectTable.Landmarks);
			objects.AddRange(objectTable.Objects);
			
			GenerateMore(details);
			
			Detail = false;
			
			GenerateMore(objects);
		}

		// GenerateMore will receive a List of GcObjectSpawnData objects, including
		// DetailObjects, Landmarks, Objects.
		protected void GenerateMore(List<components.GcObjectSpawnData> objects)
		{

			// After extracting the list of objects, loop through each object, and adjust
			// Coverages and Densities on the basis of what kind of object we are currently
			// looking at. Some of this code may be unnecessary, but I made it a bit generic to
			// avoid reading every detail of every object list in every biome OBJECTS file.
			foreach (var element in objects)
			{
				var placement = element.Placement.Value;
				var filename = element.Resource.Filename.Value;

				if (Widespread(placement) && Cool(filename))
				{
					if (LovelyShrooms(filename))
					{
						if (Detail) {
							AdjustSmallShroomCoverage(element);
						} else {
							AdjustLargeShroomCoverage(element);
						}
					}
					if (LargeSpires(filename) && FloraClump(placement))
					{
						AdjustLargeSpireCoverage(element);
					}
					else if (CaveStuff(filename))
					{
						AdjustCaveCoverage(element);
					}
					else if (TraitorousBubbles(filename))
					{
						AdjustBubbleCoverage(element);
					}
					else if (FloraClump(placement) &&
							 ScrubGrass(filename) &&
							 SparseBiome(Biome))
					{
						AdjustSparseGrassCoverage(element);
					}
					else if (BeautifulGrass(filename))
					{
						AdjustNormalGrassCoverage(element);
					}
					else
					{
						AdjustMiscellaneousCoverage(element);
					}
				}
				else if (LushForest(placement, filename))
				{
					AdjustLushForestCoverage(element);
				}
				else if (SparseForest(placement, filename))
				{
					AdjustSparseForestCoverage(element);
				}
			}
		}
		
		/************************************************************************************/
		/***********************HELPER FUNCTIONS: COVERAGE ADJUSTMENTS***********************/
		/************************************************************************************/

		protected void AdjustLushForestCoverage(components.GcObjectSpawnData element)
		{
			element.OverlapStyle = OverlapStyleEnum.None;

			if (IAmTheLorax)
			{
				element.Placement.Value = Grass;
			}
			else
			{
				element.QualityVariantData.Coverage = .7f;

				foreach (var variant in element.QualityVariants)
				{
					variant.Coverage = .7f;
				}
			}

			// element.SwapPrimaryForRandomColour = ModifyColors;

			if (ModifySize)
			{
				element.MinScale *= 1.1f;
				element.MaxScale *= 1.5f;
			}

		}

		protected void AdjustSparseForestCoverage(components.GcObjectSpawnData element)
		{
			if (element.QualityVariantData.Coverage < .05f)
			{
				element.QualityVariantData.Coverage *= 3f;
			}
			else if (element.QualityVariantData.Coverage < .1f)
			{
				element.QualityVariantData.Coverage *= 2f;
			}
			else if (element.QualityVariantData.Coverage < .20f)
			{
				element.QualityVariantData.Coverage = .25f;
			}
			else if (element.QualityVariantData.Coverage < .30f)
			{
				element.QualityVariantData.Coverage = .7f;
			}

			element.QualityVariantData.FlatDensity /= 5.5f;

			foreach (var variant in element.QualityVariants)
			{
				if (variant.Coverage < .05f)
				{
					variant.Coverage *= 1.5f;
				}
				else if (variant.Coverage < .1f)
				{
					variant.Coverage *= 2f;
				}
				else if (variant.Coverage < .20f)
				{
					variant.Coverage = .25f;
				}
				else if (variant.Coverage < .30f)
				{
					variant.Coverage = .7f;
				}

				variant.FlatDensity /= 5.5f;
			}

			if (ModifySize)
			{
				element.MaxScale *= 1.1f;
				element.MaxScale *= 1.3f;
			}

			// element.SwapPrimaryForRandomColour = ModifyColors;
		}

		protected void AdjustMiscellaneousCoverage(components.GcObjectSpawnData element)
		{
			element.QualityVariantData.Coverage *= 1.1f;

			if (element.QualityVariantData.Coverage > .5f)
			{
				element.QualityVariantData.Coverage = .5f;
			}

			foreach (var variant in element.QualityVariants)
			{
				variant.Coverage *= 1.1f;

				if (variant.Coverage > .5f)
				{
					variant.Coverage = .5f;

				}
			}

			if (ModifySize)
			{
				element.MaxScale *= 1.1f;
			}
		}

		protected void AdjustBubbleCoverage(components.GcObjectSpawnData element)
		{
			element.QualityVariantData.Coverage *= .2f;
			element.QualityVariantData.FlatDensity *= .2f;
			
			element.Resource.Seed.Seed = 0;

			foreach (var variant in element.QualityVariants)
			{
				variant.Coverage *= .2f;
				variant.FlatDensity *= .2f;
			}
		}

		protected void AdjustNormalGrassCoverage(components.GcObjectSpawnData element)
		{
			element.SwapPrimaryForRandomColour = ModifyColors;

			if (ModifySize)
			{
				element.MaxScale *= 1.3f;
				element.MaxScaleY *= .85f;
			}
		}

		protected void AdjustSparseGrassCoverage(components.GcObjectSpawnData element)
		{
			if (element.QualityVariantData.Coverage >= .1f)
			{
				element.QualityVariantData.Coverage = .4f;
			}

			foreach (var variant in element.QualityVariants)
			{
				if (variant.Coverage >= .1f)
				{
					variant.Coverage = .4f;
				}
			}

			element.SwapPrimaryForRandomColour = ModifyColors;

			if (ModifySize)
			{
				element.MinScale *= 1.2f;
				element.MaxScale *= 1.8f;
				element.MinHeight *= .85f;
				element.MaxHeight *= .85f;
			}
		}

		protected void AdjustCaveCoverage(components.GcObjectSpawnData element)
		{
			if (element.QualityVariantData.Coverage < .1f)
			{
				element.QualityVariantData.Coverage = .2f;
			}
			else
			{
				element.QualityVariantData.Coverage = .35f;
			}

			foreach (var variant in element.QualityVariants)
			{
				if (variant.Coverage < .1f)
				{
					variant.Coverage = .15f;
				}
				else
				{
					variant.Coverage = .25f;
				}
			}

			element.SwapPrimaryForRandomColour = ModifyColors;

			if (ModifySize)
			{
				element.MinScale *= 1.1f;
				element.MaxScale *= 1.4f;
			}
		}

		protected void AdjustLargeSpireCoverage(components.GcObjectSpawnData element)
		{
			if (element.QualityVariantData.Coverage < .05f)
			{
				element.QualityVariantData.Coverage *= 3f;
			}
			else if (element.QualityVariantData.Coverage < .1f)
			{
				element.QualityVariantData.Coverage *= 2f;
			}
			else if (element.QualityVariantData.Coverage < .15f)
			{
				element.QualityVariantData.Coverage = .25f;
			}
			else if (element.QualityVariantData.Coverage < .35f)
			{
				element.QualityVariantData.Coverage = .35f;
			}

			foreach (var variant in element.QualityVariants)
			{
				if (variant.Coverage < .05f)
				{
					variant.Coverage *= 1.5f;
				}
				else if (variant.Coverage < .1f)
				{
					variant.Coverage *= 2f;
				}
				else if (variant.Coverage < .15f)
				{
					variant.Coverage = .25f;
				}
				else if (variant.Coverage < .35f)
				{
					variant.Coverage = .35f;
				}
			}

			element.SwapPrimaryForRandomColour = ModifyColors;

			if (ModifySize)
			{
				element.MinScale *= 1.5f;
				element.MaxScale *= 2f;
			}

		}

		protected void AdjustLargeShroomCoverage(components.GcObjectSpawnData element)
		{
			var placement = element.Placement.Value;
			
			if (element.QualityVariantData.Coverage < .05f)
			{
				element.QualityVariantData.Coverage = .25f;
				element.QualityVariantData.FlatDensity /= 2;

				if (ModifySize)
				{
					element.MinScale *= .8f;
					element.MaxScale *= 1.2f;
				}
			}
			else if (element.QualityVariantData.Coverage < .1f)
			{
				element.QualityVariantData.Coverage *= 2f;

				if (ModifySize)
				{
					element.MinScale *= .8f;
					element.MaxScale *= 1.2f;
				}
			}
			else if (element.QualityVariantData.Coverage < .15f)
			{
				element.QualityVariantData.Coverage = .25f;
				element.QualityVariantData.FlatDensity /= 2;

				if (ModifySize)
				{
					element.MinScale *= .8f;
					element.MaxScale *= 1.2f;
				}
			}
			else if (element.QualityVariantData.Coverage < .35f)
			{	// This is the stuff that targets large shroomtree objects.
				if (Biome.Equals("LUSH") && FloraClump(placement))
				{	
					element.OverlapStyle = OverlapStyleEnum.None;
					if (ModifySize)
					{
						if (IAmTheLorax) {
							element.QualityVariantData.Coverage = .95f;
						} else {
							element.QualityVariantData.Coverage = .65f;
						}
						element.QualityVariantData.FlatDensity *= .5f;

						element.MinScale *= 1.25f;
						element.MaxScale *= 2f;
					}
					else if (IAmTheLorax)
					{
						element.Placement.Value = Grass;
					}
					else
					{
						element.QualityVariantData.Coverage = .65f;
						element.QualityVariantData.FlatDensity *= .5f;
					}
				}
				else if (Biome.Equals("LUSH"))
				{
					element.OverlapStyle = OverlapStyleEnum.None;

					if (ModifySize)
					{
						if (IAmTheLorax) {
							element.QualityVariantData.Coverage = .95f;
						} else {
							element.QualityVariantData.Coverage = .65f;
						}
						element.QualityVariantData.FlatDensity *= .5f;

						element.MinScale *= 1.25f;
						element.MaxScale *= 2f;
					}
					else
					{
						if (IAmTheLorax) {
							element.QualityVariantData.Coverage = .95f;
						} else {
							element.QualityVariantData.Coverage = .65f;
						}
						element.QualityVariantData.FlatDensity *= .5f;
					}
				}
				else
				{
					if (ModifySize)
					{
						element.QualityVariantData.Coverage = .5f;

						element.MinScale *= 1.25f;
						element.MaxScale *= 2f;
					}
					else
					{
						element.QualityVariantData.Coverage = .7f;
					}
				}
			}

			foreach (var variant in element.QualityVariants)
			{
				if (variant.Coverage < .05f)
				{
					variant.Coverage = .25f;
					variant.FlatDensity /= 2;
				}
				else if (variant.Coverage < .1f)
				{
					variant.Coverage *= 2f;
				}
				else if (variant.Coverage < .15f)
				{
					variant.Coverage = .25f;
					variant.FlatDensity /= 2;
				}
				else if (variant.Coverage < .3f)
				{
					if (Biome.Equals("LUSH") && FloraClump(placement))
					{
						variant.Coverage = .7f;
					}
					else if (Biome.Equals("LUSH"))
					{
						variant.Coverage = .5f;
					}
					else
					{
						variant.Coverage = .35f;
					}
				}
			}

			element.SwapPrimaryForRandomColour = ModifyColors;
		}

		protected void AdjustSmallShroomCoverage(components.GcObjectSpawnData element)
		{
			if (element.QualityVariantData.Coverage < .05f)
			{
				element.QualityVariantData.Coverage *= 2f;
				element.QualityVariantData.FlatDensity /= 1.3f;
			}
			else if (element.QualityVariantData.Coverage < .1f)
			{
				element.QualityVariantData.Coverage *= 2f;
				element.QualityVariantData.FlatDensity /= 1.2f;
			}
			else if (element.QualityVariantData.Coverage < .25f)
			{
				element.QualityVariantData.Coverage *= 1.5f;
			}
			else if (element.QualityVariantData.Coverage < .4f)
			{
				element.QualityVariantData.Coverage *= 1.3f;
				element.QualityVariantData.FlatDensity /= 1.3f;
			}

			foreach (var variant in element.QualityVariants)
			{
				if (variant.Coverage < .05f)
				{
					variant.Coverage *= 2f;
					variant.FlatDensity /= 1.3f;
				}
				else if (variant.Coverage < .1f)
				{
					variant.Coverage *= 2.5f;
				}
				else if (variant.Coverage < .25f)
				{
					variant.Coverage *= 1.5f;
					variant.FlatDensity /= 1.3f;
				}
				else if (variant.Coverage < .4f)
				{
					variant.Coverage *= 1.3f;
					variant.FlatDensity /= 1.3f;
				}
			}

			element.SwapPrimaryForRandomColour = ModifyColors;

			if (ModifySize)
			{
				element.MinScale *= .8f;
				element.MaxScale *= 1.2f;
			}
		}

		/************************************************************************************/
		/*****************************HELPER FUNCTIONS: PARSERS******************************/
		/************************************************************************************/

		// Extract biome's name from file location.
		protected string BiomeFromPath(string filename)
		{
			if (filename.Contains("BIOMES/BARREN"))
			{
				return "BARREN";
			}
			else if (filename.Contains("BIOMES/CAVE"))
			{
				return "CAVE";
			}
			else if (filename.Contains("BIOMES/DEAD"))
			{
				return "DEAD";
			}
			else if (filename.Contains("BIOMES/FROZEN"))
			{
				return "FROZEN";
			}
			else if (filename.Contains("BIOMES/HUGE"))
			{
				return "HUGE";
			}
			else if (filename.Contains("BIOMES/LAVA"))
			{
				return "LAVA";
			}
			else if (filename.Contains("BIOMES/LUSH"))
			{
				return "LUSH";
			}
			else if (filename.Contains("BIOMES/OBJECTS"))
			{
				return "OBJECTS";
			}
			else if (filename.Contains("BIOMES/RADIOACTIVE"))
			{
				return "RADIOACTIVE";
			}
			else if (filename.Contains("BIOMES/SCORCHED"))
			{
				return "SCORCHED";
			}
			else if (filename.Contains("BIOMES/SWAMP"))
			{
				return "SWAMP";
			}
			else if (filename.Contains("BIOMES/TOXIC"))
			{
				return "TOXIC";
			}
			else if (filename.Contains("BIOMES/UNDERWATER"))
			{
				return "UNDERWATER";
			}
			else if (filename.Contains("BIOMES/WEIRD"))
			{
				return "WEIRD";
			}

			return "ERROR";

		}

		/************************************************************************************/
		/*****************************HELPER FUNCTIONS: CHECKS*******************************/
		/************************************************************************************/

		// All of these functions return true if certain conditions about the biome type and
		// filename are met. The filename is the most precise indicator of the object that will
		// spawn.

		protected bool BeautifulGrass(string filename)
		{
			return filename.Contains("LUSHGRASS") ||
					filename.Contains("CROSSGRASS");
		}

		protected bool CaveStuff(string filename)
		{
			return filename.Contains("SMALLGLOW") ||
					filename.Contains("MEDIUMGLOW") ||
					filename.Contains("HANGINGPLANT") ||
					filename.Contains("CEILINGPLANT") ||
					filename.Contains("SMALLPLANT") ||
					filename.Contains("CAVEBUSH");
		}

		protected bool FloraClump(string placement)
		{
			return placement.Contains("FLORACLUMP");
		}

		protected bool LargeSpires(string filename)
		{
			return filename.Contains("LARGESPIRE");
		}

		protected bool LovelyShrooms(string filename)
		{
			return filename.Contains("SHROOM") &&
					!filename.Contains("MEDIUMBLUESHROOM");
		}

		protected bool LushForest(string placement, string filename)
		{
			return placement.Contains("FOREST") &&
					filename.Contains("TREE") &&
					Biome.Equals("LUSH");
		}

		protected bool ScrubGrass(string filename)
		{
			return filename.Contains("SCRUBGRASS");
		}

		protected bool SparseBiome(string name)
		{
			return name.Equals("BARREN") ||
					name.Equals("CAVE") ||
					name.Equals("DEAD") ||
					name.Equals("SCORCHED");
		}

		protected bool SparseForest(string placement, string filename)
		{
			return placement.Contains("FOREST") &&
					filename.Contains("TREE") &&
					!Biome.Equals("LUSH");
		}

		protected bool TraitorousBubbles(string filename)
		{
			return filename.Contains("BUBBLE") && !filename.Contains("GRASS");
		}

		protected bool Widespread(string placement)
		{
			return placement.Contains("FLORACLUMP") ||
					placement.Contains("GRASSCLUM") ||
					placement.Contains("CAVEROCKCLUMP") ||
					placement.Contains("JAMESPATCH");
		}

		protected bool Cool(string filename)
		{
			return filename.Contains("SHROOM") ||
					filename.Contains("BUSH") ||
					filename.Contains("FLOWER") ||
					filename.Contains("SMALLGLOW") ||
					filename.Contains("MEDIUMGLOW") ||
					filename.Contains("HANGINGPLANT") ||
					filename.Contains("CEILINGPLANT") ||
					filename.Contains("MEDIUMPLANT") ||
					filename.Contains("CAVEBUSH") ||
					filename.Contains("LARGEFUNGHI") ||
					filename.Contains("BUBBLE") ||
					filename.Contains("SPIRE") ||
					filename.Contains("BEAM") ||
					filename.Contains("SHARD") ||
					filename.Contains("SHELL") ||
					filename.Contains("CORAL");
		}

		/************************************************************************************/
		/*****************************HELPER FUNCTIONS: CHECKS*******************************/
		/************************************************************************************/

		// This function makes summoning the PakItems of a GcObject cleaner.	
		protected List<string> GetPathsFor(string classInfoType)
		{

			return Game.Mbinc.Classes.Find(
				(targetClassInfo) => targetClassInfo.Type.Name.Equals(classInfoType)
			).PakItems;

		}
	}
}