//=============================================================================

using System;
using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Text;

using NoMansSky	= libMBIN.NMS;
using Globals		= libMBIN.NMS.Globals;
using GameComps = libMBIN.NMS.GameComponents;
using ToolKit		= libMBIN.NMS.Toolkit;

//=============================================================================

// the primary goal of It Takes A Village is to increase the depth and impact
// of building a settlement from the ground up, and to tune certain aspects of
// the existing system

namespace cmk.NMS.Scripts.Mod
{
	using GcSettlementGlobals = Globals.GcSettlementGlobals;
	
	public class ItTakesAVillage : cmk.NMS.ModScript
	{
		// config
		
		// maximum wait time between decisions
		public static int MaxJudgementWait = 3600;
		// chance a given judgement will be a building
		public static float BuildingJudgementChance = 0.75f;
		// maximum roaming characters in a settlement
		public static int MaxNPCs = 64;
		// multiplier for build cost
		public static int costMult = 10;
		// multiplier for build time
		public static ulong timeMult = 2;
		
		// settlement spawn frequency
		public static float SettleSpawnFreq = 0.05f;
		// which planets should settlements spawn on?
		public static List<bool> SettleSpawnPlanetType = new List<bool> {
			false,	// Dead
			true,		// Low
			true,		// Mid
			true,		// Full
			true,		// Weird
			true		// HalfWeird
		};
		
		// settlement maximum stats
		public static int[] StatsMaxValues =
			new int[7] {
			256,					// Population
			100,					// Happiness
			int.MaxValue,	// Production
			int.MaxValue, // Upkeep
			100,					// Sentinels
			65536,				// Debt
			100						// Alert
		};
		
		// new production sets for Vy'keen
		public static List<NoMansSky.NMSString0x10> WarProducts =
			new List<NoMansSky.NMSString0x10> {
				"WAR_CURIO1", 		// Vy'keen Effigy
				"WAR_CURIO2",			// Vy'keen Dagger
				"FOOD_P_RADWILD", // Grahberry
				"STELLAR2",				// Chromatic Metal
				"ROBOT1"					// Pugneum
			};
		// new production sets for Gek
		public static List<NoMansSky.NMSString0x10> TraProducts =
			new List<NoMansSky.NMSString0x10> {
				"TRA_CURIO1",			// Gek Relic
				"TRA_CURIO2",			// GekNip
				"FOOD_P_ALL3",		// Pulpy Roots
				"ASTEROID2",			// Gold
				"PLANT_POOP"			// Faecium
			};
		// new production sets for Korvax
		public static List<NoMansSky.NMSString0x10> ExpProducts =
			new List<NoMansSky.NMSString0x10> {
				"EXP_CURIO2",			// Convergence Cube
				"TECH_COMP",			// Wiring Loom
				"FOOD_P_GLITCH",	// Hexaberry
				"ASTEROID1",			// Silver
				"SPECIAL_POOP"		// Hexite
			};

		// new economy production sets
		// poor:
		public static List<NoMansSky.NMSString0x10> PoorProducts =
			new List<NoMansSky.NMSString0x10> {
				"TRA_ALLOY1",			// Nanotube Crate
				"TRA_ALLOY2",			// Self-Repairing Heridium
				"TRA_COMMODITY1",	// Decrypted User Data
				"TRA_COMMODITY2",	// Star Silk
				"TRA_COMPONENT1",	// Enormous Metal Cog
				"TRA_COMPONENT2",	// Non-Stick Piston
				"TRA_ENERGY1",		// Spark Canister
				"TRA_ENERGY2",		// Industrial-Grade Battery
				"TRA_EXOTICS1",		// De-Scented Pheromone Bottle
				"TRA_EXOTICS2",		// Neutron Microscope
				"TRA_MINERALS1",	// Dirt
				"TRA_MINERALS2",	// Unrefined Pyrite Grease
				"TRA_TECH1",			// Decommissioned Circuit Board
				"TRA_TECH2"				// Welding Soap
			};
		// average:
		public static List<NoMansSky.NMSString0x10> NeutProducts =
			new List<NoMansSky.NMSString0x10> {
				"TRA_ALLOY2",			// Self-Repairing Heridium
				"TRA_ALLOY3",			// Optical Solvent
				"TRA_ALLOY4",			// Five Dimensional Torus
				"TRA_COMMODITY2",	// Star Silk
				"TRA_COMMODITY3",	// Comet Droplets
				"TRA_COMMODITY4",	// Ion Sphere
				"TRA_COMPONENT2",	// Non-Stick Piston
				"TRA_COMPONENT3",	// Six-Pronged Mesh Decoupler
				"TRA_COMPONENT4",	// Holographic Crankshaft
				"TRA_ENERGY2",		// Industrial-Grade Battery
				"TRA_ENERGY3",		// Ohmic Gel
				"TRA_ENERGY4",		// Experimental Power Fluid
				"TRA_EXOTICS2",		// Neutron Microscope
				"TRA_EXOTICS3",		// Instability Injector
				"TRA_EXOTICS4",		// Organic Piping
				"TRA_MINERALS2",	// Unrefined Pyrite Grease
				"TRA_MINERALS3",	// Bromide Salt
				"TRA_MINERALS4",	// Polychromatic Zirconium
				"TRA_TECH2",				// Welding Soap
				"TRA_TECH3",			// Ion Capacitor
				"TRA_TECH4"				// Autonomous Positioning Unit
			};
		// rich:
		public static List<NoMansSky.NMSString0x10> RichProducts =
			new List<NoMansSky.NMSString0x10> {
				"TRA_ALLOY4",			// Five Dimensional Torus
				"TRA_ALLOY5",			// Superconducting Fibre
				"TRA_COMMODITY4",	// Ion Sphere
				"TRA_COMMODITY5",	// Teleport Coordinators
				"TRA_COMPONENT4",	// Holographic Crankshaft
				"TRA_COMPONENT5",	// High Capacity Vector Compressor
				"TRA_ENERGY4",		// Experimental Power Fluid
				"TRA_ENERGY5",		// Fusion Core
				"TRA_EXOTICS4",		// Organic Piping
				"TRA_EXOTICS5",		// Neural Duct
				"TRA_MINERALS4",	// Polychromatic Zirconium
				"TRA_MINERALS5",	// Re-latticed Arc Crystal
				"TRA_TECH4",				// Autonomous Positioning Unit
				"TRA_TECH5"				// Quantum Accelerator
			};

		// ------------------------------------------------------------------------
		
		protected override void Execute()
		{
			// execute the main function
			Main();
		}

		// ------------------------------------------------------------------------

		protected void Main ()
		{
			// GcSettlementGlobals contains various entries but our interest is in
			// SettlementBuildingCosts, which contains entries of type
			// GcSettlementBuildingCostData.
			var settleGlobals = Mbin<Globals.GcSettlementGlobals>(
				"GCSETTLEMENTGLOBALS.MBIN"
			);
			
			// sets maximum wait time between judgements to MaxJudgementWait
			settleGlobals.JudgementWaitTimeMax = MaxJudgementWait;
			
			// sets maximum roaming character population in a village to MaxNPCs
			settleGlobals.MaxNPCPopulation = MaxNPCs;
			
			// sets up maximum stats
			settleGlobals.StatsMaxValues = StatsMaxValues;
				
			// execute the multipliers
			MultiplyBuildCosts( settleGlobals, costMult );
			MultiplyBuildTimes( settleGlobals, timeMult );
			
			// new production sets
			// for the vy'keen:
			settleGlobals.VykeenProductionElements = WarProducts;
			// for the gek:
			settleGlobals.GekProductionElements = TraProducts;
			// for the korvax:
			settleGlobals.KorvaxProductionElements = ExpProducts;
			
			// for poor economy:
			settleGlobals.PoorProductionElements = PoorProducts;
			//for average economy:
			settleGlobals.NeutralProductionElements = NeutProducts;
			// for rich economy:
			settleGlobals.RichProductionElements = RichProducts;
			
			// ----------------------------------------------------------------------
			
			// GcBuildingDefinitionTable contains placement data for all buildings
			// including a little bit we need to override settlement placement
			var BuildingTable = Mbin<GameComps.GcBuildingDefinitionTable>(
				"METADATA/SIMULATION/ENVIRONMENT/PLANETBUILDINGTABLE.MBIN"
			);
			
			// overrides for the building table:
			// force one settlement to generate on any applicable planet
			BuildingTable.BuildingPlacement[36].NumOverridesToGenerate = 1;
			// which worlds should settlements spawn on?
			for ( int i = 0; i > SettleSpawnPlanetType.Count; i++ ) {
				// if SettleSpawnPlanetType[i] is True:
				if ( SettleSpawnPlanetType[i] ) {
					// set the given planet type to the frequency value
					BuildingTable.BuildingPlacement[36].Density[i] = SettleSpawnFreq;
				}
			}
			
			// ----------------------------------------------------------------------
			
			
		}
		
		//-------------------------------------------------------------------------
		
		protected void MultiplyBuildCosts(
			GcSettlementGlobals settleGlobals,
			int multiplier
		) {
			// our goal here is to load these entries of GcSettlementBuildingCostData,
			// verify that Substances' length is greater than 0,
			// then extract AmountMin and AmountMax and multiply them by multiplier
			var settleCosts = settleGlobals.SettlementBuildingCosts;
			
			// for each set of stages in settleCosts:
			foreach ( var cost in settleCosts ) {
				
				// get stage set from settleCosts
				var settleCostStages = cost.StageCosts;
				
				// for each stage in settleCostStages:
				foreach ( var stage in settleCostStages ) {
					
					// if the Products or Substances list is not 0:
					if ( (stage.Products.Count != 0) | (stage.Substances.Count != 0) ) {
						// multiply each stage's AmountMin by multiplier
						stage.AmountMin *= multiplier;
						stage.AmountMax *= multiplier;
					}
				}
			}
		}
		
		//-------------------------------------------------------------------------
		
		protected void MultiplyBuildTimes(
			GcSettlementGlobals settleGlobals,
			ulong multiplier
		)	{			
			// get settlement build times
			var settleBuildTimes = settleGlobals.SettlementBuildingTimes;
			
			// for as many entries in settleBuildTimes:
			for (int i = 0; i < settleBuildTimes.Length; i++) {
				//multiply settleBuildTimes by the multiplier
				settleBuildTimes[i] *= multiplier;				
			}
		}
	}
}

//=============================================================================
