/**
 * @file sigscan.c
 * @author Grant Kim (you@domain.com)
 * @brief Simple signature scanner with nibble level wildcard support
 * @version 0.1
 * @date 2023-05-02
 *
 * @copyright Copyright (c) 2023
 *
 */

#include "sigscan.h"

#include "pattern.h"

#include <ctype.h> // toupper()
#include <stdlib.h>
#include <string.h>

#ifndef _NDEBUG
#  include <stdio.h>
#endif // _NDEBUG
/**
 * @brief hasWildcard?
 *
 * @param input
 * @return int 0: no, 1:yes (0x??), 2: HNibble(0x?0), 3: LNibble(0x0?),
 */
inline int hasWildcard(const char input[2])
{
  if ( !input )
  {
    return 0;
  }

  if ( input[0] == '?' || input[0] == '*' )
  {
    if ( input[1] == '?' || input[1] == '*' )
    {
      return 1;
    }
    return 2;
  }

  if ( input[1] == '?' || input[1] == '*' )
  {
    return 3;
  }

  return 0;
}

static inline char convertToHex(char in)
{
  if ( in >= '0' && in <= '9' )
  {
    return in - '0';
  }

  if ( in >= 'A' && in <= 'F' )
  {
    return in - 'A' + 10;
  }

  if ( in >= 'a' && in <= 'f' )
  {
    return in - 'a' + 10;
  }

  // assert();
  return 0;
}

static int compareRange(const void *start, const char *pattern)
{
  const char *patptr       = pattern;
  const char *srcptr       = start;
  char        patbuffer[2] = {0, 0};
  char        bytefused    = 0;
  int         pos          = 1; // %3

  while ( *patptr )
  {
    pos = 1;
    do
    {
      // Skip all whitespaces.
      if ( *patptr == ' ' || *patptr == '\t' || *patptr == '\r' ||
           *patptr == '\n' )
      {
        patptr++;

        // Safeguard
        if ( !*patptr )
        {
          return 0;
        }
        continue;
      }
      else
      {
        patbuffer[pos - 1] = *(patptr++);
        pos++;
        pos %= 3;
      }
    } while ( pos );

    switch ( hasWildcard(patbuffer) )
    {
      // No wildcard
    case 0:
      patbuffer[0] = convertToHex(patbuffer[0]);
      patbuffer[1] = convertToHex(patbuffer[1]);

      bytefused = patbuffer[0] << 0x4 | patbuffer[1];
      if ( bytefused != srcptr[0] )
      {
        return 0;
      }
      break;

      // Wildcard (??, **, ?*, *?)
    case 1:
      // No need to check
      break;

      // upper wildcard // ?0
    case 2:
      bytefused = convertToHex(patbuffer[1]) << 0x0;
      if ( bytefused != (srcptr[0] & 0x0F) )
      {
        return 0;
      }
      break;

      // lower wildcard // 0?
    case 3:
      bytefused = convertToHex(patbuffer[0]) << 0x4;
      if ( bytefused != (srcptr[0] & 0xF0) )
      {
        return 0;
      }
      break;
    }

    srcptr++;
  }

  return *patptr == '\0';
}

const void *
sonar_RangedScan(const void *start, const void *end, const char *pattern)
{
  size_t patternlength = sonar_isValidPattern(pattern);

  if ( patternlength == 0 )
  {
    // Invalid pattern input
    return 0;
  }

  // Searching backward is not implemented
  if ( start > end )
  {
    return 0;
  }

  // Where sliding window can traverse
  size_t validrange = (const char *)end - (const char *)start;

  // The pattern is larger than range
  if ( validrange < patternlength / 2 )
  {
    return 0;
  }

  validrange -= patternlength / 2;

  const char *startptr = (const char *)start;

  int found = 0;

  for ( int i = 0; i < validrange; ++i )
  {
    if ( compareRange(startptr, pattern) )
    {
      found = 1;
      break;
    }
    startptr++;
  }

  return found ? startptr : 0;
}
