;;-----------------------------------------------------------------------------------
ScriptName LSSL:LootFunctions Const hidden
{Looter Shooter, Shooter Looter
by 1000101
October 2023

Looting related static functions.}




;;-----------------------------------------------------------------------------------
;; Imports

Import LSSL:Utils





;;-----------------------------------------------------------------------------------
;; Structs

Struct LootTable
    String          sTag
    { Required: Must be unique - used to find Loot Tables throught the Public API }
    
    String          sAction = "Looting"
    { What action is being performed with this filter, used for human feedback notifications - sAction + sName + sState }
    
    String          sName
    { What the pretty name of this filter is, used for human feedback notifications - sAction + sName + sState }
    
    GlobalVariable  kControl
    { Optional: No GLOB to control the filter means it cannot be shut off }
    
    FormList        kFilter
    { Reguired: What to look for - Must be Keyword or ObjectReference.GetBaseObject()
NOTE:  Embedded Formlists are not recursed, only kFilter is examined entry-by-entry; child Formlists will be passed to FindAllReferencesOfType() }
    
    Bool            bSkipInInventory = False
    {True = Ignore this LootTable when looking in container and actor inventories }
    
    FormList        kExclude = None
    { Optional:  Exclude loose objects and containers with any of the Keywords, LocationRefTypes or, ObjectReference.GetBaseObject() in this Formlist
NOTE:  This is currently NOT checked on items in inventories due to limitations of the script engine, this is mainly to filter containers }
    
EndStruct




;;-----------------------------------------------------------------------------------
;; Loot Table Functions

LootTable Function FindLootTable( LootTable[] akTable, String asTag ) Global
    If( akTable == None )
        Return None
    EndIf
    
    Int liIndex = akTable.FindStruct( "sTag", asTag )
    If( liIndex < 0 )
        Return None
    EndIf
    
    Return akTable[ liIndex ]
EndFunction


Bool Function IsObjectExcluded( ObjectReference akObject, Actor akLooter = None, LootTable akLoot = None, Float afTimeout = 0.0, Bool abAllowStealing = False ) Global
    If( akObject == None )
        Return True     ;; No object, exclude nothing
    EndIf
    
    If( akObject.IsDeleted() )\
    ||( akObject.IsDisabled() )
        ;;Debug.Trace( "LSSL:LootFunctions :: IsObjectExcluded() :: Object state :: " + akObject, 0 )
        Return True
    EndIf
    
    ;; Is it even something we can physically work with?
    Form lkBaseObject = akObject.GetBaseObject()
    If( ! ( ObjectHasInventory   ( akObject, lkBaseObject ) \
    ||      IsObjectLockable     ( akObject, lkBaseObject ) \
    ||      IsObjectInventoryable( akObject, lkBaseObject ) ) )
        ;;Debug.Trace( "LSSL:LootFunctions :: IsObjectExcluded() :: Base rejection :: " + akObject + " :: " + lkBaseObject, 0 )
        Return True     ;; Nope, reject it!
    EndIf
    
    If(   akLooter != None )\
    &&(   akLooter.WouldBeStealing( akObject ) )\
    &&( ! abAllowStealing )
        ;;Debug.Trace( "LSSL:LootFunctions :: IsObjectExcluded() :: Living Actor :: " + akObject + " :: " + lkBaseObject, 0 )
        Return True     ;; No theft
    EndIf
    
    ;; Is it a living Actor?
    Actor lkActor = akObject As Actor
    If(   lkActor != None )\
    &&( ! lkActor.IsDead() )
        ;;Debug.Trace( "LSSL:LootFunctions :: IsObjectExcluded() :: Living Actor :: " + akObject + " :: " + lkBaseObject, 0 )
        Return True     ;; Cannot steal from peoples pockets!  Definitely no now, probably never - Revist unconscious and stunned?
    EndIf
    
    If( akLoot == None )
        ;;Debug.Trace( "LSSL:LootFunctions :: IsObjectExcluded() :: No Loot :: " + akObject + " :: " + lkBaseObject, 0 )
        Return False    ;; No loot entry, no exclusion
    EndIf
    
    FormList lkExclude = akLoot.kExclude
    If( lkExclude == None )
        ;;Debug.Trace( "LSSL:LootFunctions :: IsObjectExcluded() :: No Exclude :: " + akObject + " :: " + lkBaseObject, 0 )
        Return False    ;; No explicit exclusions
    EndIf
    
    Int liIndex = lkExclude.GetSize()
    While( liIndex > 0 );;&&( Utility.GetCurrentRealTime() <= afTimeout )
        liIndex -= 1
        
        Form lkEx = lkExclude.GetAt( liIndex )
        
        If  ( lkEx == akObject )\
        ||  ( lkEx == lkBaseObject )\
        ||( ( lkEx Is Keyword           )&&( akObject.HasKeyword   ( lkEx As Keyword         ) ) )\
        ||( ( lkEx Is LocationRefType   )&&( akObject.HasLocRefType( lkEx As LocationRefType ) ) )
            ;;Debug.Trace( "LSSL:LootFunctions :: IsObjectExcluded() :: Excluded by class :: " + akObject + " :: " + lkBaseObject, 0 )
            Return True
        EndIf
        
    EndWhile
    
    ;;Debug.Trace( "LSSL:LootFunctions :: IsObjectExcluded() :: Not excluded :: " + akObject + " :: " + lkBaseObject, 0 )
    Return False
EndFunction




;;-----------------------------------------------------------------------------------
;; Loot Functions

ObjectReference[] Function GetLootAroundActorByFilter( Actor akLooter, Form akFilter, Float afRadius, Bool abForceByKeyword = False, Bool abAllowStealing = False ) Global
    If( akLooter == None )\
    ||( akFilter == None )\
    ||( afRadius <= 0.0 )
        Return None
    EndIf
    
    ObjectReference[] lkObjects
    ;;String lsLines = "LSSL:LootFunctions :: GetLootAroundActorByFilter()"
    ;;lsLines = lsLines + "\n\takFilter = " + akFilter
    ;;lsLines = lsLines + "\n\tafRadius = " + afRadius
    
    If( akFilter Is Keyword )||( abForceByKeyword )
        ;;Debug.Trace( "LSSL:LootFunctions :: GetLootAroundActorByFilter() :: With Keyword", 0 )
        ;;lsLines = lsLines + "\n\tFindAllReferencesWithKeyword()"
        lkObjects = akLooter.FindAllReferencesWithKeyword( akFilter, afRadius )
    Else
        ;;Debug.Trace( "LSSL:LootFunctions :: GetLootAroundActorByFilter() :: By Type", 0 )
        ;;lsLines = lsLines + "\n\tFindAllReferencesOfType()"
        lkObjects = akLooter.FindAllReferencesOfType( akFilter, afRadius )
    EndIf
    
    ;; Remove anything we absolutely can't use
    Int liIndex = lkObjects.Length
    
    ;;lsLines = lsLines + "\n\t# Prefiltered lkObjects = " + liIndex
    
    While( liIndex > 0 )
        liIndex -= 1
        ObjectReference lkObject = lkObjects[ liIndex ]
        If( IsObjectExcluded( lkObject, akLooter, abAllowStealing = abAllowStealing ) )
            ;;lsLines = lsLines + "\n\t\tIsObjectExcluded() :: " + lkObject + " :: " + lkObject.GetBaseObject()
            lkObjects.Remove( liIndex, 1 )
        EndIf
    EndWhile
    
    ;;lsLines = lsLines + "\n\t# Filtered lkObjects = " + lkObjects.Length
    ;;Debug.Trace( lsLines, 0 )
    
    Return lkObjects
EndFunction


Function LootAroundActor( Actor akLooter, LootTable akLoot, Float afRadius, Float afTimeout, Bool abAllowStealing, LootTable[] akTable ) Global
    ;;Debug.Trace( "LSSL:LootFunctions :: LootAroundActor()", 0 )
    
    If( ! GetGLOBAsBool( akLoot.kControl, True ) )
        ;;Debug.Trace( "LSSL:LootFunctions :: LootAroundActor() :: Disabled by Control", 0 )
        Return
    EndIf
    
    FormList lkFilter = akLoot.kFilter
    ObjectReference[] lkObjects
    Int liCount = lkFilter.GetSize()
    If( liCount == 0 )
        Return
    EndIf
    
    ;;lkObjects = GetLootAroundActorByFilter( akLooter, lkFilter, afRadius, ( lkFilter.GetAt( 0 ) Is Keyword ), abAllowStealing = abAllowStealing )
    ;;If( lkObjects != None )
    ;;    LootObjects( akLooter, lkObjects, akLoot, afTimeout, abAllowStealing, akTable )
    ;;EndIf
    
    Int liIndex = lkFilter.GetSize()
    While( liIndex > 0 );;&&( Utility.GetCurrentRealTime() <= afTimeout )
        liIndex -= 1
        
        Form lkPhilTear = lkFilter.GetAt( liIndex )
        
        ;;Debug.Trace( "vvvvvvvv--------------------------------> " + lkPhilTear + " <--------------------------------vvvvvvvv", 0 )
        
        lkObjects = GetLootAroundActorByFilter( akLooter, lkPhilTear, afRadius, abAllowStealing = abAllowStealing )
        If( lkObjects != None )
            LootObjects( akLooter, lkObjects, lkPhilTear, akLoot, afTimeout, abAllowStealing, akTable )
        EndIf
        
        ;;Debug.Trace( "^^^^^^^^--------------------------------> " + lkPhilTear + " <--------------------------------^^^^^^^^", 0 )
        
    EndWhile
    
EndFunction


Function LootInventory( Actor akLooter, ObjectReference akTarget, LootTable akLoot, Float afTimeout, Bool abAllowStealing, LootTable[] akTable ) Global
    If( akTarget.GetItemCount( None ) == 0 )
        Return ;; It's empty, skip it
    EndIf
    
    ;;String lsLines = "LSSL:LootFunctions :: LootInventory() :: Target :: " + akTarget
    
    ObjectReference[] lkObjects
    Int liLoot = akTable.Length
    
    If( ! TryUnlock( akLooter, akTarget ) )
        ;;lsLines = lsLines + "Cannot unlock"
        ;;Debug.Trace( lsLines, 0 )
        Return  ;; Couldn't unlock it
    EndIf
    
    ;; Now loot the bastard!
    While( liLoot > 0 );;&&( Utility.GetCurrentRealTime() <= afTimeout )
        liLoot -= 1
        LootTable lkLoot = akTable[ liLoot ]
        
        ;;lsLines = lsLines +  "\n\tTable :: '" + lkLoot.sTag + "' :: "
        
        If(   lkLoot.bSkipInInventory )\
        ||( ! GetGLOBAsBool( lkLoot.kControl, True ) )
            ;;lsLines = lsLines + "Skipping"
        Else
            
            ;;lsLines = lsLines + "Looting"
            
            FormList lkFilter = lkLoot.kFilter
            ;;akTarget.RemoveItem( lkFilter, -1, True, akLooter )
            
            Int liIndex = lkFilter.GetSize()
            ;; Looter goes BRRRR!
            While( liIndex > 0 );;&&( Utility.GetCurrentRealTime() <= afTimeout )
                liIndex -= 1
                Form lkForm = lkFilter.GetAt( liIndex )
                Int liCount = akTarget.GetItemCount( lkForm )
                While( liCount > 0 );;&&( Utility.GetCurrentRealTime() <= afTimeout )
                    ;;lsLines = lsLines + "\n\t\tLooting :: " + liCount + " :: " + lkForm
                    akTarget.RemoveItem( lkForm, liCount, True, akLooter )
                    liCount = akTarget.GetItemCount( lkForm )
                EndWhile
            EndWhile
            
        EndIf
        
    EndWhile
    
    ;;Debug.Trace( lsLines, 0 )
EndFunction


Function LootObjects( Actor akLooter, ObjectReference[] akObjects, Form akFilter, LootTable akLoot, Float afTimeout, Bool abAllowStealing, LootTable[] akTable ) Global
    If( akLooter == None )
        Return
    EndIf
    
    Int liItems = akObjects.Length
    
    ;;String lsLines
    ;;lsLines = lsLines + "\n\tsTag = " + akLoot.sTag
    ;;lsLines = lsLines + "\n\takFilter = " + akFilter
    ;;lsLines = lsLines + "\n\t# Objects = " + liItems
    ;;Debug.Trace( "LSSL:LootFunctions :: LootObjects()" + lsLines, 0 )
    
    While( liItems > 0 );;&&( Utility.GetCurrentRealTime() <= afTimeout )
        liItems -= 1
        
        ObjectReference lkItem = akObjects[ liItems ]
        Form lkBase = lkItem.GetBaseObject()
        
        ;;Bool lbDeleted = lkItem.IsDeleted()
        ;;Bool lbEnabled = lkItem.IsEnabled()
        ;;Bool lbStealing = akLooter.WouldBeStealing( lkItem )
        ;;Bool lbExcluded = IsObjectExcluded( lkItem, akLoot, afTimeout )
        
        ;;/
        ;;String lsLines = ""
        ;;lsLines = lsLines + "\n\tlkItem = " + lkItem
        ;;lsLines = lsLines + "\n\t\tlkBase = " + lkBase
        ;;lsLines = lsLines + "\n\t\tlbDeleted = " + lbDeleted
        ;;lsLines = lsLines + "\n\t\tlbEnabled = " + lbEnabled
        ;;lsLines = lsLines + "\n\t\tlbStealing = " + lbStealing
        ;;lsLines = lsLines + "\n\t\tlbExcluded = " + lbExcluded
        ;;/;
        
        ;;If  ( ! lbDeleted )\
        ;;&&  (   lbEnabled )\
        ;;&&( ( ! lbStealing )\
        ;;||  (   abAllowStealing ) );;\
        ;;&&  ( ! lbExcluded )
            
            ;;Debug.Trace( "LSSL:LootFunctions :: LootObjects() :: " + lkItem + " :: " + lkBase, 0 )
            
            If    ( ObjectHasInventory( lkItem, lkBase ) )
                ;; Item is a container of some kind, parse it's inventory
                ;;lsLines = lsLines + "\n\t\tLootInventory()"
                LootInventory( akLooter, lkItem, akLoot, afTimeout, abAllowStealing, akTable )
                
            ElseIf( IsObjectLockable( lkItem, lkBase ) )
                ;; Item is a possibly locked object we want to QoL our way past
                ;;lsLines = lsLines + "\n\t\tTryUnlock()"
                TryUnlock( akLooter, lkItem )
                
            Else;;If( IsObjectInventoryable( lkItem, lkBase ) )
                ;; Lootable object sitting on the floor
                ;;lsLines = lsLines + "\n\t\tActivate()"
                lkItem.Activate( akLooter, False )
            EndIf
            
        ;;EndIf
        
        ;;Debug.Trace( "LSSL:LootFunctions :: LootObjects()" + lsLines, 0 )
        
    EndWhile
    
    ;;Debug.Trace( "LSSL:LootFunctions :: LootObjects()" + lsLines, 0 )
EndFunction



