;;-----------------------------------------------------------------------------------
ScriptName LSSL:LootQuest Extends Quest
{Looter Shooter, Shooter Looter
by 1000101
October 2023

Loot scanner quest.}




;;-----------------------------------------------------------------------------------
;; Imports

Import LSSL:Utils
Import LSSL:LootFunctions




;;-----------------------------------------------------------------------------------
;; Script Constants

Int iTID_Scan           = 0x0001000 Const
Float fScanTimeLimit    = 0.75      Const
Float fMinRadius        = 1.0       Const ;; Minimum 1.0m radius
Float fMinInterval      = 1.0       Const ;; Minimum 1.0s between scans




;;-----------------------------------------------------------------------------------
;; Loot Tables

LootTable[]     Property    kLootTable          Auto Const Mandatory




;;-----------------------------------------------------------------------------------
;; Script Properties

Keyword kLocTypeOutpost

Actor kLooter

Bool bUsingHandscanner  = False
Int iScanThreads        = -1    ;; How many threads are still running?




;;-----------------------------------------------------------------------------------
;; GlobalVariables masquerading as Properties

GlobalVariable  Property    kRadius             Auto Const Mandatory
Float Property ScanRadius Hidden
    Float Function Get()
        Return kRadius.GetValue()
    EndFunction
    Function Set( Float afValue )
        kRadius.SetValue( Math.Max( afValue, fMinRadius ) )
    EndFunction
EndProperty


GlobalVariable  Property    kInterval           Auto Const Mandatory
Float Property ScanInterval Hidden
    Float Function Get()
        Return kInterval.GetValue()
    EndFunction
    Function Set( Float afValue )
        kInterval.SetValue( Math.Max( afValue, fMinInterval ) )
    EndFunction
EndProperty

Float Property ScanTimeLimit Hidden
    Float Function Get()
        Return ScanInterval * fScanTimeLimit
    EndFunction
EndProperty


GlobalVariable  Property    kOnlyWhileScanning  Auto Const Mandatory
Bool Property HandScannerOnly Hidden
    Bool Function Get()
        Return GetGLOBAsBool( kOnlyWhileScanning )
    EndFunction
    Function Set( Bool abValue )
        SetGLOBAsBool( kOnlyWhileScanning, abValue )
        If    (  abValue &&  bUsingHandscanner )\
        ||    ( !abValue )
            ;; Only use hand scanner and currently using hand scanner
            ;; or loot regardless of scanning
            StartTimer( ScanInterval, iTID_Scan )
        ElseIf(  abValue && !bUsingHandscanner )
            ;; Only use hand scanner and not currently using hand scanner
            CancelTimer( iTID_Scan )
        EndIf
    EndFunction
EndProperty


GlobalVariable  Property    kPaused             Auto Const Mandatory
Bool Property PauseScan Hidden
    Bool Function Get()
        Return GetGLOBAsBool( kPaused )
    EndFunction
    Function Set( Bool abValue )
        SetGLOBAsBool( kPaused, abValue )
    EndFunction
EndProperty


GlobalVariable  Property    kAllowStealing      Auto Const Mandatory
Bool Property AllowStealing Hidden
    Bool Function Get()
        Return GetGLOBAsBool( kAllowStealing )
    EndFunction
    Function Set( Bool abValue )
        SetGLOBAsBool( kAllowStealing, abValue )
    EndFunction
EndProperty




;;-----------------------------------------------------------------------------------
;; Public API

;; cqf LSSL_Scanner SetFloatControl "asControl" afValue
Function SetFloatControl( String asControl, Float afValue )
    String asMSG = "Setting " + asControl + " to " + afValue
    If    ( asControl == "ScanRadius" )
        ScanRadius      = afValue
    ElseIf( asControl == "ScanInterval" )
        ScanInterval    = afValue
    Else
        asMSG = "Invalid setting '" + asControl + "'"
    EndIf
    Debug.Notification( asMSG )
EndFunction


;; cqf LSSL_Scanner ModFloatControl "asControl" afValue
Function ModFloatControl( String asControl, Float afValue )
    String asMSG = "Setting " + asControl + " to "
    If    ( asControl == "ScanRadius" )
        ScanRadius      += afValue
        asMSG = asMSG + ScanRadius
    ElseIf( asControl == "ScanInterval" )
        ScanInterval    += afValue
        asMSG = asMSG + ScanInterval
    Else
        asMSG = "Invalid setting '" + asControl + "'"
    EndIf
    Debug.Notification( asMSG )
EndFunction


;; cqf LSSL_Scanner SetBoolControl "asControl" abValue
Function SetBoolControl( String asControl, Bool abValue )
    String asMSG = "Setting " + asControl + " to " + abValue
    If    ( asControl == "HandScannerOnly" )
        HandScannerOnly = abValue
    ElseIf( asControl == "PauseScan" )
        PauseScan       = abValue
    ElseIf( asControl == "AllowStealing" )
        AllowStealing   = abValue
    Else
        asMSG = "Invalid setting '" + asControl + "'"
    EndIf
    Debug.Notification( asMSG )
EndFunction


;; cqf LSSL_Scanner ToggleBoolControl "asControl"
Function ToggleBoolControl( String asControl )
    String asMSG = "Setting " + asControl + " to "
    If    ( asControl == "HandScannerOnly" )
        HandScannerOnly = ! HandScannerOnly
        asMSG = asMSG + HandScannerOnly
    ElseIf( asControl == "PauseScan" )
        PauseScan       = ! PauseScan
        asMSG = asMSG + PauseScan
    ElseIf( asControl == "AllowStealing" )
        AllowStealing   = ! AllowStealing
        asMSG = asMSG + AllowStealing
    Else
        asMSG = "Invalid setting '" + asControl + "'"
    EndIf
    Debug.Notification( asMSG )
EndFunction


;; cqf LSSL_Scanner ToggleFilter "asFilterTag"
Function ToggleFilter( String asFilterTag )
    LootTable lkLoot = FindLootTable( kLootTable, asFilterTag )
    If( lkLoot == None )
        Debug.Notification( "Invalid filter tag '" + asFilterTag + "'" )
        Return
    EndIf
    
    String lsMSG = lkLoot.sAction +  " " + lkLoot.sName + " is now "
    
    GlobalVariable lkControl = lkLoot.kControl
    Bool lbNewState
    If( lkControl == None )
        lbNewState = True
    Else
        lbNewState = ! GetGLOBAsBool( lkControl )
        SetGLOBAsBool( lkControl, lbNewState )
    EndIf
    
    If( lbNewState )
        lsMSG = lsMSG + "enabled"
    Else
        lsMSG = lsMSG + "disabled"
    EndIf
    
    Debug.Notification( lsMSG )
    
EndFunction




;;-----------------------------------------------------------------------------------
;; Event Handlers


Event OnQuestInit()
    Debug.Trace( Self + " :: Initialized", 0 )   ;; Tag the log so we know we've started
    kLooter = Game.GetPlayer()
    kLocTypeOutpost = GetLocTypeOutpost()
    iScanThreads = 0
    Self.RegisterForMenuOpenCloseEvent( "MonocleMenu" )
    If( !HandScannerOnly )
        TryScanNow()
    EndIf
EndEvent


Event OnMenuOpenCloseEvent( String asMenuName, Bool abOpening )
    ;;Debug.Trace( Self + " :: OnMenuOpenCloseEvent()", 0 )
    If( asMenuName == "MonocleMenu" )
        bUsingHandscanner = abOpening
        If( abOpening )
            TryScanNow()
        ElseIf( HandScannerOnly )
            CancelTimer( iTID_Scan )
        EndIf
    EndIf
EndEvent


Event OnTimer( Int aiTimerID )
    ;;Debug.Trace( Self + " :: OnTimer()", 0 )
    If( aiTimerID == iTID_Scan )
        TryScanNow()
    EndIf
EndEvent




;;-----------------------------------------------------------------------------------
;; Protected API


Bool Function CanScanNow()
    Location lkLocation = kLooter.GetCurrentLocation()
    If( lkLocation != None )\
    &&( lkLocation.HasKeyword( kLocTypeOutpost ) )
        Return False     ;; Don't loot our outposts
    EndIf
    
    If( kLooter.GetCurrentShipRef() == Game.GetPlayerHomeSpaceShip() )
        Return False     ;; Don't loot the player home Ship
    EndIf
    
    Return \
      ( ( ! HandScannerOnly )\
    ||  (   bUsingHandscanner ) )\
    &&  (   iScanThreads == 0 )\
    &&  ( ! PauseScan )
EndFunction


Function TryScanNow()
    ;;Debug.Trace( Self + " :: TryScanNow()", 0 )
    If( CanScanNow() )
        ScanForLoot()
    EndIf
    If( !HandScannerOnly )||( bUsingHandscanner )
        StartTimer( ScanInterval, iTID_Scan )
    EndIf
EndFunction


Float fStartTime
Function ScanForLoot()
    
    ;;String lsLines = ""
    
    iScanThreads = 1        ;; Block for this thread
    fStartTime   = Utility.GetCurrentRealTime()
    
    Float lfTimeLimit = ScanTimeLimit ;; Allow most of the interval time to be used per scan loop
    Float lfRadius = kRadius.GetValue()
    Bool lbAllowStealing = AllowStealing
    
    Int liTableSize = kLootTable.Length
    
    ;;lsLines = lsLines + "\n\tlfTimeLimit = " + lfTimeLimit
    ;;lsLines = lsLines + "\n\tlfRadius = " + lfRadius
    ;;lsLines = lsLines + "\n\tlbAllowStealing = " + lbAllowStealing
    ;;lsLines = lsLines + "\n\tliTableSize = " + liTableSize
    
    ;; Calculate the timeout last, right before starting the threads so they get as much time as possible
    Float lfTimeout = Utility.GetCurrentRealTime() + lfTimeLimit
    
    ;;lsLines = lsLines + "\n\tlfTimeout = " + lfTimeout
    ;;Debug.Trace( Self + " :: ScanForLoot()" + lsLines, 0 )
    
    ;; Start a thread for each loot table
    ;;Int liThreadsSpawned = 0
    Int liIndex = 0
    While( liIndex < liTableSize )
        
        LSSL:LootFunctions:LootTable lkLoot = kLootTable[ liIndex ]
        If( GetGLOBAsBool( lkLoot.kControl, True ) )
            ;;Debug.Trace( Self + " :: ScanForLoot() :: " + lkLoot.sTag + " :: Threaded", 0 )
            
            iScanThreads += 1                               ;; Increment thread count
            ;;liThreadsSpawned += 1
            
            Var[] lkParams = New Var[ 4 ]
            lkParams[ 1 ] = lfRadius
            lkParams[ 2 ] = lfTimeout
            lkParams[ 3 ] = lbAllowStealing
            lkParams[ 0 ] = lkLoot                          ;; Set the table entry for the thread
            CallFunctionNoWait( "__LootThread", lkParams )  ;; Start the paused thread
            
        ;;Else
        ;;    Debug.Trace( Self + " :: ScanForLoot() :: " + lkLoot.sTag + " :: disabled", 0 )
        EndIf
        
        liIndex += 1
    EndWhile
    
    iScanThreads -= 1       ;; Unblock this thread
    ;;Debug.Trace( Self + " :: ScanForLoot() :: Threads Spawned :: " + liThreadsSpawned , 0 )
EndFunction




;;-----------------------------------------------------------------------------------
;; Internal API

;;The loot thread which will auto-decrement the thread counter
Function __LootThread( LSSL:LootFunctions:LootTable akLoot, Float afRadius, Float afTimeout, Bool abAllowStealing )
    LSSL:LootFunctions.LootAroundActor( kLooter, akLoot, afRadius, afTimeout, abAllowStealing, kLootTable )
    iScanThreads -= 1   ;; Thread complete, decrement
    If( iScanThreads == 0 )
        Float lfFinishTime = Utility.GetCurrentRealTime()
        Float lfDelta = lfFinishTime - fStartTime
        Debug.Trace( Self + " :: Total loot cycle time :: " + lfDelta + " seconds", 0 )
    EndIf
EndFunction



