;;-----------------------------------------------------------------------------------
ScriptName LSSL:Utils Const hidden
{Looter Shooter, Shooter Looter
by 1000101
October 2023

Utility functions.}




;;-----------------------------------------------------------------------------------
;; Get some core Forms

MiscObject Function GetDigipick() Global
    Return Game.GetFormFromFile( 0x0000000A, "Starfield.esm" ) As MiscObject
EndFunction

MiscObject Function GetCredits() Global
    Return Game.GetFormFromFile( 0x0000000F, "Starfield.esm" ) As MiscObject
EndFunction

Keyword Function GetLocTypeOutpost() Global
    Return Game.GetFormFromFile( 0x000234F1, "Starfield.esm" ) As Keyword
EndFunction




;;-----------------------------------------------------------------------------------
;; Hacking and Picking related Functions

Int Function GetLockLevelXP( Int aiLevel ) Global
    If( aiLevel <= 0 )
        Return 0
    EndIf
    
    String sReward
    If( aiLevel >= 100 )
        sReward = "fLockpickXPRewardVeryHard"
    ElseIf( aiLevel >= 75 )
        sReward = "fLockpickXPRewardHard"
    ElseIf( aiLevel >= 50 )
        sReward = "fLockpickXPRewardAverage"
    Else
        sReward = "fLockpickXPRewardEasy"
    EndIf
    
    Return Game.GetGameSettingFloat( sReward ) As Int
EndFunction


Int Function GetLooterLockLimit( Actor akLooter ) Global
    If( akLooter == None )\
    ||( akLooter != Game.GetPlayer() )
        Return -1   ;; Need more details on NPCs - For now I am using the quest stages for the player
    EndIf
    Quest lkSkills = Game.GetFormFromFile( 0x002C59E4, "Starfield.esm" ) As Quest
    If( lkSkills.IsStageDone( 204 ) )       ;; Security Rank 4
        Return 100  ;; Master
    ElseIf( lkSkills.IsStageDone( 203 ) )   ;; Security Rank 3
        Return 75   ;; Expert
    ElseIf( lkSkills.IsStageDone( 202 ) )   ;; Security Rank 2
        Return 50   ;; Advanced
    ElseIf( lkSkills.IsStageDone( 201 ) )   ;; Security Rank 1
        Return 25   ;; Novice
    EndIf
    Return 0        ;; Can only pick the unlocked
EndFunction


Int Function GetRandomPicksForLevel( Int aiLevel ) Global
    ;; Recurse through each level so extra picks become a chance per grade per grade
    ;; Keeps the game a little more interesting and accounts for the chance the player
    ;; may make a mistake in this tedius and annoying minigame
    ;; Results:
    ;;  Unlocked:   0   : 0
    ;;  Novice:     1   : 1
    ;;  Advanced:   1-2 : 1 + 1 (5%)
    ;;  Expert:     1-3 : 1 + 1 (5%) + 1 (10%)
    ;;  Master:     1-4 : 1 + 1 (5%) + 1 (10%) + 1 (15%)
    
    If( aiLevel <= 0 )
        Return 0    ;; No lock, no picks
    ElseIf( aiLevel <= 25 )
        Return 1    ;; Always 1 if it's locked at all
    EndIf
    
    ;; Random extra pick for this level
    Int liResult = 0
    Int liGrade = ( aiLevel / 25 )              ;; Int level to lock grade (1=novice...)
    If( Utility.RandomInt( 1, 20 ) < liGrade )  ;; 5% chance per grade to use an extra pick past novice
        liResult = 1
    EndIf
    
    ;; Return picks for this plus all previous levels
    Return liResult + GetRandomPicksForLevel( aiLevel - 25 )
EndFunction


Bool Function TryUnlock( Actor akLooter, ObjectReference akTarget ) Global
    Form lkBase = akTarget.GetBaseObject()
    If( ! IsObjectLockable( akTarget ) )\
    ||( ! akTarget.IsLocked() )
        Return True ;; Not an unlockable or not locked
    EndIf
    
    ;; Do they have a key?
    Key lkKey = akTarget.GetKey()
    If( lkKey != None )\
    &&( akLooter.GetItemCount( lkKey ) > 0 )
        ;; Has the key
        akTarget.Lock( False, False, False ) ;; SF - Unlock, Not the owner, Don't transverse
        ;;akTarget.Lock( False, False ) ;; FO4 - Unlock, Not the owner
        Return True
    EndIf
    
    ;; Is it unpickable?
    Int liLevel = akTarget.GetLockLevel()
    If( liLevel > 200 )
        ;; Levels above 200 mean requires key, only terminal, etc, etc
        Return False
    EndIf
    
    ;; Try and pick it then!
    
    MiscObject lkDigipick = GetDigipick()
    Int liPicks = akLooter.GetItemCount( lkDigipick )
    If( liPicks < 1 )
        Return False ;; No picks - DOH!
    EndIf
    
    Int liLimit = GetLooterLockLimit( akLooter )
    If( liLevel > liLimit )
        Return False ;; Not skilled enough - Scheiße!
    EndIf
    
    ;; How many times are they going to make a mistake?
    Int liRequired = GetRandomPicksForLevel( liLevel )
    
    ;; Remove all the required picks, even if they don't have enough and it wastes them all (they won't know it's a waste until after trying)
    akLooter.RemoveItem( lkDigipick, liRequired, True, None )
    
    ;; Did they succeed or waste them?
    Bool lbUnlocked = ( liPicks >= liRequired )
    
    ;; Unlock it?
    If( lbUnlocked )
        akTarget.Lock( False, False, False ) ;; SF - Unlock, Not the owner, Don't transverse
        ;;akTarget.Lock( False, False ) ;; FO4 - Unlock, Not the owner
        
        ;; Award XP if it's the player
        If( akLooter == Game.GetPlayer() )
            Game.RewardPlayerXP( GetLockLevelXP( liLevel ), False )
        EndIf
    EndIf
    
    ;; Let the caller know
    Return lbUnlocked
EndFunction




;;-----------------------------------------------------------------------------------
;; Functions to probe the nature of ObjectReferences


Bool Function ObjectHasInventory( ObjectReference akObject, Form akBase = None ) Global
    ;; Not all objects have an inventory
    If( akObject == None )&&( akBase == None )
        Return False
    EndIf
    If( akBase == None )
        akBase = akObject.GetBaseObject()
    EndIf
    Return \
        ( akBase Is ActorBase )\
    ||  ( akBase Is Container )\
    ||  ( akBase Is SpaceshipBase )
EndFunction


Bool Function IsObjectInventoryable( ObjectReference akObject, Form akBase = None ) Global
    ;; Not all objects can be put into an inventory
    If( akObject == None )&&( akBase == None )
        Return False
    EndIf
    If( akBase == None )
        akBase = akObject.GetBaseObject()
    EndIf
    Return \
        ( akBase Is Ammo )\
    ||  ( akBase Is Armor )\
    ||  ( akBase Is Book )\
    ||  ( akBase Is Flora )\
    ||  ( akBase Is Ingredient )\
    ||  ( akBase Is Key )\
    ||  ( akBase Is MiscObject )\
    ||  ( akBase Is Weapon )
EndFunction


Bool Function IsObjectLockable( ObjectReference akObject, Form akBase = None ) Global
    ;; Not all objects can have a lock
    If( akObject == None )&&( akBase == None )
        Return False
    EndIf
    If( akBase == None )
        akBase = akObject.GetBaseObject()
    EndIf
    Return \
        ( akBase Is Container )\
    ||  ( akBase Is Door )\
    ||  ( akBase Is Terminal )\
    ||  ( akBase Is SpaceshipBase )
EndFunction





;;-----------------------------------------------------------------------------------
;; Treat a GLOB like a BOOL

Bool Function GetGLOBAsBool( GlobalVariable akGLOB, Bool abDefaultOnNone = False ) Global
    If( akGLOB == None )
        Return abDefaultOnNone
    EndIf
    Return ( akGLOB.GetValueInt() != 0 )
EndFunction
Function SetGLOBAsBool( GlobalVariable akGLOB, Bool abValue ) Global
    If( akGLOB == None )
        Return
    EndIf
    Float afValue = 0.0
    If( abValue )
        afValue = 1.0
    EndIf
    akGLOB.SetValue( afValue )
EndFunction




;;-----------------------------------------------------------------------------------
;; Debug functions

Function DEBUG_DumpObjectInfo( ObjectReference akObject ) Global
    String lsLines = "LSSL:Utils.DEBUG_DumpObjectInfo()\n\takObject = " + akObject
    Form lkBase = akObject.GetBaseObject()
    lsLines = lsLines + "\n\tBaseObject = " + lkBase
    Float lfX = akObject.X
    Float lfY = akObject.Y
    Float lfZ = akObject.Z
    lsLines = lsLines + "\n\tX = " + lfX
    lsLines = lsLines + "\n\tY = " + lfY
    lsLines = lsLines + "\n\tZ = " + lfZ
    Actor lkPlayer = Game.GetPlayer()
    lfX -= lkPlayer.X
    lfY -= lkPlayer.Y
    lfZ -= lkPlayer.Z
    lsLines = lsLines + "\n\tDistance to Player = " + Math.sqrt( lfX * lfX + lfY * lfY + lfZ * lfZ )
    Debug.Trace( lsLines, 0 )
EndFunction



