ScriptName VendingMachineScript Extends ObjectReference
{ Handles Vending Machine activation behavior. }

;-- Variables ---------------------------------------
Int allowedActivations
Int usedActivations

;-- Properties --------------------------------------
Group DefaultProperties
  Int Property minActivations = 4 Auto Const
  { The minimum number of times this vending machine can be activated. }
  Int Property maxActivations = 7 Auto Const
  { The maximum number of times this vending machine can be activated. }
  Float Property doubleVendChance = 1.0 Auto Const
  { The chance (0.0 - 100.0) of the machine giving you an extra item. }
  String Property nodeToSpawnItemAtName = "Node01_B" Auto Const
  { The node on the mesh where the item should spawn at. }
  Float Property itemSpawnOffsetHorizontalScale = 1.27 Auto Const
  { The number to scale the X and Y components of the vector between this machine's origin and nodeToSpawnItemAtName's position by. }
  Float Property itemSpawnOffsetVerticalScale = 1.15 Auto Const
  { The number to scale the Z component of the vector between this machine's origin and nodeToSpawnItemAtName's position by. }
EndGroup

Group RequiredProperties
  LeveledItem Property itemsToVend Auto Const mandatory
  { The list of items to vend. }
  GlobalVariable Property cost Auto Const mandatory
  { The cost to despense an item. The cost is displayed in the Activator's activation display text. }
  wwiseevent Property WwiseEvent_VendingMachineActivate Auto Const mandatory
  { The sound to player when the vending machine successfully dispenses an item. }
  wwiseevent Property WwiseEvent_VendingMachineActivateError Auto Const mandatory
  { The sound to play if the player attempts to activate it after all of its items have been dispensed. }
  Message Property notEnoughCreditsMessage Auto Const mandatory
  { The message notification to display if the player doesn't have enough credits. }
  ActorValue Property VendingMachineResetGameTimeAV Auto Const mandatory
EndGroup

Group OptionalProperties
  LeveledItem Property rareItemsToVend Auto Const
  { An optional set of very rare items to vend. }
  Float Property rareItemChance = 0.0 Auto Const
  { The chance (0.0 - 100.0) of spawning a rare item, if rareItemsToVend is set. }
  Int Property daysUntilReset = 0 Auto Const
  { Optionally specify the days until this machine can begin vending again. By default, it stops vending until the cell is reset. }
EndGroup


;-- Functions ---------------------------------------

Function ResetVendingMachine()
  usedActivations = 0 ; #DEBUG_LINE_NO:52
  allowedActivations = Utility.RandomInt(minActivations, maxActivations) ; #DEBUG_LINE_NO:53
EndFunction

Function TryToVendItem()
  Actor playerRef = Game.GetPlayer() ; #DEBUG_LINE_NO:57
  MiscObject credits = Game.GetCredits() ; #DEBUG_LINE_NO:58
  Int costValue = cost.GetValueInt() ; #DEBUG_LINE_NO:59
  If playerRef.GetItemCount(credits as Form) < costValue ; #DEBUG_LINE_NO:62
    Self.PlayErrorSound() ; #DEBUG_LINE_NO:63
    notEnoughCreditsMessage.Show(0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0) ; #DEBUG_LINE_NO:64
    Return  ; #DEBUG_LINE_NO:65
  EndIf
  playerRef.RemoveItem(credits as Form, costValue, False, None) ; #DEBUG_LINE_NO:69
  usedActivations += 1 ; #DEBUG_LINE_NO:70
  Self.HandleVendingItem() ; #DEBUG_LINE_NO:72
EndFunction

Function HandleVendingItem()
  Self.DispenseItem() ; #DEBUG_LINE_NO:76
  Bool doubleSuccess = Utility.RandomFloat(0.0, 100.0) <= doubleVendChance ; #DEBUG_LINE_NO:79
  If doubleSuccess ; #DEBUG_LINE_NO:80
    Self.DispenseItem() ; #DEBUG_LINE_NO:81
  EndIf
EndFunction

Function DispenseItem()
  ; Add wait buffers around playing the sound so that things line up better
  Utility.Wait(0.5)
  Self.PlayDispenseSound() ; #DEBUG_LINE_NO:86
  Utility.Wait(0.4)
 
  Float[] objectPos = Self.GetSpacePosition()
  Float objectAngleR = Self.GetAngleZ()

  ObjectReference item = Self.PlaceAtNode(nodeToSpawnItemAtName, Self.GetItemToVend() as Form, 1, False, True, True, False)
  item.DisableNoWait(False)

  ; Addresses issue where picking up dispensed items would be considered stealing in some locations
  item.SetFactionOwner(None, True)
  item.SetActorOwner(Game.GetPlayer().GetBaseObject() as ActorBase, False)

  Float itemWidth = item.GetWidth() ; x
  Float itemLength = item.GetLength() ; y
  Float itemHeight = item.GetHeight() ; z

  Float itemLocalAngleY = 0
  Float itemLocalAngleP = 0
  Float itemGlobalAngleR = 0 ; same as global

  If itemWidth >= itemLength && itemWidth >= itemHeight ; like something probably
    itemLocalAngleY = 180
    itemLocalAngleP = Utility.RandomFloat(-20, 20)
    itemGlobalAngleR += 0 + objectAngleR
  ElseIf itemLength >= itemWidth && itemLength >= itemHeight ; like packaged chunks
    itemLocalAngleY = 180
    itemLocalAngleP = Utility.RandomFloat(-20, 20)
    itemGlobalAngleR += 90 + objectAngleR
  Else ; height is greatest - like boomcola, cans, coffee
    itemLocalAngleY = 90
    itemLocalAngleP = 90 + objectAngleR
    itemGlobalAngleR += 0
  EndIf

  ; Convert local rotation to global. Thanks to https://forums.nexusmods.com/topic/9011983-local-to-global-rotations-that-old-chestnut/
	Float itemGlobalAngleY = itemLocalAngleY * Math.Cos(itemGlobalAngleR) + itemLocalAngleP * Math.Sin(itemGlobalAngleR)
	Float itemGlobalAngleP = itemLocalAngleP * Math.Cos(itemGlobalAngleR) - itemLocalAngleY * Math.Sin(itemGlobalAngleR)

  ; Create a vector from the machine to nodeToSpawnItemAtName (by default, the glass tray cover origin) and scale it for some additional offset.
  ;  Otherwise the item will get stuck behind the glass
  Float[] itemGlobalPos = item.GetSpacePosition()
  itemGlobalPos[0] = (itemGlobalPos[0] - objectPos[0]) * itemSpawnOffsetHorizontalScale
  itemGlobalPos[1] = (itemGlobalPos[1] - objectPos[1]) * itemSpawnOffsetHorizontalScale
  itemGlobalPos[2] = (itemGlobalPos[2] - objectPos[2]) * itemSpawnOffsetVerticalScale

  ; Move the item there and give it a better rotation
	item.SetAngle(itemGlobalAngleY, itemGlobalAngleP, itemGlobalAngleR)
  item.MoveTo(Self, itemGlobalPos[0], itemGlobalPos[1], itemGlobalPos[2], False, False)

  ; Wait for the item to move & rotate, then show it
  Utility.Wait(0.4)
  item.Enable(True)
EndFunction

LeveledItem Function GetItemToVend()
  If rareItemsToVend as Bool && rareItemChance > 0.0 ; #DEBUG_LINE_NO:91
    Bool rareSuccess = Utility.RandomFloat(0.0, 100.0) <= rareItemChance ; #DEBUG_LINE_NO:94
    If rareSuccess ; #DEBUG_LINE_NO:95
      Return rareItemsToVend ; #DEBUG_LINE_NO:97
    Else
      Return itemsToVend ; #DEBUG_LINE_NO:100
    EndIf
  Else
    Return itemsToVend ; #DEBUG_LINE_NO:104
  EndIf
EndFunction

Function PlayErrorSound()
  Int S = WwiseEvent_VendingMachineActivateError.Play(Self as ObjectReference, None, None) ; #DEBUG_LINE_NO:109
EndFunction

Function PlayDispenseSound()
  Bool S = WwiseEvent_VendingMachineActivate.Play(Self as ObjectReference, None, None) ; #DEBUG_LINE_NO:113
EndFunction

;-- State -------------------------------------------
State Done

  Event OnActivate(ObjectReference akActivator)
    Self.PlayErrorSound() ; #DEBUG_LINE_NO:172
  EndEvent
EndState

;-- State -------------------------------------------
Auto State Initial

  Event OnLoad()
    Self.ResetVendingMachine() ; #DEBUG_LINE_NO:121
    Self.GoToState("Ready") ; #DEBUG_LINE_NO:122
  EndEvent
EndState

;-- State -------------------------------------------
State Ready

  Event OnActivate(ObjectReference akActivator)
    Self.GoToState("Working") ; #DEBUG_LINE_NO:128
    If akActivator == Game.GetPlayer() as ObjectReference ; #DEBUG_LINE_NO:129
      Self.TryToVendItem() ; #DEBUG_LINE_NO:130
    EndIf
    If usedActivations >= allowedActivations ; #DEBUG_LINE_NO:135
      Self.GoToState("SoldOut") ; #DEBUG_LINE_NO:136
    Else
      Self.GoToState("Ready") ; #DEBUG_LINE_NO:138
    EndIf
  EndEvent
EndState

;-- State -------------------------------------------
State SoldOut

  Event OnActivate(ObjectReference akActivator)
    Self.PlayErrorSound()
  EndEvent

  Event OnLoad()
    If Utility.GetCurrentGameTime() >= Self.GetValue(VendingMachineResetGameTimeAV)
      Self.ResetVendingMachine()
      Self.GoToState("Ready")
    EndIf
  EndEvent

  Event OnBeginState(String asOldState)
    If daysUntilReset > 0 ; #DEBUG_LINE_NO:151
      Self.SetValue(VendingMachineResetGameTimeAV, Utility.GetCurrentGameTime() + daysUntilReset as Float) ; #DEBUG_LINE_NO:152
    Else
      Self.GoToState("Done") ; #DEBUG_LINE_NO:154
    EndIf
  EndEvent
EndState

;-- State -------------------------------------------
State Working

  Event OnActivate(ObjectReference akActivator)
    ; Empty function
  EndEvent
EndState
