#include "PCH.h"

#include "sfse/GameData.h"
#include "sfse/GameObjects.h"
#include "sfse/GameReferences.h"
#include "sfse/GameRTTI.h"
#include "sfse/GameMenu.h"
#include "sfse_common/Log.h"

#include "Ini/Ini.h"
#include "nlohmann/json.hpp"
#include <RE/S/Script.h>
#include <atlstr.h>
#include <iostream>
#include <shlobj.h>
#include <string.h>
#include <windows.h>
#include <shellapi.h>
#pragma comment(lib, "shell32.lib")

#include <sfse/GameSettings.h>
#include <sfse_common/Errors.h>
#include <sfse_common/FileStream.h>

#include <locale>

// Console
#include <sfse/GameConsole.h>
#include <sfse/GameScript.h>









///////////////////////////////////////////////////////////  Ini  ///////////////////////////////////////////////////////////


LKIniWrite::LKIniWrite(const char* szFileName)
{
    memset(m_szFileName, 0x00, 255);
    memcpy(m_szFileName, szFileName, strlen(szFileName));
}

void LKIniWrite::WInt(const char* Section, const char* Key, int iValue)
{
    char chValue[255];
    sprintf_s(chValue, "%d", iValue);
    WritePrivateProfileString(CA2CT(Section), CA2CT(Key), CA2CT(chValue), CA2CT(m_szFileName));
}

void LKIniWrite::WFloat(const char* Section, const char* Key, float fValue)
{
    char chValue[255];
    sprintf_s(chValue, "%f", fValue);
    WritePrivateProfileString(CA2CT(Section), CA2CT(Key), CA2CT(chValue), CA2CT(m_szFileName));
}

void LKIniWrite::WBool(const char* Section, const char* Key, bool bValue)
{
    char chValue[255];
    sprintf_s(chValue, "%s", bValue ? "1" : "0");
    WritePrivateProfileString(CA2CT(Section), CA2CT(Key), CA2CT(chValue), CA2CT(m_szFileName));
}

void LKIniWrite::WStr(const char* Section, const char* Key, const char* chValue)
{
    WritePrivateProfileString(CA2CT(Section), CA2CT(Key), CA2CT(chValue), CA2CT(m_szFileName));
}

LKIniRead::LKIniRead(const char* szFileName)
{
    memset(m_szFileName, 0x00, 255);
    memcpy(m_szFileName, szFileName, strlen(szFileName));
}

inline int LKIniRead::RInt(const char* Section, const char* Key, int iDefValue)
{
    int iValue = GetPrivateProfileInt(Section, Key, iDefValue, m_szFileName);
    return iValue;
}

inline float LKIniRead::RFloat(const char* Section, const char* Key, float fDefValue)
{
    char  out[255];
    char  def[255];
    float fValue;
    sprintf_s(def, "%f", fDefValue);
    GetPrivateProfileString(Section, Key, def, out, 255, m_szFileName);
    fValue = static_cast<float>(atof(out));
    return fValue;
}

inline bool LKIniRead::RBool(const char* Section, const char* Key, bool bDefValue)
{
    char out[255];
    char def[255];
    bool bValue;
    sprintf_s(def, "%s", bDefValue ? "1" : "0");
    GetPrivateProfileString(Section, Key, def, out, 255, m_szFileName);
    bValue = (strcmp(out, "1") == 0 || strcmp(out, "True") == 0 || strcmp(out, "true") == 0) ? 1 : 0;
    return bValue;
}

inline const char* LKIniRead::RStr(const char* Section, const char* Key, const char* chDefValue)
{
    char* c   = new char[10000];
    LPSTR out = c;
    memset(out, 0, 10000);
    GetPrivateProfileString(Section, Key, chDefValue, out, 10000, m_szFileName);
    return out;
}

inline static bool DoesIniExist(std::string FileName, std::string bOptionalSubfolder = "")
{
    if (FileName.length() <= 4)
        return false;
    std::string pt1 = ".\\Data\\SFSE\\Plugins\\";
    if (!bOptionalSubfolder.empty())
        pt1 = bOptionalSubfolder;
    std::string pt2  = FileName.c_str();
    std::string pt3  = ".ini";
    const auto  path = pt1 + pt2 + pt3;
    if (std::filesystem::exists(path.c_str()) == true) {
        return true;
    }
    return false;
}

inline static bool WriteIni(std::string FileName, std::string Section, std::string Key, std::string Value, std::string bOptionalSubfolder = "")
{
    if (FileName.length() <= 4 || Section.empty() || Key.empty())
        return false;
    std::string pt1 = ".\\Data\\SFSE\\Plugins\\";
    if (!bOptionalSubfolder.empty())
        pt1 = bOptionalSubfolder;
    std::string pt2  = FileName.c_str();
    std::string pt3  = ".ini";
    const auto  path = pt1 + pt2 + pt3;
    LKIniWrite  iniWriter(path.c_str());
    iniWriter.WStr(Section.c_str(), Key.c_str(), Value.c_str());
    if (std::filesystem::exists(path.c_str()) == true) {
        LKIniRead   INI(path.c_str());
        std::string theString = INI.RStr(Section.c_str(), Key.c_str(), Value.c_str());
        if (strcmp(theString.c_str(), Value.c_str()) == 0) {
            return true;
        }
    }
    return false;
}

inline static std::string ReadIni(std::string FileName, std::string Section, std::string Key, std::string bOptionalSubfolder = "")
{
    if (FileName.length() <= 4 || Section.empty() || Key.empty())
        return "";
    std::string DefValue = "";
    std::string pt1      = ".\\Data\\SFSE\\Plugins\\";
    if (!bOptionalSubfolder.empty())
        pt1 = bOptionalSubfolder;
    std::string pt2  = FileName.c_str();
    std::string pt3  = ".ini";
    const auto  path = pt1 + pt2 + pt3;
    if (std::filesystem::exists(path.c_str()) == false) {
        return "";
    }
    LKIniRead   INI(path.c_str());
    std::string ReturnString = INI.RStr(Section.c_str(), Key.c_str(), DefValue.c_str());
    if (ReturnString.empty() || ReturnString.length() <= 0) {
        return "";
    }
    return ReturnString;
}










///////////////////////////////////////////////////////////  UTILITY  ///////////////////////////////////////////////////////////


inline static unsigned long long baseAddr = 0;


namespace Classes
{
    class __declspec(novtable) BGSObjectInstanceExtra : public RE::BSExtraData
    {
    public:
        static constexpr auto RTTI{ RE::RTTI::BGSObjectInstanceExtra };
        static constexpr auto VTABLE{ RE::VTABLE::BGSObjectInstanceExtra };
        static constexpr auto TYPE{ RE::ExtraDataType::kObjectInstance };

        // members
        std::uint32_t modCount{ 0 }; // 1C
        // const RE::BSTArray<RE::BGSMod::Attachment::Mod> mods;            // 20
        void* mods;
    };

    class __declspec(novtable) ExtraLeveledCreature : public RE::BSExtraData
    {
    public:
        static constexpr auto RTTI{ RE::RTTI::ExtraLeveledCreature };
        static constexpr auto VTABLE{ RE::VTABLE::ExtraLeveledCreature };
        static constexpr auto TYPE{ RE::ExtraDataType::kLeveledCreature };

        // members
        RE::TESNPC* leveledActorBase;
    };

    class __declspec(novtable) ExtraTextDisplayData : public RE::BSExtraData
    {
    public:
        static constexpr auto RTTI{ RE::RTTI::ExtraTextDisplayData };
        static constexpr auto VTABLE{ RE::VTABLE::ExtraTextDisplayData };
        static constexpr auto TYPE{ RE::ExtraDataType::kTextDisplayData };

        // members
        RE::BSFixedString displayName; // 18
        std::uint64_t     unk2{ 0 };
        std::uint64_t     unk3{ 0 };
        std::uint64_t     unk4{ 0 };
        std::uint64_t     unk5{ 0 };
        std::uint16_t     unk6{ 0 }; // customNameLength{ 0 }; // 4C
    };

    class __declspec(novtable) ExtraOriginalBaseObject : public RE::BSExtraData
    {
    public:
        static constexpr auto RTTI{ RE::RTTI::ExtraOriginalBaseObject };
        static constexpr auto VTABLE{ RE::VTABLE::ExtraOriginalBaseObject };
        static constexpr auto TYPE{ RE::ExtraDataType::kOriginalItemBase };

        // members
        uint32_t originalBaseObjectID;
    };
    
}

namespace Helpers
{

    inline static void ExecuteCommand(const char* asCommand)
    {
        static REL::Relocation<void**> ConsoleRTTI{ REL::ID(808335) };
        using func_t = void (*)(const void*, const char*);
        REL::Relocation<func_t> func{ REL::ID(166307) };
        func(*ConsoleRTTI, asCommand);
    }

    inline static std::string GetFormIDAsHex(RE::TESForm* paramForm, bool beNice = true)
    {
        if (!paramForm)
            return "";
        std::string s = "";
        s             = std::format("{:x}", paramForm->GetFormID());
        if (s.empty())
            return "";
        if (!beNice)
            return s;
        std::transform(s.begin(), s.end(), s.begin(), [](unsigned char c) -> char { return static_cast<char>(::toupper(c)); });
        std::stringstream ss;
        ss << std::setw(8) << std::setfill('0') << s;
        std::string s2 = ss.str();
        return s2;
    }

    inline static std::string GetUIntFormIDAsHex(uint32_t formID, bool beNice = true)
    {
        if (formID <= 0 || formID > UINT32_MAX)
            return "";
        std::string s = "";
        s             = std::format("{:x}", formID);
        if (s.empty())
            return "";
        if (!beNice)
            return s;
        std::transform(s.begin(), s.end(), s.begin(), [](unsigned char c) -> char { return static_cast<char>(::toupper(c)); });
        std::stringstream ss;
        ss << std::setw(8) << std::setfill('0') << s;
        std::string s2 = ss.str();
        return s2;
    }

    inline static std::string GetAddressAsHex(unsigned long long address, bool bAsOffset = false)
    {
        if (address <= 0)
            return "";
        std::string s = "";
        if (bAsOffset) {
            auto baseAddr = (unsigned long long)GetModuleHandle("Starfield.exe");
            if (baseAddr > 0) {
                auto offset = address - baseAddr;
                if (offset <= 0) {
                    logger::error("GetAddressAsHex: error: calculated offset <= 0 ???");
                }
                else {
                    s = std::format("{:x}", offset);
                    if (s.empty())
                        return "";
                    std::transform(s.begin(), s.end(), s.begin(), [](unsigned char c) -> char { return static_cast<char>(::toupper(c)); });
                    s = "Starfield.exe + 0x" + s;
                    return s;
                }
            }
        }
        else {
            s = "";
        }
        s = std::format("{:x}", address);
        if (s.empty())
            return "";
        std::transform(s.begin(), s.end(), s.begin(), [](unsigned char c) -> char { return static_cast<char>(::toupper(c)); });
        return s;
    }

    inline static std::string GetTESFullName(RE::TESForm* paramForm)
    {
        if (!paramForm) {
            logger::error("GetTESFullName: error: paramForm is nullptr.");
            return "";
        }
        TESForm* Form = reinterpret_cast<TESForm*>(paramForm); // is it necessary? Yes...
        if (!Form)
            return "";

        auto FormType = Form->GetSavedFormType();
        if (FormType == 74 || FormType == 75) {
            RE::TESObjectREFR* paramRef = (RE::TESObjectREFR*)paramForm;
            if (paramRef != nullptr) {
                paramForm = paramRef->GetBaseObject();
            }
            else {
                logger::error("GetTESFullName: error: FormType is Ref but paramRef is nullptr.");
            }
        }
        if (FormType == 13 || FormType == 30 || (FormType >= 34 && FormType <= 37) || FormType == 40 || (FormType >= 46 && FormType <= 50) || (FormType >= 53 && FormType <= 54)
            || FormType == 57 || FormType == 62 || FormType == 91 || FormType == 106 || FormType == 152 || FormType == 159)
        {
            std::string Name = dynamic_cast<TESFullName*>(Form)->strFullName.c_str();
            if (Name.empty() == false && Name.size() > 0) {
              //  logger::info("GetTESFullName: info: returning = {}", Name);
                return Name;
            }
        }
        else {
            logger::error("GetTESFullName: error: FormType is not supported.");
        }
        logger::error("GetTESFullName: error: unknown.");
        return "";
    }

}

inline static std::string ToLowerStr(std::string _sourceString)
{
    std::transform(_sourceString.begin(), _sourceString.end(), _sourceString.begin(), [](unsigned char c) -> char { return static_cast<char>(::tolower(c)); });
    return _sourceString;
}

inline static RE::TESObjectREFR* CastAsRERef(TESObjectREFR* Ref)
{
    if (Ref != nullptr)
        return reinterpret_cast<RE::TESObjectREFR*>(Ref);
    return nullptr;
}

inline static RE::TESForm* CastAsREForm(TESForm* Form)
{
    if (Form != nullptr)
        return reinterpret_cast<RE::TESForm*>(Form);
    return nullptr;
}

inline static unsigned long long GetFormAddress(RE::TESForm* Form)
{
    if (Form != nullptr)
        return reinterpret_cast<unsigned long long>(Form);
    return 0;
}










///////////////////////////////////////////////////////////  PLUGIN  ///////////////////////////////////////////////////////////

inline static bool bUseMaxShipTransferDistance = false;
bool bLogging = false;


inline static void LogInfo(const std::string& message, const std::string& callerfunction = __builtin_FUNCTION())
{
    if (bLogging)
        logger::info("{}", "[" + callerfunction + "] " + message);
}

inline static void LogError(const std::string& message, const std::string& callerfunction = __builtin_FUNCTION())
{
    if (bLogging)
        logger::error("{}", "[" + callerfunction + "] " + message);
}

inline static void LogWarning(const std::string& message, const std::string& callerfunction = __builtin_FUNCTION())
{
    if (bLogging)
        logger::warn("{}", "[" + callerfunction + "] " + message);
}

inline static RE::Actor* GetPlayerActor()
{
    RE::TESForm* PlayerForm = RE::TESForm::LookupByID(0x14);
    if (!PlayerForm)
        return nullptr;
    RE::Actor* PlayerActor = (RE::Actor*)PlayerForm;
    if (PlayerActor != nullptr && PlayerActor->GetSavedFormType() == 75)
        return PlayerActor;
    return nullptr;
}

inline static bool HasKeyword(RE::TESForm* theForm, RE::BGSKeyword* theKeyword)
{
    if (!theForm || !theKeyword)
        return false;
    TESForm* theForm2 = reinterpret_cast<TESForm*>(theForm);
    if (!theForm2)
        return false;
    BGSKeywordForm* KeywordForm = DYNAMIC_CAST(theForm2, TESForm, BGSKeywordForm);
    if (!KeywordForm)
        return false;
    RE::BGSKeywordForm* KeywordForm2 = reinterpret_cast<RE::BGSKeywordForm*>(KeywordForm);
    if (!KeywordForm2)
        return false;
    for (uint32_t i = 0; i < KeywordForm2->GetNumKeywords(); i++) {
        if (KeywordForm2->keywords[i] && KeywordForm2->keywords[i]->formID == theKeyword->formID) {
            return true;
        }
    }
    return false;
}

inline static Script::SCRIPT_FUNCTION* GetFirstConsoleCommand()
{
    // Starfield.exe + 0x0557F0D0 in v1.8.88.0 (REL::ID = 841465). Script::SCRIPT_FUNCTION*.
    static REL::Relocation<unsigned long long> FirstConsoleCommand{ REL::ID(841465) };
    unsigned long long                         k = FirstConsoleCommand.get();
    Script::SCRIPT_FUNCTION*                   j = reinterpret_cast<Script::SCRIPT_FUNCTION*>(k);
    if (!j) {
        LogError("GetFirstConsoleCommand: error: found command is nullptr.");
        return nullptr;
    }
    else {
        LogInfo("GetFirstConsoleCommand: info: found first console command.");
        return j;
    }
    return nullptr;
}

inline static RE::BGSKeyword* GetElementalBlastAllowedResourceKeyword()
{
    RE::TESForm* theForm = RE::TESForm::LookupByID(0x0006FB66);
    if (!theForm) {
        LogError("theForm is nullptr.");
        return nullptr;
    }
    RE::BGSKeyword* theKeywordForm = (RE::BGSKeyword*)theForm;
    if (!theKeywordForm) {
        LogError("theKeywordForm is nullptr.");
        return nullptr;
    }
    if (theKeywordForm->GetSavedFormType() != 4) {
        LogError("theKeywordForm is not BGSKeyword.");
        return nullptr;
    }
    return theKeywordForm;
}

inline static RE::BGSKeyword* GetDFOBCutterKeyword()            // KeywordCutter_DO [DFOB:00338659] points to ma_Cutter [KYWD:001F70DF] in vanilla
{
    RE::TESForm* theForm = RE::TESForm::LookupByID(0x00338659); // use the form (should be keyword) the DFOB points to for better mod compatibility just because you never know
    if (!theForm) {
        LogError("theForm is nullptr.");
        return nullptr;
    }
    RE::BGSDefaultObject* theDefaultObject = (RE::BGSDefaultObject*)theForm;
    if (!theDefaultObject) {
        LogError("theDefaultObject is nullptr.");
        return nullptr;
    }
    if (theDefaultObject->GetSavedFormType() != 121) {
        LogError("theKeywordForm is not BGSDefaultObject.");
        return nullptr;
    }
    auto DefaultObjectForm = theDefaultObject->object;
    if (!DefaultObjectForm) {
        LogError("DefaultObjectForm is nullptr (DFOB does not point to any form).");
        return nullptr;
    }
    if (DefaultObjectForm->GetSavedFormType() != 4) {
        LogError("DefaultObjectForm is not BGSKeyword.");
        return nullptr;
    }
    RE::BGSKeyword* DefaultObjectKeywordForm = (RE::BGSKeyword*)DefaultObjectForm;
    if (!DefaultObjectKeywordForm) {
        LogError("DefaultObjectKeywordForm is nullptr.");
        return nullptr;
    }
    return DefaultObjectKeywordForm;
}

inline static std::string GetGameSetting(std::string sSetting, int iSettingType) // iSettingType: 1 = float, 2 = int, 3 = bool
{
    if (sSetting.empty() == false && sSetting.size() > 0) {
        auto Set1 = (*SettingT<GameSettingCollection>::pCollection)->GetSetting(sSetting.c_str());
        if (Set1) {
            if (iSettingType == 1) {
                return std::to_string(Set1->data.f32);
            }
            else if (iSettingType == 2) {
                return std::to_string(Set1->data.u32);
            }
            else if (iSettingType == 3) {
                return std::to_string(Set1->data.u8);
            }
            else if (iSettingType == 4) {
                return Set1->data.s;
            }
        }
    }
    return "";
}

inline static RE::TESObjectREFR* GetPlayerHomeSpaceship()
{
    /*
    uintptr_t   baseAddr     = (uintptr_t)GetModuleHandle("Starfield.exe"); // vanilla, non-CLib
    uintptr_t   relativeAddr = baseAddr + 0x2A5DBA0;        // v1.9.67.
    const auto  func         = reinterpret_cast<RE::TESObjectREFR* (*)()>(relativeAddr);
    return func();
    */
    REL::Relocation<uintptr_t> funcPtr{ REL::ID(171075) };
    uintptr_t                  addr = funcPtr.address();
    LogInfo("funcPtr.address " + std::to_string(funcPtr.address()));        // without these LogInfo calls the function crashes (??)
    LogInfo("funcPtr.offset " + std::to_string(funcPtr.offset()));
    LogInfo("funcPtr.get " + std::to_string(funcPtr.get()));
    const auto refPtr = reinterpret_cast<RE::TESObjectREFR** (*)()>(addr);
    auto RefPtr = refPtr();
    if (!RefPtr) {
        LogError("RefPtr is nullptr");
        return nullptr;
    }
    auto Ref = *RefPtr;
    if (!Ref) {
        LogError("Ref is nullptr");
        return nullptr;
    }
    return Ref;
}

inline static bool WornHasKeyword(RE::Actor* theActor, RE::BGSKeyword* theKeyword)
{
    if (!theActor || !theKeyword) {
        LogError("mandatory func param is nullptr.");
        return false;
    }
    REL::Relocation<uintptr_t> funcPtr{ REL::ID(106992) };
    uintptr_t                  addr = funcPtr.address();
    LogInfo("funcPtr.address " + std::to_string(funcPtr.address()));
    LogInfo("funcPtr.offset " + std::to_string(funcPtr.offset()));
    LogInfo("funcPtr.get " + std::to_string(funcPtr.get()));
    const auto func = reinterpret_cast<int (*)(RE::Actor*, RE::BGSKeyword*)>(addr);
    return func(theActor, theKeyword);
}

inline static int64_t GetWornKeywordItemCount(RE::Actor* theActor, RE::BGSKeyword* theKeyword, BYTE unk1 = 0, BYTE unk2 = 1, int64_t unk3 = 0)
{
    if (!theActor || !theKeyword) {
        LogError("mandatory func param is nullptr.");
        return false;
    }
    REL::Relocation<uintptr_t> funcPtr{ REL::ID(106610) };
    uintptr_t                  addr = funcPtr.address();
    LogInfo("funcPtr.address " + std::to_string(funcPtr.address()));
    LogInfo("funcPtr.offset " + std::to_string(funcPtr.offset()));
    LogInfo("funcPtr.get " + std::to_string(funcPtr.get()));
    const auto func = reinterpret_cast<int (*)(RE::Actor*, RE::BGSKeyword*, bool, bool, int64_t)>(addr);
    return func(theActor, theKeyword, false, true, false);
}

inline static void HandleOnPlayerItemAdded(uint32_t sourceContainerID, uint32_t itemFormID, uint32_t itemCount)
{
    // To prevent unexpected function calls
    auto UISingleton = UI::GetSingleton();
    if (!UISingleton) {
        LogError("UISingleton is nullptr.");
        return;
    }
    bool bMenusOK = false;
    if (UISingleton->IsMenuOpenCustom("MonocleMenu") == true && UISingleton->IsMenuOpenCustom("ContainerMenu") == false && UISingleton->IsMenuOpenCustom("BarterMenu") == false && UISingleton->IsMenuOpenCustom("WorkshopMenu") == false && UISingleton->IsMenuOpenCustom("SpaceshipHudMenu") == false && UISingleton->IsMenuOpenCustom("MainMenu") == false && UISingleton->IsMenuOpenCustom("PauseMenu") == false)
    {
        bMenusOK = true;
    }
    if (!bMenusOK) {
        LogError("MenuCheck error.");
        return;
    }

    LogInfo("called with params: sourceContainerID [" + std::to_string(sourceContainerID) + " | " + Helpers::GetUIntFormIDAsHex(sourceContainerID) + "], itemFormID [" + std::to_string(itemFormID) + " | " + Helpers::GetUIntFormIDAsHex(itemFormID) + "].");

    if (GetKeyState(0x01) & 0x8000) { // left mouse button
    }
    else {
        LogError("GetKeyState error.");
        return;
    }

    // PlayerRef
    auto         PlayerRef  = GetPlayerActor();
    if (!PlayerRef) {
        LogError("PlayerRef is nullptr.");
        return;
    }

    LogInfo("checking item...");

    // Item
    RE::TESForm* itemBaseForm = RE::TESForm::LookupByID(itemFormID);
    if (!itemBaseForm) {
        LogError("itemBaseForm is nullptr.");
        return;
    }
    auto itemFormType = itemBaseForm->GetSavedFormType();
    if (itemFormType != 40) {
        LogError("itemBaseForm is not MISC.");
        return;
    }
    RE::TESObjectMISC* MiscItem = (RE::TESObjectMISC*)itemBaseForm;
    if (!MiscItem) {
        LogError("MiscItem is nullptr.");
        return;
    }
    std::string MiscItemHexFormID = Helpers::GetFormIDAsHex(MiscItem);
    if (MiscItemHexFormID.empty()) {
        LogError("MiscItemHexFormID is empty.");
        return;
    }
    auto ElementalBlastAllowedResource = GetElementalBlastAllowedResourceKeyword();
    if (!ElementalBlastAllowedResource) {
        LogError("ElementalBlastAllowedResource is nullptr.");
        return;
    }
    if (HasKeyword(itemBaseForm, ElementalBlastAllowedResource) == false) {
        LogError("itemBaseForm HasKeyword 'ElementalBlastAllowedResource' == False.");
        return;
    }
    std::string MiscItemName = Helpers::GetTESFullName(MiscItem);
    if (MiscItemName.empty()) {
        LogError("MiscItemName is empty. Setting name to 'Item'...");
    }

    LogInfo("checking itemCount...");

    // Count
    std::string sItemCount = std::to_string(itemCount);
    if (sItemCount.empty()) {
        LogError("sItemCount is empty.");
        return;
    }

    LogInfo("checking ship...");

    // Ship
    auto HomeShip = GetPlayerHomeSpaceship();
    if (!HomeShip) {
        LogError("Player has no HomeSpaceship");
        return;
    }
    else {
        LogInfo("Player's HomeSpaceship is " + Helpers::GetAddressAsHex(reinterpret_cast<unsigned long long>(HomeShip)));
    }
    auto SpaceshipFormType = HomeShip->GetSavedFormType();
    if (SpaceshipFormType != 74) { // 'SpaceshipReference' is TESObjectREFR
        LogError("Player's HomeSpaceship is not TESObjectREFR. It has formType [" + std::to_string(SpaceshipFormType) + "].");
        return;
    }
    std::string HomeShipHexFormID = Helpers::GetFormIDAsHex(HomeShip);
    if (HomeShipHexFormID.empty()) {
        LogError("HomeShipHexFormID is empty.");
        return;
    }

    LogInfo("checking source...");

    // Source
    if (sourceContainerID > 0) {
        LogError("item has sourceContainer (i.e. not transferred by breaking it up with the Cutter). sourceContainerID is [" + std::to_string(sourceContainerID) + " | " + Helpers::GetUIntFormIDAsHex(sourceContainerID) + "].");
        return;
    }

    // Distance check (optional, off by default, can be enabled in INI)
    if (bUseMaxShipTransferDistance) {
        std::string sMaxShipTransferDistance = GetGameSetting("fMaxShipTransferDistance", 1);
        if (!sMaxShipTransferDistance.empty() && sMaxShipTransferDistance.find_first_not_of(".0123456789") == std::string::npos) {
            float fMaxShipTransferDistance = -1.00f;
            try {
                fMaxShipTransferDistance = std::stof(sMaxShipTransferDistance);
            }
            catch (...) {
                LogError("std::stof exception thrown for string '" + sMaxShipTransferDistance + "'. Skipping distance check...");
            }
            if (fMaxShipTransferDistance > 0) {
                float Distance = PlayerRef->data.location.GetDistance(HomeShip->data.location);
                if (Distance <= 0)
                    LogError("PlayerRef.GetDistance(HomeShip) is 0. (Unknown error: this distance shouldn't be 0). Skipping distance check...");
                else {
                    if (Distance > fMaxShipTransferDistance) {
                        LogWarning("PlayerRef.GetDistance(HomeShip) is (" + std::to_string(Distance) + ") while fMaxShipTransferDistance is (" + std::to_string(fMaxShipTransferDistance) + "). Exiting function...");
                        return;
                    }
                    else
                        LogInfo("PlayerRef.GetDistance(HomeShip) is (" + std::to_string(Distance) + ") while fMaxShipTransferDistance is (" + std::to_string(fMaxShipTransferDistance) + "). Distance-check OK.");
                }
            }
        }
    }

    LogInfo("checking inventory...");

    // Inventory
    auto InventoryList = PlayerRef->inventoryList.lock_read().operator->();
    if (!InventoryList) {
        LogError("InventoryList of PlayerRef is nullptr.");
        return;
    }
    uint32_t InventoryItemCount = InventoryList->data.size();
    if (InventoryItemCount <= 0 || InventoryItemCount > UINT32_MAX) {
        LogError("InventoryItemCount is out of range.");
        return;
    }

    LogInfo("checking DFOB...");

    // Cutter, DFOB
    auto CutterKeyword = GetDFOBCutterKeyword();
    if (!CutterKeyword) {
        LogError("CutterKeyword is nullptr.");
        return;
    }

    LogInfo("checking WornHasKeyword...");

    // Cutter equipped?
    auto CutterEquipped = WornHasKeyword(PlayerRef, CutterKeyword);
    if (!CutterEquipped) {
        LogError("PlayerRef.WornHasKeyword 'CutterKeyword' == False.");
        return;
    } else
        LogInfo("PlayerRef.WornHasKeyword 'CutterKeyword' == True.");

    LogInfo("checking GetWornKeywordItemCount...");

    // Cutter equipped? Pt. 2
    auto KeywordItemCount = GetWornKeywordItemCount(PlayerRef, CutterKeyword, 0, 1, 0);
    if (KeywordItemCount <= 0) {
        LogError("PlayerRef.WornKeywordItemCount 'CutterKeyword' == 0.");
        return;
    }
    else
        LogInfo("PlayerRef.WornKeywordItemCount 'CutterKeyword' == " + std::to_string(KeywordItemCount) + ".");

    LogInfo("beginning loop...");

    // Transfer 1 item to HomeShip
    for (uint32_t i = 0; i < InventoryList->data.size(); i++) {
        for (int j = InventoryList->data[i].stacks.size() - 1; j >= 0; j--) {

            auto& ExtraData    = InventoryList->data[i].stacks[j].extra;
            auto& Count        = InventoryList->data[i].stacks[j].count;
            auto& ItemBoundObj = InventoryList->data[i].object;

            if (ItemBoundObj->formID == itemBaseForm->formID) {

                if (InventoryList->data[i].IsEquipped()) {
                    LogError("found matching formID [" + MiscItemHexFormID + "] in Player's inventory at itemIndex [" + std::to_string(i) + "] on Stack [" + std::to_string(j) + "]. (Unknown error: MiscItems cannot be equipped). Exiting loop.");
                    return;
                } else
                    LogInfo("found matching formID [" + MiscItemHexFormID + "] in Player's inventory at itemIndex [" + std::to_string(i) + "] on Stack [" + std::to_string(j) + "]. Transferring to ship...");

                // AddItem (native AddItem's second parameter is abSilent but it's not important)
                std::string asCommand = HomeShipHexFormID + ".AddItem " + MiscItemHexFormID + " " + sItemCount;
                Helpers::ExecuteCommand(asCommand.c_str());

                // RemoveItem (silently; second param is 'Flag' not abSilent? Call Papyrus.)
                asCommand = "14.CF \"ObjectReference.RemoveItem\" " + MiscItemHexFormID + " " + sItemCount + " true";
                Helpers::ExecuteCommand(asCommand.c_str());

                return;
            }
        }
    }

    LogError("function ended unexpectedly.");

}

class AutoAddedItemTransferTESContainerChangedEventClass : public BSTEventSink<TESContainerChangedEvent>
{
public:
    virtual ~AutoAddedItemTransferTESContainerChangedEventClass(){};

    virtual BSEventNotifyControl ProcessEvent(const TESContainerChangedEvent& _event, BSTEventSource<TESContainerChangedEvent>* _source) override // 01 override
    {
        if (_event.targetContainerFormID == 0x14 && _event.refFormID <= 0 && _event.itemFormID > 0 && _event.itemFormID < UINT32_MAX && _event.count == 1)      // breaking up a deposit should always mean _event.count == 1
            HandleOnPlayerItemAdded(_event.sourceContainerFormID, _event.itemFormID, _event.count);

        return BSEventNotifyControl::kContinue;
    }
};

AutoAddedItemTransferTESContainerChangedEventClass AutoAddedItemTransferTESContainerChangedEventSink;

namespace Thread
{

    static bool bMonocleMenuOpen = false;
    bool               bThreadLock            = false;
    bool               bFuncLock              = false;

    static void TryToSetItemAddedGS();

    static DWORD Thread_TryToSetItemAddedGS(void* param)
    {
        (void)param;
        while (bMonocleMenuOpen == true) {
            Sleep(500);
            if (bMonocleMenuOpen != true)
                break;
            TryToSetItemAddedGS();
        }
        logger::info("Thread_TryToSetItemAddedGS: info: exiting.");
        return 0;
    }

    // Change the 'AddedItem' HUDMessage accordingly. GameSetting 'SAddItemtoInventory' (default value: 'Added.').
    static void TryToSetItemAddedGS()
    {
        if (!bMonocleMenuOpen || GetForegroundWindow() != FindWindowA(NULL, "Starfield") || bFuncLock)
            return;
        else
            bFuncLock = true;

        bool bShouldPatchGS = true;

        if (GetKeyState(0x01) & 0x8000) { // left mouse button

        }
        else
            bShouldPatchGS = false;

        if (bShouldPatchGS) {
            auto PlayerRef = GetPlayerActor();
            if (!PlayerRef) {
                bShouldPatchGS = false;
            }
            auto CutterKeyword = GetDFOBCutterKeyword();
            if (!CutterKeyword) {
                bShouldPatchGS = false;
            }
            auto CutterEquipped = WornHasKeyword(PlayerRef, CutterKeyword);
            if (!CutterEquipped) {
                bShouldPatchGS = false;
            }
        }

        std::string sCurrentGSVal = GetGameSetting("SAddItemtoInventory", 4);

        if (bShouldPatchGS && !sCurrentGSVal.empty() && std::strcmp(sCurrentGSVal.c_str(), "Added to your ship.") != 0) {
            std::string asCommand = "SetGS \"SAddItemtoInventory\" \"Added to your ship.\"";
            Helpers::ExecuteCommand(asCommand.c_str());
            LogInfo("called command [" + asCommand + "].");
        }
        else if (!bShouldPatchGS && !sCurrentGSVal.empty() && std::strcmp(sCurrentGSVal.c_str(), "Added.") != 0) {
            std::string asCommand = "SetGS \"SAddItemtoInventory\" \"Added.\"";
            Helpers::ExecuteCommand(asCommand.c_str());
            LogInfo("called command [" + asCommand + "].");
        }

        bFuncLock = false;
    }

    class AutoAddedItemTransferMenuOpenCloseEventClass : public RE::BSTEventSink<RE::MenuOpenCloseEvent>
    {
        RE::BSEventNotifyControl ProcessEvent(const RE::MenuOpenCloseEvent& a_event, RE::BSTEventSource<RE::MenuOpenCloseEvent>*)
        {
            if (std::strcmp(a_event.menuName.c_str(), "MonocleMenu") == 0) {
                if (a_event.opening) {
                    Thread::bMonocleMenuOpen = true;
                    Thread::bFuncLock        = false;
                    if (Thread::bThreadLock == false && CreateThread(NULL, 8192, Thread::Thread_TryToSetItemAddedGS, NULL, 0, NULL) != NULL) {
                        LogInfo("::MenuOpenCloseEvent: created thread.");
                        Thread::bThreadLock = true;
                    }
                    else
                        LogInfo("::MenuOpenCloseEvent: couldn't create thread.");
                }
                else {
                    Thread::bMonocleMenuOpen = false;
                    Thread::bThreadLock      = false;
                    // failsafe
                    std::string sCurrentGSVal = GetGameSetting("SAddItemtoInventory", 4);
                    if (!sCurrentGSVal.empty() && std::strcmp(sCurrentGSVal.c_str(), "Added.") != 0) {
                        std::string asCommand = "SetGS \"SAddItemtoInventory\" \"Added.\"";
                        Helpers::ExecuteCommand(asCommand.c_str());
                        LogInfo("called command [" + asCommand + "].");
                    }
                }
            }
            return RE::BSEventNotifyControl::kContinue;
        }
    };
    AutoAddedItemTransferMenuOpenCloseEventClass AutoAddedItemTransferMenuOpenCloseEventSink;

}









/////////////////////////////////////////////////////////// SFSE ///////////////////////////////////////////////////////////


inline static void HandleOnPostDataLoad(SFSE::MessagingInterface::Message* Message) noexcept
{
    if (Message->type == SFSE::MessagingInterface::kPostDataLoad) {

        LogInfo("kPostDataLoad received.");

        baseAddr = (unsigned long long)GetModuleHandle("Starfield.exe");

        // optional INI: whether to use 'fMaxShipTransferDistance' (def: 250) or not (not by default, can be enabled in INI)
        if (DoesIniExist("AutoAddedItemTransfer") == true) {
            std::string sUseMaxShipTransferDistance = ReadIni("AutoAddedItemTransfer", "General", "bUseMaxShipTransferDistance");
            if (sUseMaxShipTransferDistance.empty() == false && std::strcmp(sUseMaxShipTransferDistance.c_str(), "1") == 0) {
                LogInfo("found 'AutoAddedItemTransfer.ini' and INI key 'bUseMaxShipTransferDistance' = 1. Homeship distance will be performed.");
                bUseMaxShipTransferDistance = true;
            } else
                LogInfo("found 'AutoAddedItemTransfer.ini' but INI key 'bUseMaxShipTransferDistance' either does not exist or its value is not 1. Homeship distance will be skipped.");
            std::string sLogging = ReadIni("AutoAddedItemTransfer", "General", "bLogging");
            if (sLogging.empty() == false && std::strcmp(sLogging.c_str(), "1") == 0) {
                LogInfo("found 'AutoAddedItemTransfer.ini' and INI key 'bLogging' = 1. Logging ON.");
                bLogging = true;
            }
            else
                LogInfo("found 'AutoAddedItemTransfer.ini' but INI key 'bLogging' either does not exist or its value is not 1. Logging OFF.");  // this line won't be logged though
        }

        GetEventSource<TESContainerChangedEvent>()->RegisterSink(&AutoAddedItemTransferTESContainerChangedEventSink);
        RE::UI::GetSingleton()->RegisterSink(&Thread::AutoAddedItemTransferMenuOpenCloseEventSink);
        
    }
}

extern "C" DLLEXPORT constinit auto SFSEPlugin_Version = []() noexcept {
    SFSE::PluginVersionData data{};
    // data.PluginVersion(PluginVersionInfo::MAJOR);
    constexpr REL::Version SFSELUGINVERSION(1, 0, 0, 0);
    data.PluginVersion(SFSELUGINVERSION);
    data.PluginName(PluginVersionInfo::PROJECT);
    data.AuthorName(PluginVersionInfo::AUTHORNAME);
    data.UsesAddressLibrary(false);
    data.IsLayoutDependent(true);
    data.CompatibleVersions({ SFSE::RUNTIME_SF_1_9_67, 0 });
    return data;
}();

extern "C" DLLEXPORT bool SFSEAPI SFSEPlugin_Load(const SFSE::LoadInterface* a_sfse)
{
    SFSE::Init(a_sfse);
    SFSE::GetMessagingInterface()->RegisterListener(HandleOnPostDataLoad);
    return true;
}
