
#include <stdio.h>
#include <stdint.h>
#include <string>
#include <vector>
#include <map>
#include <iostream>
#include <fstream>
#include <sstream>
#include <stdarg.h>

#include "pugixml.hpp"

const char SwapFileName[] = "BoneSwaps.xml";

static std::map<int, int> BoneSwapsList;

static void ExitWithMessage(const std::string& message)
{
	printf("%s\nPress Enter to exit...", message.c_str());
	int unused = 0;
	unused = getchar();
}

static std::string Format(const char* fmt, ...)
{
	char buf[2048] = {};
	va_list va_alist;
	va_start(va_alist, fmt);
	vsnprintf(buf, _countof(buf), fmt, va_alist);
	va_end(va_alist);
	return std::string(buf);
}

static void GenerateDefaultSwapFile()
{
	pugi::xml_document doc;

	pugi::xml_node swapsList = doc.append_child("BoneSwapList");

	static const std::map<int, int> defaultSwaps = {
		{822, 51586},
	};

	for (const auto& defSwap : defaultSwaps)
	{
		pugi::xml_node swap = swapsList.append_child("Swap");

		swap.append_attribute("OldID") = defSwap.first;
		swap.append_attribute("NewID") = defSwap.second;
	}

	doc.save_file(SwapFileName);
}

static bool PopulateReplacementList()
{
	pugi::xml_document doc;
	pugi::xml_parse_result result = doc.load_file(SwapFileName);

	if (!result)
	{
		std::cout << "Could not parse " << SwapFileName << " - " << result.description() << " - Offset " << result.offset << std::endl
			<< "Generating default file" << std::endl;

		GenerateDefaultSwapFile();

		result = doc.load_file(SwapFileName);
	}

	if (!result)
	{
		std::cout << "Failed to generate default file - " << result.description() << " - Offset " << result.offset << std::endl << "Aborting process." << std::endl;
		return false;
	}

	for (const auto& swapChild : doc.first_child())
	{
		BoneSwapsList.insert({swapChild.attribute("OldID").as_int(), swapChild.attribute("NewID").as_int()});
	}

	return true;
}

static std::string ProcessBoneString(const std::string& boneString)
{
	std::stringstream ss(boneString);

	std::vector<std::string> rawStrs;

	std::string token;

	while (std::getline(ss, token, ' '))
	{
		rawStrs.emplace_back(token);
	}

	std::string returnVal;

	for (const auto& str : rawStrs)
	{
		int number = 0;
		std::stringstream s(str);
		s >> number;

		if (number > 0)
		{
			for (const auto& it : BoneSwapsList)
			{
				if (it.first == number)
				{
					number = it.second;
					break;
				}
			}

			returnVal += std::to_string(number) + " ";
		}
		else if (str.size() > 1)
		{
			returnVal += str + " ";
		}
	}

	return returnVal;
}

static void FindBoneStrings(const std::string& fileName)
{
	pugi::xml_document doc;
	pugi::xml_parse_result result = doc.load_file(fileName.c_str());

	if (!result)
	{
		std::cout << "Could not parse " << fileName << " - " << result.description() << " - Offset " << result.offset << std::endl;
		return;
	}

	for (const auto& drawable : doc.child("RDR2DrawableDictionary").child("Drawables"))
	{
		pugi::xml_node drawableLODs[3] = {
			drawable.child("LodHigh"),
			drawable.child("LodMed"),
			drawable.child("LodLow")
		};

		for (int i = 0; i < 3; i++)
		{
			if (!drawableLODs[i].empty())
			{
				for (const auto& model : drawableLODs[i].child("Models"))
				{
					std::string newBoneString = ProcessBoneString(model.child("BoneMapping").child_value());

					model.child("BoneMapping").first_child().set_value(newBoneString.c_str());
				}
			}
		}
	}

	doc.save_file(fileName.c_str(), " ");
}

static void ProcessFile(const std::string& fileName)
{
	std::string fileExt = (fileName.size() > 5) ? fileName.substr(fileName.rfind('.'), 4) : "";
	
	for (auto& c : fileExt)
	{
		c = tolower(c);
	}

	FILE* fileCheck = fopen(fileName.c_str(), "r");

	if (fileCheck == nullptr)
	{
		printf("ERROR: Could not open file %s\n", fileName.c_str());
		return;
	}
	
	if (fileExt == ".xml")
	{
		printf("Processing file %s\n", fileName.c_str());

		fclose(fileCheck);
		
		FindBoneStrings(fileName);
	}
	else
	{
		printf("Ignoring file %s\n", fileName.c_str());
	}

	fclose(fileCheck);
}

int main(int argc, char* argv[])
{
	if (argc <= 1)
	{
		ExitWithMessage("No input file found. Click and drag a file onto the exe to process it.");
		return 0;
	}

	if (!PopulateReplacementList())
	{
		ExitWithMessage("Could not open replacement file.");
		return 1;
	}

	for (int i = 1; i < argc; ++i)
	{
		ProcessFile(argv[i]);
	}

	ExitWithMessage("All files finished");
	return 0;
}
