// BETTERAPI_IMPLEMENTATION can only be defined in one file
// and must be defined before including betterapi.h in that one file
#define BETTERAPI_IMPLEMENTATION


// For people that want to port Cheat Engine or ASI mods to sfse without including
// sfse code into the project, define BETTERAPI_ENABLE_SFSE_MINIMAL before 
// including betterapi.h, this should get you 90% of the way to a
// plug and play sfse plugin with no brain required:
#define BETTERAPI_ENABLE_SFSE_MINIMAL


// you can include betterapi.h anywhere you want to use the api
#include "betterapi.h"


/* Other defines and configuration */
#define MOD_NAME	"Example Plugin"
#define AUTHOR_NAME	"An Earth Human"
#define GAME_VERSION    BC_MAKE_VERSION(1, 12, 36)


// Global state
static const BetterAPI* API = NULL;
static const struct simple_draw_t* UI = NULL;
static LogBufferHandle LogHandle = 0;
static uint32_t GamePauseCounter = 0;
static uint32_t ButtonClickCount = 0;


// Forward Declarations of functions implemented below
static void MyDrawCallback(void* imgui_context); /* draw the UI for the plugin */
static void MySaveLoadCallback(ConfigAction action); /* when the betterconsole save file is loaded or saved */
static void MyHotkeyCallback(uintptr_t userdata); /* when the hotkey assigned to your mod is pressed */
static void ToggleGamePause(); /* example for the demo */


// If betterconsole is installed and your mod is compatible then
// this function will get called *magically* even though its static
static int OnBetterConsoleLoad(const struct better_api_t* BetterAPI) {
        // First register your mod with betterconsole by providing a mod name
        // you could save the handle for later use, but most of the time you can
        // just register any callbacks you want upfront then forget about the handle
        RegistrationHandle my_mod_handle = BetterAPI->Callback->RegisterMod("My Mod Name");

        // Registering any callback is optional, but you will probably want
        // to at least register the draw callback to show a mod menu
        BetterAPI->Callback->RegisterDrawCallback(my_mod_handle, &MyDrawCallback);

        // If you have a set of global config options, you could make betterconsole
        // take care of saving, loading, and editing these options with a gui
        BetterAPI->Callback->RegisterConfigCallback(my_mod_handle, &MySaveLoadCallback);

        LogHandle = BetterAPI->LogBuffer->Create("Example Log", NULL);

        // Hotkey Registration happens in two steps: setting the callback and setting the hotkey
        // your mod can only have one hotkey callback, but you can request multiple hotkey
        // entries and tell them apart in your callback by the "userdata" value
        BetterAPI->Callback->RegisterHotkeyCallback(my_mod_handle, &MyHotkeyCallback); 
        BetterAPI->Callback->RequestHotkey(my_mod_handle, "Toggle Game Pause", 0); 
        BetterAPI->Callback->RequestHotkey(my_mod_handle, "Reset Click Counter", 1); 
        BetterAPI->Callback->RequestHotkey(my_mod_handle, "Reset Pause Counter", 2); 

        // Maybe save the betterapi pointer to a global variable use elsewhere
        API = BetterAPI;

        // I often find it convenient to assign individual api pointers to global variables
        UI = BetterAPI->SimpleDraw;

        return 0; // return 0 for success or any positive number to indicate a failure
}


// If registered, this callback will be called every frame when BetterConsole
// is open and the tab for your mod is in focus
static void MyDrawCallback(void* imgui_context) {

        // Several widget types are supported including text...
        UI->Text("Hello There!");

        // ...and buttons, Butons return true on the frame they are clicked
        if (UI->Button("Click Me")) {
                ++ButtonClickCount;
        }

        // Test supports printf format specifiers
        UI->Text("The button has been clicked %u times.", ButtonClickCount);
        UI->Text("The game has been paused %u times.", GamePauseCounter);

        if (UI->Button("Reset Click Counter")) {
                ButtonClickCount = 0;
        }

        // Sameline keeps the next widget on the same row as the last one
        UI->SameLine();

        if (UI->Button("Toggle Game Pause")) {
                ToggleGamePause();
        }

        UI->ShowLogBuffer(LogHandle, true);
}


// If registered, this will be called when BetterConsole loads its config
// file, when the config file is being saved, and when the user is editing
// settings in the settings menu. The API->Config functions create default
// actions for all 3 events.
static void MySaveLoadCallback(ConfigAction action) {
        API->Config->ConfigU32(action, "ButtonClickCount", &ButtonClickCount);
        API->Config->ConfigU32(action, "GamePauseCounter", &GamePauseCounter);
}


// If registered, this will be called when the hotkey combo for your mods
// hotkey request was pressed. If you request multiple hotkeys, use the
// `userdata` from the request to tell them apart.
static void MyHotkeyCallback(uintptr_t userdata) {
        // you will probably want this to be an enum
        // or a pointer to an internal data structure
        if (userdata == 0) {
                ToggleGamePause();
        }
        else if (userdata == 1) {
                ButtonClickCount = 0;
        }
        else if (userdata == 2) {
                GamePauseCounter = 0;
        }
}


// Some other things you can do with betterconsole
static void ToggleGamePause() {
        const char * const command = "ToggleGamePause";
        char buffer[128];

        // if you havent noticed, this file uses no c library, windows api, or any other function
        // only the betterconsole api, this is intentional
        API->Stdlib->snprintf(buffer, sizeof(buffer), "Running Command: %s, Pause Counter: %u", command, GamePauseCounter);

        // add the message we formated into the buffer to be displayed
        API->LogBuffer->Append(LogHandle, buffer);

        // run a console command
        API->GameHook->ConsoleRun(command);

        // do something with our global variable
        GamePauseCounter++;
}


// if you want your dll to be loadable with sfse you need to export two symbols:
// `SFSEPlugin_Version` and `SFSEPlugin_Load` paste this in one translation unit
// and modify to suit your needs:

// Step 1) Export this struct so sfse knows your DLL is compatible
BC_DLLEXPORT SFSEPluginVersionData SFSEPlugin_Version = {
        1,     // SFSE api version, 1 is current

        1,     // Plugin api version, 1 is current

        MOD_NAME,       // Mod/Plugin Name (limit: 255 characters)

        AUTHOR_NAME,    // Mod Author(s)   (limit: 255 characters)

               // Address Independance:
        1,     // 0 - hardcoded offsets (game version specific)
               // 1 - signature scanning (not version specific)

               // Structure Independance:
        1,     // 0 - relies on specific game structs
               // 1 - mod does not care if game structs change

                                   // Compatible Game Versions:
        {                          // A list of up to 15 game versions
                GAME_VERSION,      // At least one game version is required
                0                  // The list must be terminated with 0
        },                         // if address & structure independent
                                   // then this is minimum version required

        0,      // 0 = does not rely on any specific sfse version

        0, 0    // reserved fields, must be 0
};


// Step 2) Export this function so sfse knows to load your dll.
//         Doing anything inside the function is optional.
BC_DLLEXPORT bool SFSEPlugin_Load(const SFSEInterface * sfse) {
        return true;
}