;========================================================================================
; VersionManagerQuestScript
;   - Manages versioning for all registered quests.
;   - Quests register at startup and get restarted if outdated.
;   - Uses delayed confirmation to ensure version updates.
;========================================================================================

Scriptname VersionManagerQuestScript extends Quest

;========================================================================================
; STRUCT: QuestVersionData
;   - Holds quest-specific versioning information.
;========================================================================================
Struct QuestVersionData
    String QuestName
    String LastVersion      ; Version before restart
    String CurrentVersion   ; Version after restart (for verification)
    Quest QuestInstance
    bool PendingRestart     ; True if waiting for restart confirmation
EndStruct

;========================================================================================
; VARIABLES
;========================================================================================
QuestVersionData[] RegisteredQuests  ; Holds all registered quests and their versions

Event OnQuestInit()
    RegisteredQuests = new QuestVersionData[0]
EndEvent

;========================================================================================
; FUNCTION: RequestVersionManagement
;   - Called by quests at startup (OnQuestInit or OnPlayerLoadGame).
;   - Registers a quest if not already present.
;   - If version mismatch is found, schedules a restart.
;========================================================================================
Function RequestVersionManagement(Quest questInstance, String version)
    if !questInstance
        Debug.Trace("[VersionManager] ERROR: RegisterQuest() received None quest!", 2)
        return
    endif

    String questName = questInstance as String
    int questIndex = FindQuestIndex(questName)

    if questIndex == -1
        ; First-time registration
        QuestVersionData newQuest = new QuestVersionData
        newQuest.QuestName = questName
        newQuest.LastVersion = version
        newQuest.CurrentVersion = version
        newQuest.QuestInstance = questInstance
        newQuest.PendingRestart = false
        RegisteredQuests.Add(newQuest)

        Debug.Trace("[VersionManager] RegisterQuest(): Registered new quest: " + questName + " (Version: " + version + ")")
    else
        ; Existing quest: Check version mismatch
        if RegisteredQuests[questIndex].CurrentVersion != version
            Debug.Trace("[VersionManager] RegisterQuest(): Version mismatch for " + questName + " | Stored: " + RegisteredQuests[questIndex].CurrentVersion + " | Current: " + version)

            ; Store previous version before restart
            RegisteredQuests[questIndex].LastVersion = RegisteredQuests[questIndex].CurrentVersion
            RegisteredQuests[questIndex].CurrentVersion = version

            ; Schedule restart
            RegisteredQuests[questIndex].PendingRestart = true
            RestartQuest(questIndex)
        else
            Debug.Trace("[VersionManager] RegisterQuest(): Quest " + questName + " is up-to-date (Version: " + version + ")")
        endif
    endif
EndFunction

;========================================================================================
; FUNCTION: RestartQuest
;   - Restarts the specified quest.
;   - Schedules a delayed check to confirm the restart was successful.
;========================================================================================
Function RestartQuest(int questIndex)
    if questIndex < 0 || questIndex >= RegisteredQuests.Length
        Debug.Trace("[VersionManager] ERROR: RestartQuest() received invalid quest index!", 2)
        return
    endif

    Quest questInstance = RegisteredQuests[questIndex].QuestInstance
    if !questInstance
        Debug.Trace("[VersionManager] ERROR: RestartQuest() found None QuestInstance!", 2)
        return
    endif

    String questName = RegisteredQuests[questIndex].QuestName
    Debug.Trace("[VersionManager] RestartQuest(): Restarting quest: " + questName)

    questInstance.Stop()
    Utility.Wait(1.0)  ; Short delay for clean restart
    questInstance.Start()

    ; Schedule a delayed confirmation check using the quest index
    StartTimer(2.0, questIndex)
EndFunction

;========================================================================================
; EVENT: OnTimer
;   - Checks if a restarted quest has correctly updated its version.
;========================================================================================
Event OnTimer(int timerID)
    int questIndex = timerID  ; TimerID maps directly to the quest index

    if questIndex < 0 || questIndex >= RegisteredQuests.Length
        Debug.Trace("[VersionManager] ERROR: OnTimer() received invalid TimerID!", 2)
        return
    endif

    if RegisteredQuests[questIndex].PendingRestart
        Quest questInstance = RegisteredQuests[questIndex].QuestInstance
        String lastVersion = RegisteredQuests[questIndex].LastVersion
        String currentVersion = RegisteredQuests[questIndex].CurrentVersion

        if questInstance.IsRunning()
            if lastVersion != currentVersion
                Debug.Trace("[VersionManager] OnTimer(): Quest " + RegisteredQuests[questIndex].QuestName + " updated successfully from " + lastVersion + " to " + currentVersion)
                RegisteredQuests[questIndex].PendingRestart = false
            else
                Debug.Trace("[VersionManager] ERROR: OnTimer() - Quest " + RegisteredQuests[questIndex].QuestName + " restart failed! Expected: " + currentVersion + " but still at " + lastVersion, 2)
            endif
        else
            Debug.Trace("[VersionManager] ERROR: OnTimer() - Quest " + RegisteredQuests[questIndex].QuestName + " is NOT running after restart attempt!", 2)
        endif
    endif
EndEvent

;========================================================================================
; FUNCTION: FindQuestIndex
;   - Searches for a quest in the RegisteredQuests array.
;   - Returns its index or -1 if not found.
;========================================================================================
int Function FindQuestIndex(String questName)
    int index = 0
    while index < RegisteredQuests.Length
        if RegisteredQuests[index].QuestName == questName
            return index
        endif
        index += 1
    endwhile
    return -1
EndFunction
