Scriptname GSN:Objects:ShipPanel extends ObjectReference

GSN:Main Property GSN_Main Const Auto Mandatory
GSN:StorageManager Property GSN_StorageManager Const Auto Mandatory

Group ContainerAliases
    ReferenceAlias Property ActiveContainer Const Auto Mandatory
    ReferenceAlias Property MasterContainer Const Auto Mandatory
EndGroup

Keyword Property SpaceshipLinkedInterior Auto Const Mandatory
ActorValue Property Carryweight Const Auto Mandatory
Keyword Property Contraband Const Auto Mandatory

Actor Player
SpaceshipReference Ship
bool Connected = false

; BASE OBJECT OVERRIDES
; ---------------------
Event OnLoad()
    RegisterForCustomEvent(GSN_Main, "EnableMod")
    RegisterForCustomEvent(GSN_Main, "DisableMod")
    
    If IsEnabled() ; Panel is not deleted
        AddInventoryEventFilter(Contraband)
        SetValue(Carryweight, GSN_StorageManager.GetCarryWeight())
    EndIf

    If GSN_Main.IsModRunning()
        BlockActivation(false)
    Else
        BlockActivation(true, true)
    EndIf
EndEvent

Event OnUnload()
    UnregisterForCustomEvent(GSN_Main, "EnableMod")
    UnregisterForCustomEvent(GSN_Main, "DisableMod")
    RemoveInventoryEventFilter(Contraband)
    RemoveItems()
EndEvent

Event OnLockStateChanged()
    If IsLocked()
        RemoveItems()
    Else
        GetItems()
    EndIf
EndEvent

Function GetItems()
    If IsEnabled()
        ActiveContainer.GetRef().RemoveAllItems(Self)
    EndIf
EndFunction

Function RemoveItems()
    RemoveAllItems(ActiveContainer.GetRef())
EndFunction

Event GSN:Main.EnableMod(GSN:Main akSender, var[] akArgs)
    BlockActivation(false)
    Utility.wait(3)
    If FindContainer()
        ConnectContainer(true)
    EndIf
EndEvent

Event GSN:Main.DisableMod(GSN:Main akSender, var[] akArgs)
    ConnectContainer(false)
    BlockActivation(true, true)
EndEvent

; ---------------------

Event OnWorkshopObjectPlaced(ObjectReference akReference)
    Lock(true, true)
    StopAllSequences()
    StartSequence("Off", true)
    Player = Game.GetPlayer()
    Ship = GetCurrentShipRef()
    RegisterEvents(true)

    If FindContainer()
        ConnectContainer(true)
    EndIf
EndEvent

Event OnWorkshopObjectRemoved(ObjectReference akReference)
    SetConnectionStatus(false)
    RegisterEvents(false)
    Player = None
    Ship = None
EndEvent

Function RegisterEvents(Bool bRegister)
    If bRegister
        RegisterForRemoteEvent(Player, "OnPlayerLoadGame")
        RegisterForRemoteEvent(Player, "OnEnterShipInterior")
        RegisterForRemoteEvent(Ship, "OnShipDock")
        RegisterForRemoteEvent(Ship, "OnShipUndock")
        RegisterForRemoteEvent(Ship, "OnLocationChange")
    Else
        UnregisterForRemoteEvent(Player, "OnPlayerLoadGame")
        UnregisterForRemoteEvent(Player, "OnEnterShipInterior")
        UnregisterForRemoteEvent(Ship, "OnShipDock")
        UnregisterForRemoteEvent(Ship, "OnShipUndock")
        UnregisterForRemoteEvent(Ship, "OnLocationChange")
    EndIf
EndFunction

; SHIP EVENTS
; -----------
Event Actor.OnEnterShipInterior(Actor akActor, ObjectReference akShip)
    If (akShip as SpaceshipReference) == Ship
        If FindContainer()
            ConnectContainer(true)
        Else
            ConnectContainer(false)
        EndIf
    EndIf
EndEvent

Event Actor.OnPlayerLoadGame(Actor akSender)
    If Is3DLoaded()
        StartEffectSequence(Connected)
    EndIf
EndEvent

Event SpaceshipReference.OnShipDock(SpaceshipReference akSender, bool abComplete, SpaceshipReference akDocking, SpaceshipReference akParent)
    If abComplete && FindContainerOnDockedShip(akParent)
        ConnectContainer(true)
    EndIf
EndEvent

Event SpaceshipReference.OnShipUndock(SpaceshipReference akSender, bool abComplete, SpaceshipReference akUndocking, SpaceshipReference akParent)
    If Connected
        ConnectContainer(false)
    EndIf
EndEvent

Event SpaceshipReference.OnLocationChange(SpaceshipReference akSender, Location akOldLoc, Location akNewLoc)
    If Connected
        ConnectContainer(False)
    EndIf
EndEvent

; FIND CONTAINER
; ----
ObjectReference Function FindContainer()
    ObjectReference TheContainer
    If Ship.isDocked()
        TheContainer = FindContainerOnDockedShip(Ship.GetFirstDockedShip())
    ElseIf !Ship.IsInSpace()    
        TheContainer = FindContainerInLoadedArea()
    EndIf
    return TheContainer
EndFunction

; Look for container in ship exterior's loaded area
ObjectReference Function FindContainerInLoadedArea()
    ObjectReference ActiveC = ActiveContainer.GetRef()
    If ActiveC.Is3DLoaded() && !ActiveC.IsLocked()
        Return ActiveC
    EndIf
    Return None
EndFunction

; Look for container in docked ship's interior cell
ObjectReference Function FindContainerOnDockedShip(SpaceshipReference DockedShip)
    Cell DockedInterior = DockedShip.GetLinkedCell(SpaceshipLinkedInterior)

    ; Dock to The Key
    ; Linked cell is not the actual cell where ship services and the container are located 
    If Utility.IntToHex(DockedInterior.GetFormID()) == "00234E05"
        DockedInterior = Game.GetForm(2730214) as Cell
    EndIf

    ObjectReference TheContainer
    TheContainer = GSN_StorageManager.GetContainerFromCell(DockedInterior)
    If TheContainer && !TheContainer.IsLocked()
        Return TheContainer
    EndIf
EndFunction

; CONNECT CONTAINER
; ----
Function ConnectContainer(Bool bConnect)
    If bConnect
        Self.Lock(false, true)
        SetConnectionStatus(true)
    Else
        Self.Lock(true, true)  
        SetConnectionStatus(false)
    EndIf
EndFunction

Function SetConnectionStatus(bool bConnect)
    If bConnect != Connected
        Connected = bConnect
        If bConnect
            Debug.Notification("Storage Network is online.")
            StopAllSequences()
            StartSequence("Idle", true)
        Else
            Debug.Notification("Storage Network disconnected.")
            StopAllSequences()
            StartSequence("Off", true)
        EndIf
    EndIf
    StartEffectSequence(Connected)
EndFunction

Function StartEffectSequence(bool bIdle)
    If bIdle
        StopAllSequences()
        StartSequence("Idle", true)
    Else
        StopAllSequences()
        StartSequence("Off", true)
    EndIf
EndFunction

; ACTIVATION PERK
; ----
Function ShowInfo()
    BlockActivation(true, true)
    Int Capacity = GSN_StorageManager.GetCarryWeight()
    Int Used = Math.Round(ActiveContainer.GetRef().GetWeightInContainer())
    String Text = ""

    Text += "Mass: "+ Used + "/" + Capacity
    Text += "\n---------------"
    Text += "\nStatus: Offline"

    Debug.Notification(Text)
    Utility.Wait(2)
    BlockActivation(false)
EndFunction

Event OnItemAdded(Form akBaseItem, int aiItemCount, ObjectReference akItemReference, ObjectReference akSourceContainer, int aiTransferReason)
    If akBaseItem.HasKeyword(Contraband)
        RemoveItem(akBaseItem, aiItemCount, true, akSourceContainer)
        Debug.Notification("Contraband is not permitted.")
    EndIf
EndEvent
