Scriptname GSN:StorageManager extends Quest

Group Quests
    GSN:Main Property GSN_Main Auto Const
EndGroup

Group PlayerData
    Int[] Property PlayerContainers Auto Hidden
    Key[] Property PlayerKeys Auto Hidden
EndGroup

Group ContainerAliases
    ReferenceAlias Property ActiveContainer Auto Const
    ReferenceAlias Property MasterContainer Auto Const
    FormList Property GSN_ContainerTypes Auto Const
EndGroup

Group Perks
    ConditionForm Property GSN_HasPerkPayloads_Rank1 Mandatory Auto
    ConditionForm Property GSN_HasPerkPayloads_Rank2 Mandatory Auto
    ConditionForm Property GSN_HasPerkPayloads_Rank3 Mandatory Auto
    ConditionForm Property GSN_HasPerkPayloads_Rank4 Mandatory Auto
EndGroup

Group Other
    GlobalVariable Property GSN_Global_StorageBaseUnit Auto Const Mandatory
    GlobalVariable Property GSN_Global_ContainersUnlocked Auto Const Mandatory
EndGroup

bool ModRunning = false
Int[] ContainerCarryWeights
ObjectReference[] DiscoveredContainers
Cell[] DiscoveredCells
Bool FirstStart = false

CustomEvent StorageAdded

Function OnPlayerLoadGame()
    ContainerCarryWeights = New Int[GSN_ContainerTypes.GetSize()]
    ContainerCarryWeights[0] = 100  ; GSN_Container_Small
    ContainerCarryWeights[1] = 300  ; GSN_Container_Medium
    ContainerCarryWeights[2] = 450  ; GSN_Container_Large
    GSN_Global_StorageBaseUnit.SetValueInt(PayloadsPerkMult(ContainerCarryWeights[1]))
EndFunction

Function InitMod()
    PlayerContainers = New Int[GSN_ContainerTypes.GetSize()]
    PlayerKeys = New Key[0]
    DiscoveredContainers = New ObjectReference[0]
    DiscoveredCells = New Cell[0]
EndFunction

Function AddContainer(ObjectReference TheContainer)
    ; Called when container is unlocked
    Key TheKey = TheContainer.GetKey()
    If !CheckKey(TheKey) ; Only do this the first time key was used
        Int OldCarryWeight = GetCarryWeight()
        IncrementPlayerContainers(TheContainer)
        StoreKey(TheContainer.GetKey())
        Int NewCarryWeight = GetCarryWeight()
        NotifyStorageAdded(OldCarryWeight, NewCarryWeight)
        GSN_Global_ContainersUnlocked.SetValue(GSN_Global_ContainersUnlocked.GetValue() + 1.0)
    EndIf
EndFunction

Function AddStorage()
    ; Called from VendorManager
    ; Add a standard (medium) unit of storage to the pool
    Int OldCarryWeight = GetCarryWeight()
    PlayerContainers[1] += 1
    Int NewCarryWeight = GetCarryWeight()
    Var[] kargs = new Var[1]
    kargs[0] = NewCarryWeight
    SendCustomEvent("StorageAdded", kargs)
    NotifyStorageAdded(OldCarryWeight, NewCarryWeight)    
EndFunction

Function NotifyStorageAdded(Int PrevSize, Int NewSize)
    Int StorageAdded = NewSize - PrevSize
    Debug.Notification("Storage added (" + StorageAdded + ")")
EndFunction

Function IncrementPlayerContainers(ObjectReference TheContainer)
    Int i = 0
    While i < GSN_ContainerTypes.GetSize()
        If TheContainer.HasKeyword(GSN_ContainerTypes.GetAt(i) as Keyword)
            PlayerContainers[i] += 1
        EndIf
        i += 1
    EndWhile
EndFunction

Function StoreKey(key TheKey)
    PlayerKeys = AddArrayElement(PlayerKeys as Form[], TheKey as Form) as Key[]
EndFunction

Bool Function CheckKey(key TheKey)
    If PlayerKeys.find(TheKey) >= 0
        return true ; Player has used this key before
    EndIf
    Return false
EndFunction

Int Function GetCarryWeight()
    Int StoragePool = 0
    ; Map PlayerContainers to CarryWeights
    Int i = 0
    While i < GSN_ContainerTypes.GetSize()
        StoragePool += (PlayerContainers[i] * ContainerCarryWeights[i])
        i += 1
    EndWhile
    StoragePool = PayloadsPerkMult(StoragePool)
    Return StoragePool
EndFunction

Int Function PayloadsPerkMult(Int valueToMult)
    float PerkMult = 1
    If GSN_HasPerkPayloads_Rank1.IsTrue()
        PerkMult = 1.1
    ElseIf GSN_HasPerkPayloads_Rank2.IsTrue()
        PerkMult = 1.2
    ElseIf GSN_HasPerkPayloads_Rank3.IsTrue()
        PerkMult = 1.3
    ElseIf GSN_HasPerkPayloads_Rank4.IsTrue()
        PerkMult = 1.5
    EndIf

    Int ReturnValue = (Math.Round(valueToMult * PerkMult))
    return ReturnValue
EndFunction

Function TrackContainer(ObjectReference TheContainer)
    Cell TheCell = TheContainer.GetParentCell()
    If DiscoveredCells.find(TheCell) < 0
        DiscoveredCells = AddArrayElement(DiscoveredCells as Form[], TheCell as Form) as Cell[]
        DiscoveredContainers = AddArrayElement(DiscoveredContainers as Form[], TheContainer as Form) as ObjectReference[]
    EndIf
EndFunction

Form[] Function AddArrayElement(Form[] TheArray, Form ElementToAdd)
    int NewSize = TheArray.Length + 1
    Form[] NewArray = New Form[NewSize]
    Int i = 0
    While i < TheArray.length
        NewArray[i] = TheArray[i]
        i += 1
    EndWhile
    NewArray[NewArray.length - 1] = ElementToAdd
    Return NewArray
EndFunction

ObjectReference Function GetContainerFromCell(Cell TheCell)
    Int lookup = DiscoveredCells.find(TheCell)
    If lookup >= 0
        return DiscoveredContainers[lookup]
    EndIf
EndFunction