Scriptname GSN:StorageManager extends Quest

Group Scripts
    GSN:Main Property GSN_Main Auto Const
EndGroup

Group PlayerData
    Int[] Property PlayerContainers Auto Hidden
    Key[] Property PlayerKeys Auto Hidden
EndGroup

Group ContainerAliases
    ReferenceAlias Property ActiveContainer Auto Const
    ReferenceAlias Property MasterContainer Auto Const
    FormList Property GSN_ContainerTypes Auto Const
EndGroup

Group Perks
    ConditionForm Property GSN_HasPerkPayloads_Rank1 Mandatory Auto
    ConditionForm Property GSN_HasPerkPayloads_Rank2 Mandatory Auto
    ConditionForm Property GSN_HasPerkPayloads_Rank3 Mandatory Auto
    ConditionForm Property GSN_HasPerkPayloads_Rank4 Mandatory Auto
EndGroup

Group Globals
    GlobalVariable Property GSN_Global_StorageCapacity Auto Const Mandatory
    GlobalVariable Property GSN_Global_ContainersUnlocked Auto Const Mandatory
EndGroup

Group Messages
    String Property StorageAddedMessageString Auto Const Mandatory
EndGroup

bool ModRunning = false
Int[] ContainerCarryWeights
ObjectReference[] DiscoveredContainers
Cell[] DiscoveredCells
Bool FirstStart = false

CustomEvent StorageAdded

Function OnPlayerLoadGame()
    ContainerCarryWeights = New Int[GSN_ContainerTypes.GetSize()]
    ContainerCarryWeights[0] = 80  ; GSN_Container_Small
    ContainerCarryWeights[1] = 250  ; GSN_Container_Medium
    ContainerCarryWeights[2] = 400  ; GSN_Container_Large
EndFunction

Function InitMod()
    PlayerContainers = New Int[GSN_ContainerTypes.GetSize()]
    PlayerKeys = New Key[0]
    DiscoveredContainers = New ObjectReference[0]
    DiscoveredCells = New Cell[0]
EndFunction

Function AddContainer(ObjectReference TheContainer)
    ; Called when container is unlocked
    Key TheKey = TheContainer.GetKey()
    If !CheckKey(TheKey) ; Only do this the first time key was used
        Int OldCarryWeight = GetCarryWeight()
        IncrementPlayerContainers(TheContainer)
        StoreKey(TheContainer.GetKey())
        Int NewCarryWeight = GetCarryWeight()
        GSN_Global_ContainersUnlocked.SetValue(GSN_Global_ContainersUnlocked.GetValue() + 1.0)
        NotifyStorageAdded()
    EndIf
EndFunction

Function AddStorage()
    ; Called from VendorManager
    ; Add a standard (medium) unit of storage to the pool
    Int OldCarryWeight = GetCarryWeight()
    PlayerContainers[1] += 1
    Int NewCarryWeight = GetCarryWeight()
    Var[] kargs = new Var[1]
    kargs[0] = NewCarryWeight
    SendCustomEvent("StorageAdded", kargs)
    NotifyStorageAdded()
EndFunction

; Calculate and set the StorageCapacity Global for the currently loaded container
; This si for display purposed only - Used in Ship Tech dialogue and "Storage added" notifications
Function SetStorageCapacity(ObjectReference TheContainer)
    Int Size
    Int i = 0
    While i < GSN_ContainerTypes.GetSize()
        If TheContainer.HasKeyword(GSN_ContainerTypes.GetAt(i) as Keyword)
            Size = ContainerCarryWeights[i]
        EndIf
        i += 1
    EndWhile

    Size = PayloadsPerkMult(Size)
    GSN_Global_StorageCapacity.SetValueInt(Size)
EndFunction

Function NotifyStorageAdded()
    ; Using Debug.Notification because AFAICT there isn't a way to do text replacement in Messages
    Debug.Notification(StorageAddedMessageString + " (" + GSN_Global_StorageCapacity.GetValueInt() +")")
EndFunction

Function IncrementPlayerContainers(ObjectReference TheContainer)
    Int i = 0
    While i < GSN_ContainerTypes.GetSize()
        If TheContainer.HasKeyword(GSN_ContainerTypes.GetAt(i) as Keyword)
            PlayerContainers[i] += 1
        EndIf
        i += 1
    EndWhile
EndFunction

Function StoreKey(key TheKey)
    PlayerKeys = GSN_Utils.AddArrayElement(PlayerKeys as Form[], TheKey as Form) as Key[]
EndFunction

Bool Function CheckKey(key TheKey)
    If PlayerKeys.find(TheKey) >= 0
        return true ; Player has used this key before
    EndIf
    Return false
EndFunction

Int Function GetCarryWeight()
    Int StoragePool = 0
    ; Map PlayerContainers to CarryWeights
    Int i = 0
    While i < GSN_ContainerTypes.GetSize()
        StoragePool += (PlayerContainers[i] * ContainerCarryWeights[i])
        i += 1
    EndWhile
    StoragePool = PayloadsPerkMult(StoragePool)
    Return StoragePool
EndFunction

Int Function PayloadsPerkMult(Int valueToMult)
    float PerkMult = 1
    If GSN_HasPerkPayloads_Rank1.IsTrue()
        PerkMult = 1.1
    ElseIf GSN_HasPerkPayloads_Rank2.IsTrue()
        PerkMult = 1.2
    ElseIf GSN_HasPerkPayloads_Rank3.IsTrue()
        PerkMult = 1.3
    ElseIf GSN_HasPerkPayloads_Rank4.IsTrue()
        PerkMult = 1.5
    EndIf

    Int ReturnValue = (Math.Round(valueToMult * PerkMult))
    return ReturnValue
EndFunction

Function TrackContainer(ObjectReference TheContainer)
    Cell TheCell = TheContainer.GetParentCell()
    If DiscoveredCells.find(TheCell) < 0
        DiscoveredCells = GSN_Utils.AddArrayElement(DiscoveredCells as Form[], TheCell as Form) as Cell[]
        DiscoveredContainers = GSN_Utils.AddArrayElement(DiscoveredContainers as Form[], TheContainer as Form) as ObjectReference[]
    EndIf
EndFunction

ObjectReference Function GetContainerFromCell(Cell TheCell)
    Int lookup = DiscoveredCells.find(TheCell)
    If lookup >= 0
        return DiscoveredContainers[lookup]
    EndIf
EndFunction