Scriptname GSN:Objects:ShipPanel extends ObjectReference

Group Scripts 
    GSN:Main Property GSN_Main Const Auto Mandatory
    GSN:StorageManager Property GSN_StorageManager Const Auto Mandatory
EndGroup

Group ContainerAliases
    ReferenceAlias Property ActiveContainer Const Auto Mandatory
    ReferenceAlias Property MasterContainer Const Auto Mandatory
EndGroup

Group Messages
    Message Property GSN_Message_StorageOnline Const Auto Mandatory
    Message Property GSN_Message_StorageOffline Const Auto Mandatory
    Message Property GSN_Message_ContrabandNotPermitted Const Auto Mandatory
    String Property InfoMessageMass Auto Const Mandatory
    String Property InfoMessageStatusOffline Auto Const Mandatory
EndGroup

Keyword Property SpaceshipLinkedInterior Auto Const Mandatory
ActorValue Property Carryweight Const Auto Mandatory
Keyword Property Contraband Const Auto Mandatory

Actor Player
SpaceshipReference Ship
bool Connected = false
bool loaded = false

Event OnWorkshopObjectPlaced(ObjectReference akReference)
    Lock(true, true)
    StopAllSequences()
    StartSequence("Off", true)
    Player = Game.GetPlayer()
    Ship = GetCurrentShipRef()
    RegisterEvents(true)
    AddInventoryEventFilter(Contraband)

    If FindContainer()
        ConnectContainer(true)
    EndIf
EndEvent

Event OnWorkshopObjectRemoved(ObjectReference akReference)
    SetConnectionStatus(false)
    RegisterEvents(false)
    Player = None
    Ship = None
EndEvent

Function RegisterEvents(Bool bRegister)
    If bRegister
        RegisterForCustomEvent(GSN_Main, "EnableMod")
        RegisterForCustomEvent(GSN_Main, "DisableMod")
        RegisterForRemoteEvent(Player, "OnPlayerLoadGame")
        RegisterForRemoteEvent(Player, "OnEnterShipInterior")
        RegisterForRemoteEvent(Player, "OnExitShipInterior")
        RegisterForRemoteEvent(Ship, "OnShipDock")
        RegisterForRemoteEvent(Ship, "OnShipUndock")
        RegisterForRemoteEvent(Ship, "OnLocationChange")
    Else
        UnregisterForCustomEvent(GSN_Main, "EnableMod")
        UnregisterForCustomEvent(GSN_Main, "DisableMod")
        UnregisterForRemoteEvent(Player, "OnPlayerLoadGame")
        UnregisterForRemoteEvent(Player, "OnEnterShipInterior")
        UnregisterForRemoteEvent(Player, "OnExitShipInterior")
        UnregisterForRemoteEvent(Ship, "OnShipDock")
        UnregisterForRemoteEvent(Ship, "OnShipUndock")
        UnregisterForRemoteEvent(Ship, "OnLocationChange")
    EndIf
EndFunction

Event OnLoad()
    LoadPanel()
EndEvent

Event OnUnload()
    UnloadPanel()
EndEvent

Event Actor.OnEnterShipInterior(Actor akActor, ObjectReference akShip)
    If (akShip as SpaceshipReference) == Ship && GSN_Main.IsModRunning()
        LoadPanel()
    EndIf
EndEvent

Event Actor.OnExitShipInterior(Actor akActor, ObjectReference akShip)
    If (akShip as SpaceshipReference) == Ship
        UnloadPanel()
    EndIf
EndEvent

Function LoadPanel()
    If !loaded
        loaded = true
        If IsEnabled() && GSN_Main.IsModRunning() ; Panel is not deleted
            SetValue(Carryweight, GSN_StorageManager.GetCarryWeight())
            BlockActivation(false)
            If FindContainer()
                ConnectContainer(true)
            Else
                ConnectContainer(false)
            EndIf
        Else
            ConnectContainer(false)
            BlockActivation(true, true)
        EndIf
    EndIf
EndFunction

Function UnloadPanel()
    If loaded
        loaded = false
        RemoveItems()
    EndIf
EndFunction

Event OnLockStateChanged()
    If IsLocked()
        RemoveItems()
    Else
        GetItems()
    EndIf
EndEvent

Function GetItems()
    If IsEnabled()
        ActiveContainer.GetRef().RemoveAllItems(Self)
    EndIf
EndFunction

Function RemoveItems()
    ObjectReference ActiveRef = ActiveContainer.GetRef()
    If ActiveRef.Is3DLoaded()
        RemoveAllItems(ActiveContainer.GetRef())
    Else
        RemoveAllItems(MasterContainer.GetRef())
    EndIf
EndFunction



; ---------------------
Event GSN:Main.EnableMod(GSN:Main akSender, var[] akArgs)
    BlockActivation(false)
    If Is3DLoaded()
        Utility.wait(3) ; Wait for container on landing pad to retrieve items first
        If FindContainer()
            ConnectContainer(true)
        EndIf
    EndIf
EndEvent

Event GSN:Main.DisableMod(GSN:Main akSender, var[] akArgs)
    If Is3DLoaded()
        ConnectContainer(false)
        BlockActivation(true, true)
    EndIf
EndEvent

Event Actor.OnPlayerLoadGame(Actor akSender)
    If Is3DLoaded()
        StartEffectSequence(Connected)
    EndIf
EndEvent

Event SpaceshipReference.OnShipDock(SpaceshipReference akSender, bool abComplete, SpaceshipReference akDocking, SpaceshipReference akParent)
    ; Check panel is 3D loaded because on some ships (e.g Siren of the Stars), docking events occur when they shouldn't.
    If GSN_Main.IsModRunning() && abComplete && FindContainerOnDockedShip(akParent) && Is3DLoaded()  
        ; Debug.Notification("Ship docked")
        ConnectContainer(true)
    EndIf
EndEvent

Event SpaceshipReference.OnShipUndock(SpaceshipReference akSender, bool abComplete, SpaceshipReference akUndocking, SpaceshipReference akParent)
    ; Check panel is 3D loaded because on some ships (e.g Siren of the Stars), docking events occur when they shouldn't.
    If Connected && akUndocking == Ship && Is3DLoaded()
        ; Debug.Notification("Ship undocked")
        ConnectContainer(false)
    EndIf
EndEvent

Event SpaceshipReference.OnLocationChange(SpaceshipReference akSender, Location akOldLoc, Location akNewLoc)
    If Connected
        ConnectContainer(False)
    EndIf
EndEvent

; FIND CONTAINER
; ----
ObjectReference Function FindContainer()
    ObjectReference TheContainer
    If Ship.isDocked()
        TheContainer = FindContainerOnDockedShip(Ship.GetFirstDockedShip())
    ElseIf !Ship.IsInSpace()    
        TheContainer = FindContainerInLoadedArea()
    EndIf
    return TheContainer
EndFunction

; Look for container in ship exterior's loaded area
ObjectReference Function FindContainerInLoadedArea()
    ObjectReference ActiveC = ActiveContainer.GetRef()
    If ActiveC.Is3DLoaded() && !ActiveC.IsLocked()
        Return ActiveC
    EndIf
    Return None
EndFunction

; Look for container in docked ship's interior cell
ObjectReference Function FindContainerOnDockedShip(SpaceshipReference DockedShip)
    Cell DockedInterior = DockedShip.GetLinkedCell(SpaceshipLinkedInterior)
    ; Debug.Notification("Interior cell: " + DockedInterior)

    ; Dock to The Key
    ; Linked cell is not the actual cell where ship services and the container are located 
    If Utility.IntToHex(DockedInterior.GetFormID()) == "00234E05"
        DockedInterior = Game.GetForm(2730214) as Cell ; 0029A8E6
    EndIf

    ; Siren of the Stars
    ; Usually the cell with docking port is the "ShipCell" cell. Any additional cells inside are "Interior"
    ; In this case, Bethesda mixed these up. You actually dock directly to an "Interior" cell and the Ballroom cell is the "ShipCell"
    If Utility.IntToHex(DockedInterior.GetFormId()) == "001A041B"
        DockedInterior = Game.GetForm(2323412) as Cell ; 002373D4
    EndIf

    ObjectReference TheContainer
    TheContainer = GSN_StorageManager.GetContainerFromCell(DockedInterior)

    ; Debug.Notification("Container " + TheContainer)

    If TheContainer && !TheContainer.IsLocked()
        Return TheContainer
    EndIf
EndFunction

; CONNECT CONTAINER
; ----
Function ConnectContainer(Bool bConnect)
    If bConnect
        Self.Lock(false, true)
        SetConnectionStatus(true)
    Else
        Self.Lock(true, true)  
        SetConnectionStatus(false)
    EndIf
EndFunction

Function SetConnectionStatus(bool bConnect)
    If bConnect != Connected
        Connected = bConnect
        If bConnect
            GSN_Message_StorageOnline.Show()
        Else
            GSN_Message_StorageOffline.Show()
        EndIf
    EndIf
    StartEffectSequence(Connected)
EndFunction

Function StartEffectSequence(bool bConnected)
    If bConnected
        StopAllSequences()
        StartSequence("Idle", true)
    Else
        StopAllSequences()
        StartSequence("Off", true)
    EndIf
EndFunction

; ACTIVATION PERK
; ----
Function ShowInfo()
    BlockActivation(true, true)
    Int Capacity = GSN_StorageManager.GetCarryWeight()
    Int Used = Math.Round(ActiveContainer.GetRef().GetWeightInContainer())
    String Text = ""

    Text += InfoMessageMass + ": "+ Used + "/" + Capacity
    Text += "\n"
    Text += InfoMessageStatusOffline ;"Status: Offline"

    Debug.Notification(Text)
    Utility.Wait(2)
    BlockActivation(false)
EndFunction

Event OnItemAdded(Form akBaseItem, int aiItemCount, ObjectReference akItemReference, ObjectReference akSourceContainer, int aiTransferReason)
    If akBaseItem.HasKeyword(Contraband)
        RemoveItem(akBaseItem, aiItemCount, true, akSourceContainer)
        GSN_Message_ContrabandNotPermitted.Show()
    EndIf
EndEvent