Scriptname GSN:Objects:Container extends ObjectReference

GSN:Main Property GSN_Main Const Auto Mandatory
GSN:StorageManager Property GSN_StorageManager Const Auto Mandatory
GSN:Objects:IDCardReader Property LinkedCardReader Auto Hidden
Keyword Property GSN_KW_LinkedCardReader Const Auto Mandatory
ActorValue Property Carryweight Const Auto Mandatory

Group ContainerAliases
    ReferenceAlias Property ActiveContainer Const Auto Mandatory
    ReferenceAlias Property MasterContainer Const Auto Mandatory
EndGroup

Keyword Property Contraband Const Auto Mandatory
Message Property GSN_Message_ContrabandNotPermitted Const Auto Mandatory

Bool isDeactivated = false

Function HandleOnload()
EndFunction

Function HandleOnUnload()
EndFunction

Function HandleLockStateChanged()
EndFunction

Event OnLoad()
    If IsEnabled()
        InitialiseContainer()
        If GSN_Main.IsModRunning()
            If isDeactivated
                OnEnableMod()
            EndIf
            If !IsLocked() 
                GetItems()
            EndIf
            HandleOnload()
        ElseIf !isDeactivated
            DeactivateContainer()
        EndIf
    EndIf
EndEvent

Function InitialiseContainer()
    RegisterForCustomEvent(GSN_Main, "EnableMod")
    RegisterForCustomEvent(GSN_Main, "DisableMod")
    RegisterForCustomEvent(GSN_StorageManager, "StorageAdded")
    AddInventoryEventFilter(Contraband)
    SetValue(CarryWeight, GSN_StorageManager.GetCarryWeight())
    GSN_StorageManager.SetStorageCapacity(Self)
    RegisterCardReader()
EndFunction

Function ShutdownContainer()
    UnregisterForCustomEvent(GSN_Main, "EnableMod")
    UnregisterForCustomEvent(GSN_Main, "DisableMod")
    UnregisterForCustomEvent(GSN_StorageManager, "StorageAdded")
    RemoveInventoryEventFilter(Contraband)
    RemoveItems()
EndFunction

Event OnUnload()
    If IsEnabled()
        ShutdownContainer()
        HandleOnUnload()
    EndIf
EndEvent

Event OnWorkshopObjectRemoved(ObjectReference akReference)
    ShutdownContainer()
EndEvent

Event OnItemAdded(Form akBaseItem, int aiItemCount, ObjectReference akItemReference, ObjectReference akSourceContainer, int aiTransferReason)
    If akBaseItem.HasKeyword(Contraband)
        RemoveItem(akBaseItem, aiItemCount, true, akSourceContainer)
        GSN_Message_ContrabandNotPermitted.Show()
    EndIf
EndEvent

Function GetItems()
    ObjectReference MasterRef = MasterContainer.GetRef()
    MasterRef.RemoveAllItems(Self) ; Get items from Master
    ActiveContainer.ForceRefTo(Self) ; This is now ActiveContainer
EndFunction

Function RemoveItems()
    ObjectReference MasterRef = MasterContainer.GetRef()
    ObjectReference ActiveRef = ActiveContainer.GetRef()
    If Self == ActiveRef 
        RemoveAllItems(MasterRef) ; Move items back to master
        ActiveContainer.ForceRefTo(MasterRef) ; MasterContainer is now ActiveContainer
    EndIf
EndFunction

Event OnLockStateChanged()
    If IsLocked()
        RemoveItems()
    Else
        GetItems()
    EndIf
    HandleLockStateChanged()
EndEvent

Event GSN:Main.EnableMod(GSN:Main akSender, var[] akArgs)
    OnEnableMod()
EndEvent

Event GSN:Main.DisableMod(GSN:Main akSender, var[] akArgs)
   OnDisableMod()
EndEvent

Event GSN:StorageManager.StorageAdded(GSN:StorageManager akSender, var[] akArgs)
    SetValue(CarryWeight, akArgs[0] as float)
EndEvent

Function OnEnableMod()
    If GSN_StorageManager.CheckKey(GetKey()) || GetWorkshop()
        UnlockContainer()
    Else
        LinkedCardReader.PlayIdleState()
        BlockActivation() ; Blocks the default "Unlock" activation, but displays default "Requires key - New Atlantis" messages
    EndIf
    isDeactivated = false
EndFunction

Function OnDisableMod()
    DeactivateContainer()
EndFunction

; UNLOCK
; Called when player opens the container with the correct key, 
; and when re-enabling mod if this container was already unlocked by the player
Function UnlockContainer()
    BlockActivation(true, true)
    Utility.wait(0.1)
    LinkedCardReader.PlaySuccessState()
    Lock(false)
    BlockActivation(false, false)
EndFunction

; LOCK
; Called when the player manually locks the outpost container for safe deletion.
Function LockContainer()
    BlockActivation(true, true)
    Lock(true, true)
    Utility.wait(0.1)
    LinkedCardReader.PlayIdleState()
    BlockActivation(false, false)
EndFunction

; LOCKDOWN
; Called on DisableMod Event and when the container is loaded but mod is already disabled.
; Also called when player decides to "deactivate" container for safe deletion
Function DeactivateContainer()
    BlockActivation(true, true)
    LinkedCardReader.PlayLockdownState()
    Lock(true, true) ; Lock AFTER the animation - ives time for the ShipPanel (if loaded) to give us the items back before we lock and send to master
    If GSN_Main.IsModRunning()
        BlockActivation(false, false) ; Allows outpost container interaction to re-enable
    Else
        isDeactivated = true
    EndIf
EndFunction

; Link ID Card Reader
Function RegisterCardReader()
    LinkedCardReader = GetLinkedRef(GSN_KW_LinkedCardReader) as GSN:Objects:IDCardReader
    If LinkedCardReader
        LinkedCardReader.BlockActivation(true, true)
        If IsLocked()
            If isDeactivated
                BLockActivation(true, true)
            Else
                BlockActivation() ; Blocks the default "Unlock" activation, but displays default "Requires key - New Atlantis" messages
            EndIf
        Else
            LinkedCardReader.PlayUnlockedIdleState() ; Ensures reader doesn't eventually return back to locked appearance
        EndIf
    EndIf
EndFunction

Event OnActivate(ObjectReference akActionRef) ; Still fires despite BlockActivation()
    If IsLocked()
        If GetKey() && akActionRef.GetItemCount(GetKey()) > 0
            UnlockContainer()
        Else
            LinkedCardReader.PlayFailureState()
        EndIf
    EndIf
EndEvent