Scriptname GSN:ShipManager extends Quest

Group Scripts
    GSN:Main Property GSN_Main Auto Const
    GSN:StorageManager Property GSN_StorageManager Auto Const
EndGroup

Group References
    ReferenceAlias Property HomeShip Auto Const
    LocationAlias Property HomeShipInteriorLocation Auto Const
    ReferenceAlias Property HomeShipPilotSeat Auto Const
    ReferenceAlias Property ActiveContainer Auto Const
    ReferenceAlias Property MasterContainer Auto Const
EndGroup

Group Keywords
    Keyword Property GSN_KW_ShipPanel Auto Const
    Keyword Property SpaceshipLinkedInterior Auto Const Mandatory
    ActorValue Property Carryweight Const Auto Mandatory
EndGroup

Group Globals
    GlobalVariable Property GSN_Global_ShipIsConnected Auto Const Mandatory
    GlobalVariable Property GSN_Global_DEBUG Auto Const Mandatory
EndGroup

Group Messages
    Message Property GSN_Message_StorageOnline Const Auto Mandatory
    Message Property GSN_Message_StorageOffline Const Auto Mandatory
    String Property InfoMessageUsage Auto Const Mandatory
    String Property InfoMessageStatusOffline Auto Const Mandatory
    String Property InfoMessageStatusOnline Auto Const Mandatory
EndGroup

CustomEvent OnConnect
CustomEvent OnDisconnect

Actor Player
SpaceshipReference Ship
ObjectReference PilotSeat

ObjectReference[] Panels
ObjectReference MasterPanel
Bool ShipInteriorLoaded
Bool Connected

Int DebugIndex = 0
Function DebugMessage(string text)
    If GSN_Global_DEBUG.GetValue() > 0
        DebugIndex += 1
        Debug.Notification("Debug " + DebugIndex + ": \n" + text)
    Else
        DebugIndex = 0
    EndIf
EndFunction

; SETUP
; -----
Function StartMod()
    Player = Game.GetPlayer()
    Ship = HomeShip.GetShipRef()
    PilotSeat = HomeShipPilotSeat.GetRef()
    RegisterPlayerEvents(true)
    RegisterShipEvents(true)
    If PlayerIsOnShip()
        DebugMessage("Player is on ship.")
        FindPanels()
        SetupPanels()
        TryToConnect()
    EndIf
EndFunction

Function StopMod()
    Disconnect()
    RegisterPlayerEvents(false)
    RegisterShipEvents(false)
    Player = None
    Ship = None
    PilotSeat = None
    MasterPanel = None
    Panels.clear()
EndFunction

Function RegisterPlayerEvents(Bool bRegister)
    If bRegister
        RegisterForRemoteEvent(Player, "OnHomeShipSet")
    Else
        UnregisterForRemoteEvent(Player, "OnHomeShipSet")
    EndIf
EndFunction

Function RegisterShipEvents(Bool bRegister)
    If bRegister
        RegisterForRemoteEvent(PilotSeat, "OnCellLoad")
        RegisterForRemoteEvent(PilotSeat, "OnUnload")
        RegisterForRemoteEvent(Ship, "OnShipDock")
        RegisterForRemoteEvent(Ship, "OnShipUndock")
        RegisterForRemoteEvent(Ship, "OnLocationChange")
    Else
        UnregisterForRemoteEvent(PilotSeat, "OnCellLoad")
        UnregisterForRemoteEvent(PilotSeat, "OnUnload")
        UnregisterForRemoteEvent(Ship, "OnShipDock")
        UnregisterForRemoteEvent(Ship, "OnShipUndock")
        UnregisterForRemoteEvent(Ship, "OnLocationChange")
    EndIf
EndFunction

Bool Function PlayerIsOnShip()
    Return PilotSeat.Is3DLoaded()
EndFunction

; EVENTS
; ------
Event Actor.OnHomeShipSet(Actor akSender, SpaceshipReference akShip, SpaceshipReference akPrevious)
    Disconnect()
    RegisterShipEvents(false)
    MasterPanel = None
    Panels.clear()
    HomeShip.RefillDependentAliases()
    Ship = HomeShip.GetShipRef()
    PilotSeat = HomeShipPilotSeat.GetRef()
    RegisterShipEvents(true)
    DebugMessage("Homeship changed: " + Ship)

    If PlayerIsOnShip()
        DebugMessage("Player is on ship.")
        FindPanels()
        SetupPanels()
        TryToConnect()
    EndIf
EndEvent

Event ObjectReference.OnCellLoad(ObjectReference akSender)
    DebugMessage("Enter ship.")
    FindPanels()
    SetupPanels()
    TryToConnect()
EndEvent

Event ObjectReference.OnUnload(ObjectReference akSender)
    DebugMessage("Exit ship.")
    If Connected
        Utility.Wait(1.0)
        RemoveItems()
    EndIf
EndEvent

Event SpaceshipReference.OnShipDock(SpaceshipReference akSender, bool abComplete, SpaceshipReference akDocking, SpaceshipReference akParent)
    If abComplete && akDocking == Ship
        DebugMessage("Ship Docked.")
        TryToConnect()
    EndIf
EndEvent

Event SpaceshipReference.OnShipUndock(SpaceshipReference akSender, bool abComplete, SpaceshipReference akUndocking, SpaceshipReference akParent)
    If !abComplete && akUndocking == Ship
        DebugMessage("Ship Undocked.")
        Disconnect()
    EndIf
EndEvent

Event SpaceshipReference.OnLocationChange(SpaceshipReference akSender, Location akOldLoc, Location akNewLoc)
    If akOldLoc != akNewLoc
        DebugMessage("Ship location changed.")
        TryToConnect()
    EndIf
EndEvent



; CONNECTION
; ----------
Function TryToConnect()
    If Panels.Length > 0 \
    && MasterPanel \
    && PlayerIsOnShip() \
    && FindContainer()
        ActiveContainer.GetRef().RemoveAllItems(MasterPanel)
        MasterPanel.lock(false)
        SetConnectionState(true)
    ElseIf Connected
        Disconnect()
    EndIf
EndFunction

Function Disconnect()
    RemoveItems()
    MasterPanel.lock(true)
    SetConnectionState(false)
EndFunction

Function SetConnectionState(bool bConnect)
    If bConnect != Connected
        Connected = bConnect
        If Connected
            GSN_Global_ShipIsConnected.SetValueInt(1)
            SendCustomEvent("OnConnect")
            GSN_Message_StorageOnline.Show()
        Else
            GSN_Global_ShipIsConnected.SetValueInt(0)
            SendCustomEvent("OnDisconnect")
            GSN_Message_StorageOffline.Show()
        EndIf
    EndIf
EndFunction

Function RemoveItems()
    If MasterPanel
        ObjectReference ActiveRef = ActiveContainer.GetRef()
        ObjectReference MasterRef = MasterContainer.GetRef()
        If ActiveRef.Is3DLoaded()
            MasterPanel.RemoveAllItems(ActiveRef)
            DebugMessage("Items moved to Active Container: "+ ActiveRef)
        Else
            MasterPanel.RemoveAllItems(MasterRef)
            DebugMessage("Items moved to Master Container: "+ MasterRef)
        EndIf
    EndIf
EndFunction

; PANELS
; ------
Function FindPanels()
    ObjectReference NearTo = PilotSeat
    DebugMessage("Pilot seat: "+ NearTo)

    ObjectReference[] PanelList = New ObjectReference[0]
    ObjectReference[] PanelSearch = NearTo.FindAllReferencesWithKeyword(GSN_KW_ShipPanel, 100)

    Int i = 0
    While i < PanelSearch.Length
        If PanelSearch[i].IsEnabled()
            PanelList = GSN_Utils.AddArrayElement(PanelList as Form[], PanelSearch[i]) as ObjectReference[]
        EndIf
        i += 1
    EndWhile

    Panels = PanelList
EndFunction

Function RegisterPanel(ObjectReference ThePanel, Bool bRegister)
    If bRegister
        Panels = GSN_Utils.AddArrayElement(Panels as Form[], ThePanel) as ObjectReference[]
    Else
        Panels = GSN_Utils.RemoveArrayElement(Panels as Form[], ThePanel) as ObjectReference[]
    EndIf
    SetupPanels()
    TryToConnect()
EndFunction

Function SetupPanels()
    If Panels.length == 0
        MasterPanel = None
        SetConnectionState(false)
    Else
        If Panels.Find(MasterPanel) < 0
            MasterPanel = Panels[0]
        EndIf

        Int TheCarryWeight = GSN_StorageManager.GetCarryWeight()
        Int i = 0
        While i < Panels.Length
            Panels[i].SetValue(CarryWeight, TheCarryWeight)
            If Panels[i] != MasterPanel
                Panels[i].SetLockLevel(6)
                Panels[i].Lock(true)
            EndIf
            i += 1
        EndWhile
    EndIf
    DebugMessage("Panels loaded: "+ Panels.Length + "\nMaster Panel:\n" + (MasterPanel as GSN:Objects:ShipPanel))
EndFunction


; CONTAINER
; ---------
Bool Function FindContainer()
    If (Ship.IsDocked() && FindContainerOnDockedShip(Ship.GetFirstDockedShip())) \
    || !Ship.IsInSpace() && FindContainerInLoadedArea()
        Return true
    EndIf
    Return false
EndFunction

; Look for container in ship exterior's loaded area
Bool Function FindContainerInLoadedArea()
    ObjectReference ActiveC = ActiveContainer.GetRef()
    If ActiveC.Is3DLoaded() && !ActiveC.IsLocked()
        Return true
    EndIf
    Return false
EndFunction

; Look for container in docked ship's interior cell
Bool Function FindContainerOnDockedShip(SpaceshipReference DockedShip)
    Cell DockedInterior = DockedShip.GetLinkedCell(SpaceshipLinkedInterior)

    ; Exception: The Key
    If Utility.IntToHex(DockedInterior.GetFormID()) == "00234E05"
        DockedInterior = Game.GetForm(2730214) as Cell ; 0029A8E6
    EndIf

    ; Exception: Siren of the Stars
    If Utility.IntToHex(DockedInterior.GetFormId()) == "001A041B"
        DockedInterior = Game.GetForm(2323412) as Cell ; 002373D4
    EndIf

    ObjectReference TheContainer = GSN_StorageManager.GetContainerFromCell(DockedInterior)

    If TheContainer && !TheContainer.IsLocked()
        Return true
    EndIf

    Return false
EndFunction


Function ShowInfo()
    Float  WeightInContainer
    If Connected
        WeightInContainer = MasterPanel.GetWeightInContainer()
    Else
        WeightInContainer = ActiveContainer.GetRef().GetWeightInContainer()
    EndIf

    Int Used = Math.Round(WeightInContainer)
    Int Capacity = GSN_StorageManager.GetCarryWeight()
    
    String Text = ""
    Text += InfoMessageUsage + " "+ Used + "/" + Capacity
    Text += "\n"

    If Connected
        Text += InfoMessageStatusOnline ;"Status: Offline"
    Else
        Text += InfoMessageStatusOffline ;"Status: Offline"
    EndIf
    Debug.Notification(Text)
EndFunction

Function ActivateMasterPanel()
    MasterPanel.Activate(Player)
EndFunction