local M = {}

local player, stats, goals = {}, {}, {}
local fgScenario, saveData, gameState, mapNodes
local logTag = "gullCoast"
local version = 1

local hornState = false
local prevHornState = false
local hornCount = 0
local hornSteps = {2, 12}
local totalMapNodes = 0
local activeMapNodes = 0
local mapNodeSteps = {0.5, 0.9, 1}
local mapNodeStepIdx = 0
local saveFile = "settings/cloud/gull_coast_gameplay.json"
local achievementsFile = "levels/gull_coast/achievements.json"
local stylesFile = "levels/gull_coast/styles.json"
local messages = {"Welcome to Gull Coast!", "Press your horn twice (H) to view achievement progress."}
local mainTimer, tickTimer, messageTimer, hornTimer = 0, 0, 0, 0
local tickTime = 0.25
local crawlSpeed = 2.5
local fastSpeed = 25
local options = {showRoadNodes = false, disableAchievements = false}
local vecUp = vec3(0, 0, 1)

local function createSaveData()
  saveData = {
    achievements = {
      data = {},
      total = 0,
      disable = false
    },
    scenarios = {
      pass = {},
      win = {},
      quick = {}
    },
    totalTime = 0,
    totalDistance = 0,
    vehiclesSpawned = 0,
    version = version
  }
  if goals then
    for k, v in pairs(goals) do
      saveData.achievements.data[k] = false
    end
    saveData.achievements.total = tableSize(saveData.achievements.data)
  else
    saveData.achievements.disable = true
  end
  jsonWriteFile(saveFile, saveData)
end

local function updateScenarioStats()
  stats.scenariosPassCount = #saveData.scenarios.pass
  stats.scenariosWinCount = #saveData.scenarios.win
  stats.scenariosQuickCount = #saveData.scenarios.quick
end

local function getGameState()
  local state
  if fgScenario then
    state = "scenario"
  elseif not editor.active and be:getEnabled() and core_gamestate.state.state == "freeroam" then
    state = "freeroam"
  else
    state = "other"
  end

  return state
end


local function setHornState(state)
  hornState = state == 1
end

local function dumpAll()
  dump(player)
  dump(saveData)
  dump(stats)
end

local function onExtensionLoaded()
  extensions.load("gullCoast_infoBox")
  local json = jsonReadFile(stylesFile)
  if json then
    gullCoast_infoBox.setStyles(json)
  end

  saveData = jsonReadFile(saveFile)
  goals = jsonReadFile(achievementsFile)
  if saveData then
    if saveData.version ~= version then
      createSaveData()
      saveData._welcome = true
    else
      updateScenarioStats()
    end
  else
    createSaveData()
    saveData._welcome = true
  end
  stats.driveDist = 0
end

local function onExtensionUnloaded()
  if fgScenario then
    core_flowgraphManager.removeManager(fgScenario)
    fgScenario = nil
  end
  jsonWriteFile(saveFile, saveData)
  extensions.unload("gullCoast_infoBox")
end

local function getCondition(cond)
  local var1 = type(cond[1]) ~= "string" and cond[1] or stats[cond[1]]
  local var2 = type(cond[3]) ~= "string" and cond[3] or stats[cond[3]]
  if var1 and var2 then
    if cond[2] == "lte" then
      if var1 <= var2 then
        return true
      end
    elseif cond[2] == "gte" then
      if var1 >= var2 then
        return true
      end
    else
      if var1 == var2 then
        return true
      end
    end
  end
  return false
end

local function testAchievements()
  stats.achievementsDone = 0
  stats.achievementsTotal = saveData.achievements.total
  for k, v in pairs(goals) do
    if not v.type or gameState == v.type then
      if saveData.achievements.data[k] == false and getCondition(v.cond) then
        saveData.achievements.data[k] = true

        local box = {}
        box.image = "achievement"
        box.time = 8
        box.subtitle = v.title
        box.text = {v.desc}
        gullCoast_infoBox.queueInfo(box)
      end
    end
    if saveData.achievements.data[k] == true then
      stats.achievementsDone = stats.achievementsDone + 1
    end
  end
end

local function onGullCoastScenarioReady()
  log("I", logTag, "onGullCoastScenarioReady called")
  for _, fg in ipairs(extensions.core_flowgraphManager.getAllManagers()) do
    local key = fg.variables:get("uniqueKey")
    if key and key ~= "" then
      fgScenario = fg
      log("I", logTag, "Scenario unique key: "..key)

      if tableContains(saveData.scenarios.win, key) then
        fgScenario.variables:changeBase("useVehicleSelector", true)
        log("I", logTag, "Vehicle selector unlocked for this scenario!")
      end
      break
    end
  end
end

local function onGullCoastScenarioFinish()
  log("I", logTag, "onGullCoastScenarioFinish called")
  if fgScenario then
    local key = fgScenario.variables:get("uniqueKey")
    stats[key.."Pass"] = true
    if not tableContains(saveData.scenarios.pass, key) then
      table.insert(saveData.scenarios.pass, key)
    end

    if not fgScenario.variables:get("weakWin") then
      stats[key.."Win"] = true
      if not tableContains(saveData.scenarios.win, key) then
        table.insert(saveData.scenarios.win, key)
      end

      fgScenario.variables:changeBase("useVehicleSelector", true) -- player must win scenario to use vehicle selector
      log("I", logTag, "Vehicle selector unlocked for this scenario!")
    end

    if fgScenario.variables:get("quickWin") then
      stats[key.."Quick"] = true
      if not tableContains(saveData.scenarios.quick, key) then
        table.insert(saveData.scenarios.quick, key)
      end
    end

    if not saveData.achievements.disable then
      testAchievements()
    end
    jsonWriteFile(saveFile, saveData)
  end
end

local function onInfoBoxValueChanged(data)
  if not data then return end
  for k, v in pairs(data) do
    options[k] = v
  end
end

local function onNavgraphReloaded()
  totalMapNodes = 0
end

local function setupPlayer()
  table.clear(player)
  local obj = be:getPlayerVehicle(0)
  if obj then
    player.id = obj:getID()
    player.timers = {reverse = 0}
    player.pos = vec3()
    player.dirVec = vec3()
    player.dirVecUp = vec3()
    player.vel = vec3()
    player.prevPos = vec3(obj:getPosition())
    player.speed = 0
    player.headPassSpeed = 0
    player.closestOtherVehId = 0
    player.otherFlipValue = 0
  end
end

local function trackStats(dt)
  if not player.id or not be:getObjectByID(player.id) then return end
  if map.objects[player.id] then
    local obj = be:getObjectByID(player.id)
    player.pos:set(obj:getPosition())
    player.dirVec:set(obj:getDirectionVector())
    player.dirVecUp:set(obj:getDirectionVectorUp())
    player.vel:set(obj:getVelocity())
    player.speed = player.vel:length()
    player.damage = map.objects[player.id].damage
    player.isDriving = (obj.group ~= "traffic" and player.speed > crawlSpeed and player.speed < 90 and math.abs(player.vel:normalized():dot(vecUp)) <= 0.8) and true or false

    local tickDist = player.pos:distance(player.prevPos)
    player.prevPos:set(player.pos)
    if player.isDriving then
      if tickDist <= 50 then
        stats.driveDist = stats.driveDist + tickDist
        saveData.totalDistance = saveData.totalDistance + tickDist
      end
      if player.dirVec:dot(player.vel:normalized()) <= -0.8 then
        player.timers.reverse = player.timers.reverse + dt
        stats.reverseDriveTime = player.timers.reverse
      else
        player.timers.reverse = 0
      end
      stats.speed = player.speed
    else
      player.timers.reverse = 0
      stats.speed = 0
    end
    if player.headPassSpeed > 0 and player.headPassSpeed - player.speed <= 10 then -- one tick after head pass speed set; tests for player crash
      stats.headPassSpeed = player.speed
    end
    player.headPassSpeed = 0
    stats.damage = player.damage
    stats.time = mainTimer
    saveData.totalTime = saveData.totalTime + dt
    if player.speed > fastSpeed and gullCoast_infoBox.state then
      gullCoast_infoBox.setupMainBox()
    end

    if totalMapNodes == 0 then
      mapNodes = map.getMap().nodes
      totalMapNodes = tableSize(mapNodes)
      activeMapNodes = 0
      mapNodeStepIdx = 0
    else
      local a, b, dist = map.findClosestRoad(player.pos)
      if a then
        local n1, n2 = mapNodes[a], mapNodes[b]
        if dist <= math.max(n1.radius, n2.radius) + 1 then
          local xnorm = clamp(player.pos:xnormOnLine(n1.pos, n2.pos), 0, 1)

          if xnorm > 0.5 then
            a, b = b, a
            n1, n2 = n2, n1
          end

          if not n1.targeted then
            activeMapNodes = activeMapNodes + 1
            stats.roadProgress = activeMapNodes / totalMapNodes
            n1.targeted = true

            if mapNodeSteps[mapNodeStepIdx + 1] and stats.roadProgress >= mapNodeSteps[mapNodeStepIdx + 1] then
              mapNodeStepIdx = mapNodeStepIdx + 1
              if not saveData.achievements.disable then
                ui_message("Road driving progress: "..round(mapNodeSteps[mapNodeStepIdx] * 100).."%", 5, "gullCoastRoads", "explore")
              end
            end
          end
        end
      end
    end

    local bestDist = math.huge
    local bestId
    for k, v in pairs(map.objects) do
      if k ~= player.id then
        local dist = v.pos:squaredDistance(player.pos)
        if dist < bestDist then
          bestDist = dist
          bestId = k
        end

        if player.isDriving and dist <= square(80) and player.headPassSpeed == 0 then
          local posAhead = v.pos + v.dirVec * (10 + player.speed / 4)
          local point = linePointFromXnorm(v.pos, posAhead, clamp(player.pos:xnormOnLine(v.pos, posAhead), 0, 1))
          local pointDist = point:distance(player.pos)
          if pointDist <= 5 and v.dirVec:dot(player.dirVec) <= -0.7 then
            player.headPassSpeed = player.speed
          end
        end
      end
    end

    if bestId then
      if player.closestOtherVehId ~= bestId then
        player.otherFlipValue = 0
      else
        if bestDist <= square(15) then
          local dirVecUpDot = map.objects[bestId].dirVecUp:dot(vecUp)
          local value = math.max(0, -dirVecUpDot)
          if value - player.otherFlipValue < 0.7 then
            player.otherFlipValue = lerp(player.otherFlipValue, value, 0.5)
          end

          if player.otherFlipValue >= 0.7 then
            stats.flipAction = true
          end
        else
          player.otherFlipValue = 0
        end
      end
      player.closestOtherVehId = bestId
    end
  end
end

local offColor = ColorF(1, 1, 1, 0.5)
local onColor = ColorF(1, 0.4, 0, 0.5)
local function drawRoadNodes()
  for k, n in pairs(mapNodes) do
    if n.pos:squaredDistance(getCameraPosition()) <= square(400) then
      debugDrawer:drawSphere(n.pos, 0.5, n.targeted and onColor or offColor)
    end
  end
end

local function onUpdate(dt, dtSim)
  if not player.id then
    setupPlayer()
  end
  if fgScenario and fgScenario.runningState == "stopped" then
    core_flowgraphManager.removeManager(fgScenario)
    fgScenario = nil
  end
  gameState = getGameState()

  if gameState == "freeroam" and player.id then
    if saveData._welcome and mainTimer >= 3 then
      if messageTimer == 0 then
        ui_message(messages[1], 9, "gullCoast", "info")
        table.remove(messages, 1)
        if not messages[1] then saveData._welcome = nil end
      end
      messageTimer = messageTimer + dt
      if messageTimer >= 10 then
        messageTimer = 0
      end
    end

    if tickTimer == 0 then
      saveData.achievements.disable = options.disableAchievements
      trackStats(tickTime)
      if not saveData.achievements.disable and mainTimer >= 3 then
        testAchievements()
      end
    end
    tickTimer = tickTimer + dtSim
    if tickTimer >= tickTime then
      tickTimer = 0
    end

    mainTimer = mainTimer + dt

    if options.showRoadNodes then
      drawRoadNodes()
    end
  end

  if player.id and be:getObjectByID(player.id) then
    queueCallbackInVehicle(be:getObjectByID(player.id), "extensions.mainLevel.setHornState", "electrics.values.horn")
  else
    hornState = false
    prevHornState = false
  end
  if player.id and player.speed <= fastSpeed then
    if prevHornState ~= hornState then
      if hornState then
        hornCount = hornCount + 1
        hornTimer = 0
      end
      prevHornState = hornState
    end

    local idx = options.disableAchievements and 2 or 1
    if hornCount == hornSteps[idx] then
      local box = {}
      box.width = 900
      box.height = 440
      box.subtitle = "Achievements"
      box.fontSize = 1
      box.goals = deepcopy(saveData.achievements)
      box.goalsRef = deepcopy(goals)
      box.options = options
      gullCoast_infoBox.setupMainBox(box)
      jsonWriteFile(saveFile, saveData)
      hornCount = 0
    end
    if hornTimer >= 0.5 then
      hornTimer = 0
      hornCount = 0
    end
    if hornCount > 0 then
      hornTimer = hornTimer + dt
    end
  end
end

local function onBeamNGTrigger(data)
  if gameState == "freeroam" and data.subjectID == player.id and data.event == "enter" then
    if string.find(data.triggerName, "special") then
      stats[data.triggerName] = true
    end
  end
end

local function onVehicleSpawned(id)
  if not player.id or not be:getObjectByID(player.id) or player.id == id then
    setupPlayer()
  end
  if gameState == "freeroam" then
    saveData.vehiclesSpawned = saveData.vehiclesSpawned + 1
  end
end

local function onVehicleSwitched()
  setupPlayer()
end

local function onVehicleDestroyed(id)
  if id == be:getPlayerVehicleID(0) then
    setupPlayer()
  end
end

local function onGameStateUpdate()
  mainTimer = 0
  totalMapNodes = 0
  setupPlayer()
  if saveData then
    updateScenarioStats()
    jsonWriteFile(saveFile, saveData)
  end
end

local function onExit()
  jsonWriteFile(saveFile, saveData)
end

M.createSaveData = createSaveData
M.setHornState = setHornState
M.dumpAll = dumpAll

M.onExtensionLoaded = onExtensionLoaded
M.onExtensionUnloaded = onExtensionUnloaded
M.onGullCoastScenarioReady = onGullCoastScenarioReady
M.onGullCoastScenarioFinish = onGullCoastScenarioFinish
M.onInfoBoxValueChanged = onInfoBoxValueChanged
M.onNavgraphReloaded = onNavgraphReloaded
M.onUpdate = onUpdate
M.onBeamNGTrigger = onBeamNGTrigger
M.onVehicleSpawned = onVehicleSpawned
M.onVehicleSwitched = onVehicleSwitched
M.onVehicleResetted = onVehicleResetted
M.onVehicleDestroyed = onVehicleDestroyed
M.onGameStateUpdate = onGameStateUpdate
M.onExit = onExit

return M