local M = {}

local imguiUtils = require("ui/imguiUtils")
local im = ui_imgui

local logTag = "infoBox"

M.state = false

local ready = im.BoolPtr(true)
local requeue = false
local baseColor = im.ImVec4(1, 1, 1, 1)
local options = {disableAchievements = im.BoolPtr(false), showRoadNodes = im.BoolPtr(false)}
local queue, styles, mainData = {}, {}, {}
local timers = {
  queue = 0,
  resetS = 0,
  resetA = 0
}
local elements = {
  text = 0,
  windowBase = 2,
  windowBorder = 5,
  comboBase = 7,
  comboHover = 8,
  titleBarInactive = 10,
  titleBarActive = 11,
  mark = 18,
  buttonBase = 21,
  buttonHover = 22,
  buttonPress = 23,
  collapserBase = 24,
  collapserHover = 25,
  collapserPress = 26,
  separator = 27,
  resizerBase = 30,
  resizerHover = 31,
  resizerPress = 32
}

local function parseColor(old) -- parses a string color to an ImVec4
  local new = old and stringToTable(old, " ") or nil
  if not new or #new ~= 4 then
    log("I", logTag, "Invalid color string! Using default color...")
    return baseColor
  else
    return im.ImVec4(tonumber(new[1]), tonumber(new[2]), tonumber(new[3]), tonumber(new[4]))
  end
end

local function setStyles(data) -- sets various style data for the imgui window
  if not data then data = {} end
  table.clear(styles)
  styles.elements = {}
  styles.images = {}
  styles.baseWidth = data.baseWidth or 600
  styles.baseHeight = data.baseHeight or 400
  styles.fontSize1 = data.fontSize1 or 1
  styles.fontSize2 = data.fontSize2 or 1
  styles.color1 = parseColor(data.color1)
  styles.color2 = parseColor(data.color2)
  styles.color3 = parseColor(data.color3)

  if data.elements then
    for k, v in pairs(data.elements) do
      if elements[k] and v then
        styles.elements[elements[k]] = parseColor(v)
      end
    end
  end

  if data.images then
    for k, v in pairs(data.images) do
      styles.images[k] = imguiUtils.texObj(v)
    end
  end
end

local function resizeBox(data) -- returns the expected dimensions of the text box
  if not data or data.noResize then return styles.baseWidth, styles.baseHeight end
  local width, height, bestWidth = 0, 0, 0
  local size = styles.fontSize1
  for i, v in ipairs(data.text) do
    local lineWidth = string.len(tostring(v)) * (7.2 + size * 2)
    if lineWidth >= bestWidth then
      bestWidth = lineWidth
    end
  end
  width = bestWidth

  if data.image then
    width = math.max(260, width)
    height = height + 40
  end

  if data.title then height = height + size * 26 end
  if data.subtitle then height = height + size * 26 end
  for i, v in ipairs(data.text) do
    height = height + size * 26
  end
  width = math.ceil(clamp(width, 80, 1280))
  height = math.ceil(clamp(height, 20, 720))

  return width, height
end

local function setupColumns(num, width) -- sets up equal width imgui columns
  im.Columns(num)
  for i = 0, num - 1 do
    im.SetColumnWidth(i, width)
  end
end

local function showInfoBox(data) -- shows a pop-up UI in the corner
  if not data.width then data.width = styles.baseWidth end
  if not data.height then data.height = styles.baseHeight end

  im.SetNextWindowPos(im.ImVec2(0, 60), im.Cond_Always)
  im.SetNextWindowSize(im.ImVec2(data.width, data.height), im.Cond_Always)

  for k, v in pairs(styles.elements) do
    im.PushStyleColor2(k, v)
  end
  if im.Begin("Info Box##gullCoast", ready, im.flags(im.WindowFlags_NoTitleBar, im.WindowFlags_NoResize, im.WindowFlags_NoScrollbar)) then
    im.SetWindowFontScale(styles.fontSize1)
    if data.image and styles.images[data.image] then
      im.Image(styles.images[data.image].texId, im.ImVec2(256, 24), im.ImVec2(0, 0), im.ImVec2(1, 1), im.GetStyleColorVec4(im.Col_Text))
    end
    if data.title then
      im.TextColored(styles.color1, data.title)
    end
    if data.subtitle then
      im.TextColored(styles.color2, data.subtitle)
    end
    for i, v in ipairs(data.text) do
      im.Text(v)
    end
    im.End()
  end
  for k, v in pairs(styles.elements) do
    im.PopStyleColor()
  end
end

local function showMainBox(data) -- shows a main window
  for k, v in pairs(styles.elements) do
    im.PushStyleColor2(k, v)
  end

  im.SetNextWindowPos(im.ImVec2(24, 24), im.Cond_Always)
  im.SetNextWindowSize(im.ImVec2(data.width, data.height), im.Cond_FirstUseEver)

  if im.Begin("Achievements##gullCoast", ready, im.flags(im.WindowFlags_NoTitleBar)) then
    im.SetWindowFontScale(styles.fontSize2)
    if data.image and styles.images[data.image] then
      im.Image(styles.images[data.image].texId, im.ImVec2(256, 24), im.ImVec2(0, 0), im.ImVec2(1, 1), im.GetStyleColorVec4(im.Col_Text))
    end
    if data.title then
      im.TextColored(styles.color1, data.title)
    end
    if data.subtitle then
      im.TextColored(styles.color2, data.subtitle)
    end

    if not data.goals.done then
      data.goals.done, data.goals.notDone = {}, {}
      for k, state in pairs(data.goals.data) do
        if state then
          table.insert(data.goals.done, k)
        else
          table.insert(data.goals.notDone, k)
        end
      end
      table.sort(data.goals.done)
      table.sort(data.goals.notDone)
    end

    im.TextColored(styles.color3, "Done: ")
    local width = im.GetContentRegionAvailWidth() / 4
    setupColumns(4, width)
    for i, v in ipairs(data.goals.done) do
      im.TextUnformatted(data.goalsRef[v].title)
      im.NextColumn()
    end
    im.Columns(1)
    im.TextColored(styles.color3, "Not Done: ")
    setupColumns(4, width)
    for i, v in ipairs(data.goals.notDone) do
      im.TextUnformatted(data.goalsRef[v].title)
      im.NextColumn()
    end
    im.Columns(1)
    im.TextColored(styles.color3, "Progress: ")
    im.SameLine()
    im.Text(#data.goals.done.." / "..data.goals.total)
    im.Separator()

    if im.Checkbox("Disable Achievements", options.disableAchievements) then
      extensions.hook("onInfoBoxValueChanged", {disableAchievements = options.disableAchievements[0]})
    end
    if im.Checkbox("Show Road Progress", options.showRoadNodes) then
      extensions.hook("onInfoBoxValueChanged", {showRoadNodes = options.showRoadNodes[0]})
    end

    if im.Button("Close") then
      M.state = false
    end
  im.End()
  end
  for k, v in pairs(styles.elements) do
    im.PopStyleColor()
  end
end

local function queueInfo(data)
  if not data then return end

  if not data.time then data.time = 1e6 end
  if type(data.text) ~= "table" then
    data.text = {data.text}
  end
  if not data.width or not data.height then
    data.width, data.height = resizeBox(data)
  end
  if timers.queue == 0 then
    timers.queue = data.time
  end

  local new = data
  table.insert(queue, new)
end

local function stopInfo()
  table.clear(queue)
end

local function setupMainBox(data)
  stopInfo()
  M.state = type(data) == "table"
  if M.state then
    if not data.width or not data.height then
      data.width, data.height = 800, 420
    end
    for k, v in pairs(data.options or {}) do
      options[k] = im.BoolPtr(v)
    end
    mainData = data
  end
end

local function onExtensionLoaded()
  setStyles()
end

local function onUpdate(dt)
  local q = queue[1]
  if q then
    if timers.queue > 0.25 then
      showInfoBox(q)
    elseif timers.queue <= 0 then
      requeue = true
    end
  end

  if requeue and q then
    timers.queue = queue[2] and queue[2].time or 0
    table.remove(queue, 1)
    requeue = false
  end

  if M.state then
    showMainBox(mainData)
  end
  for k, v in pairs(timers) do
    timers[k] = v > 0 and v - dt or 0
  end
end

M.setStyles = setStyles
M.queueInfo = queueInfo
M.stopInfo = stopInfo
M.setupMainBox = setupMainBox
M.onUpdate = onUpdate
M.onExtensionLoaded = onExtensionLoaded

return M