		--------------------------------------
-- MotorsportsComplex DragLights script --
	-- Made by 2FastRacing | www.2fast.racing --
		--------------------------------------

local M = {}
local Objects = {}
local lane_side = { 0, 0 }
local in_prestage = { false, false }
local has_finished = { false, false }
local in_stage = { false, false }
local in_stage_time = 0
local in_false_start = false
local in_race = false
local in_race_time = 0

local function InPreStage() --Check prestage status
	local ret = 0
	if in_prestage[1] then ret = ret + 1 end
	if in_prestage[2] then ret = ret + 1 end
	return ret
end

local function allInStage()
	local numPrestage = InPreStage()
	if numPrestage == 0 then
		return false
	end

	local num = 0 --math
	if in_stage[1] then num = num + 1 end
	if in_stage[2] then num = num + 1 end
	return num == numPrestage
end

local function hideLaneLights(lane) --Set lights hidden by default
	Objects[lane].lights.prestage:setHidden(true)
	Objects[lane].lights.stage:setHidden(true)
	for _, v in ipairs(Objects[lane].lights.amber) do
		v:setHidden(true)
	end
	--Objects[lane].lights.green:setHidden(true)
	Objects[lane].lights.green:setHidden(true)
	Objects[lane].lights.red:setHidden(true)
end

local function hideDisplay(display) --Set display hidden by default
	for _, v in ipairs(display.digits) do
		v:setHidden(true)
	end
	display.dot:setHidden(true)
end

local function resetLaneDisplays(lane)
	hideDisplay(Objects[lane].displays.time)
	hideDisplay(Objects[lane].displays.speed)
end

local function setDisplay(display, number, digits) --Set the drag displays ready
	if number == 0 then
		hideDisplay(display)
		return
	end

	local str_left = tostring(math.floor(number) % (10 ^ digits))
	local str_right = tostring(number % 1):sub(3)

	for i = 1, 5, 1 do
		local c = '0'
		if i <= digits then
			local index = i - (digits - #str_left)
			c = str_left:sub(index, index)
		else
			local index = i - digits
			c = str_right:sub(index, index)
		end

		if c ~= '' then
			local path = 'art/shapes/quarter_mile_display/display_' .. c .. '.dae'

			display.digits[i]:preApply()
			display.digits[i]:setField('shapeName', 0, path)
			display.digits[i]:setHidden(false)
			display.digits[i]:postApply()
		else
			display.digits[i]:setHidden(true)
		end
	end

	display.dot:setHidden(false)
end

local function resetLane(lane)
	hideLaneLights(lane)
	resetLaneDisplays(lane)
end

local function resetAllLanes() --Reset all lanes
	resetLane(1)
	resetLane(2)

	lane_side = { 0, 0 }

	in_prestage = { false, false }
	in_stage = { false, false }
	has_finished = { false, false }
	in_false_start = false
	in_race = false
end

local function stopWithReason(reason) --Stop race & print reason.
	print('== Stopping: ' .. reason)
	resetAllLanes()
	--TODO: Show notification
end

local function getObjects(side) --Identify map objects
	local ret = {}

	ret.lights = {}
	ret.lights.prestage = scenetree.findObject('lightPrestage' .. side)
	ret.lights.stage = scenetree.findObject('lightStage' .. side)
	ret.lights.amber = {
		scenetree.findObject('lightAmber' .. side .. '1'),
		scenetree.findObject('lightAmber' .. side .. '2'),
		scenetree.findObject('lightAmber' .. side .. '3'),
	}
	ret.lights.green = scenetree.findObject('lightGreen' .. side)
	ret.lights.red = scenetree.findObject('lightRed' .. side)

	ret.displays = {}
	ret.displays.time = {
		digits = {
			scenetree.findObject('timeDisp' .. side .. '1'),
			scenetree.findObject('timeDisp' .. side .. '2'),
			scenetree.findObject('timeDisp' .. side .. '3'),
			scenetree.findObject('timeDisp' .. side .. '4'),
			scenetree.findObject('timeDisp' .. side .. '5'),
		},
		dot = scenetree.findObject('timeDisp' .. side .. 'Dot'),
	}
	ret.displays.speed = {
		digits = {
			scenetree.findObject('speedDisp' .. side .. '1'),
			scenetree.findObject('speedDisp' .. side .. '2'),
			scenetree.findObject('speedDisp' .. side .. '3'),
			scenetree.findObject('speedDisp' .. side .. '4'),
			scenetree.findObject('speedDisp' .. side .. '5'),
		},
		dot = scenetree.findObject('speedDisp' .. side .. 'Dot'),
	}

	return ret
end

local function loadObjects() --GetObjects
	Objects = {
		getObjects('L'),
		getObjects('R'),
	}
end

local function findVehicleLane(vehicle) --Get Vehicle Lane
	if lane_side[1] == vehicle then
		return 1
	elseif lane_side[2] == vehicle then
		return 2
	end
	return 0
end

local function onClientPostStartMission() --initialize the script
	loadObjects()
	resetAllLanes()

	print('== 2FastRacing Drag script initialized')
end

local function onExtensionLoaded() --initialize the script on reload
	loadObjects()

	if Objects[1].lights.prestage ~= nil then --fix lights
		resetAllLanes()

		print('== 2FastRacing Drag script initialized from reload')
	end
end

local function monitorLaneTriggers(lane, data)
	local side = ''
	if lane == 1 then
		side = 'L'
	elseif lane == 2 then
		side = 'R'
	end

	if data.event == 'exit' then
		if in_race and not has_finished[lane] then
			if data.triggerName == 'laneTrig' .. side then
				stopWithReason(side .. ' lane crossed.')
			end

		elseif in_prestage[lane] and not in_stage[lane] then
			if data.triggerName == 'prestageTrig' .. side then
				in_prestage[lane] = false
				Objects[lane].lights.prestage:setHidden(true)
			end

		elseif in_stage[lane] then
			if data.triggerName == 'stageTrig' .. side then
				stopWithReason(side .. ' backed out of stage.')
			end
		end
	end

	if data.event == 'enter' then --If not in race ignore triggers.
		if not in_race then
			if not in_prestage[lane] then
				if data.triggerName == 'prestageTrig' .. side then
					if in_false_start then
						in_false_start = false
						Objects[lane].lights.red:setHidden(true)
					end

					in_prestage[lane] = true
					Objects[lane].lights.prestage:setHidden(false)
					resetLaneDisplays(lane)

					lane_side[lane] = data.subjectID
				end

			elseif not in_stage[lane] then
				if data.triggerName == 'startTrig' .. side then
					has_finished[lane] = false
					in_stage[lane] = true
					in_stage_time = 0
					Objects[lane].lights.stage:setHidden(false)
				end

			elseif in_stage[lane] then
				if data.triggerName == 'falseStartTrig' then
					stopWithReason(side .. ' jumped the tree.')
					in_false_start = true
					Objects[lane].lights.red:setHidden(false)
				end
			end
		end
	end
end

local function onBeamNGTrigger(data)
	monitorLaneTriggers(1, data)
	monitorLaneTriggers(2, data)

	if data.event == 'enter' then
		if in_race then
			if data.triggerName == 'finishTrig' then
				local vehicle = be:getObjectByID(data.subjectID)
				local lane = findVehicleLane(data.subjectID)

				local finishtime = in_race_time
				local finishspeed = vehicle:getVelocity():len() * 2.2369362920544

				print('====== Lane ' .. lane .. ' ======')
				print('==  Finish time: ' .. finishtime)
				print('== Finish speed: ' .. finishspeed)

				setDisplay(Objects[lane].displays.time, finishtime, 2)
				setDisplay(Objects[lane].displays.speed, finishspeed, 3)

				in_prestage[lane] = false
				in_stage[lane] = false
				has_finished[lane] = true
				lane_side[lane] = 0
				hideLaneLights(lane)

				if InPreStage() == 0 then
					print('-- Race ended')
					in_race = false
				end
			end
		end
	end
end

local function setAllLightsEnabled(group, value)
    for i = 0, group.obj:getCount(), 1 do
        local id = group.obj:idAt(i)
        local obj = scenetree.findObjectById(id)
        if obj and obj.obj:isSubClassOf('LightBase') then
            obj.obj:setLightEnabled( value )
        end
    end
end

local function onUpdate(dtReal, dtSim, dtRaw)
	if in_race then
		in_race_time = in_race_time + dtSim

	elseif allInStage() then
		in_stage_time = in_stage_time + dtSim * 2

		local seconds = math.floor(in_stage_time - 1)

		for lane = 1, 2, 1 do
			if in_stage[lane] then
				for i, v in ipairs(Objects[lane].lights.amber) do
					v:setHidden(seconds ~= i)
				end
				Objects[lane].lights.green:setHidden(seconds < 4)
			end
		end

		if seconds >= 4 then
			in_race = true
			in_race_time = 0
		end
	end
	local tod = scenetree.tod
    if not tod then return end

    local value = false
    if tod.time > 0.24 and tod.time < 0.77 then
        value = true
    end

    if lastValue == value then return end
    lastValue = value
print('DynamicLights Loaded!')
    if scenetree.NightLights then
      setAllLightsEnabled(scenetree.NightLights, value )
    end
    if scenetree.LightPosts then
      setAllLightsEnabled(scenetree.LightPosts, value )
    end
end

M.onClientPostStartMission = onClientPostStartMission
M.onExtensionLoaded = onExtensionLoaded
M.onBeamNGTrigger = onBeamNGTrigger
M.onUpdate = onUpdate

return M
