-- written by DaddelZeit
-- DO NOT USE WITHOUT PERMISSION

local M = {}
M.type = "auxiliary"

local min = math.min
local abs = math.abs

local engine

local lastGear
local state = { "idle", "stopping", "wait", "starting" }
local currentState = 1
local processRunning = false
local frequencyCalled = 0
local tempDisabled
local timer = 0
local timeSinceStop = 0
local origStarterValues = {}
local shaftDevice

local startStopEnabled = true
local startStopModeEnabled = true

local speedThreshold
local pitchThreshold
local brakeThreshold
local minEngineTemp
local steerThreshold
local idleTimeNeeded
local needThrottle
local useSmartAdaption

-- smart stuff
local brakeTimer = 0 -- from 1 (~5kmh) to 0 / pedal input to allow premature braking

local function nextState()
  currentState = math.max((currentState + 1) % 5, 1) -- because lua tables start at 1
end

local function handleStarterSpeed(call)
  if call == 1 then
    engine.starterTorque = origStarterValues.starterTorque * 2.5
    engine.starterMaxAV = engine.idleAV / 2
    --engine.starterThrottleKillTime = origStarterValues.starterThrottleKillTime / 2.5
    --engine.idleStartCoef = 0.8
  else
    engine.starterTorque = origStarterValues.starterTorque
    engine.starterMaxAV = origStarterValues.starterMaxAV
    --engine.starterThrottleKillTime = origStarterValues.starterThrottleKillTime
    --engine.idleStartCoef = origStarterValues.idleStartCoef
  end
end

local brakeMeasurementTaken = false
local function applyData()
  idleTimeNeeded = 3 - brakeTimer * electrics.values.brake * 3 * (1+(controller.mainController.drivingAggression or 0))
  speedThreshold = 5 + brakeTimer * 3 + electrics.values.brake
end

local function dataHandling(dt)
  local evals = electrics.values
  if evals.wheelspeed <= 1.5 then
    if evals.brake ~= 0 and not brakeMeasurementTaken then
      brakeTimer = brakeTimer + dt
    end

    if evals.wheelspeed < 0.05 and not brakeMeasurementTaken then
      applyData()
      brakeMeasurementTaken = true
    end
  else
    brakeMeasurementTaken = false
    brakeTimer = 0
  end
end

local function disconnectShaft()
  shaftDevice.isPhysicallyDisconnected = true
  shaftDevice:setMode("connected")
end

local function connectShaft()
  shaftDevice.isPhysicallyDisconnected = false
  shaftDevice:setMode("connected")
end

local function zeitADASUpdateAuto(dt)
  if electrics.values.fuel == 0 then return end -- system is enabled
  if useSmartAdaption then dataHandling(dt) end

  local evals = electrics.values

  timer = timer + dt
  if timer >= 5 then
    frequencyCalled = 0
    timer = 0
  end

  if frequencyCalled >= 2 or evals.brakeOverride then
    tempDisabled = true
  end

  if tempDisabled then
    if evals.wheelspeed > 5 then
      tempDisabled = false
    end
  end

  if not evals.watertemp or evals.ignitionLevel ~= 2 then return end
  if not processRunning then
    local _, pitch = obj:getRollPitchYaw()
    if (evals.watertemp <= minEngineTemp) or (pitchThreshold <= pitch) or (evals.parkingbrake ~= 0) or (evals.steering > steerThreshold) or (evals.steering < -steerThreshold) then return end
    if ((evals.gear == "N" and lastGear == "N" and evals.gearboxMode == "realistic") or (evals.gear == "P" and lastGear == "P") or (evals.gear == "R" and lastGear == "R") or (evals.gear == "S"..evals.gearIndex and lastGear == "S"..evals.gearIndex) or (evals.gear == "M"..evals.gearIndex and lastGear == "M"..evals.gearIndex)) and not processRunning then lastGear = evals.gear return end
    if tempDisabled or not startStopEnabled or not startStopModeEnabled then return end
  end

  if state[currentState] == "idle" then
    if evals.wheelspeed <= speedThreshold and evals.brake > brakeThreshold then
      timeSinceStop = timeSinceStop + dt
      if timeSinceStop >= idleTimeNeeded then
        frequencyCalled = frequencyCalled + 1
        nextState()
        controller.mainController.setEngineIgnition(false)
        electrics.values.startStopActive = 1
        processRunning = true
        timeSinceStop = 0
        handleStarterSpeed(1)
      end
    else
      handleStarterSpeed(0)
    end
  elseif state[currentState] == "stopping" then
    if evals.rpm < 50 or evals.throttle ~= 0 then
      nextState()
    end
  elseif state[currentState] == "wait" then
    local isNextState = false
    if needThrottle then
      if evals.brake <= brakeThreshold and evals.throttle ~= 0 then
        isNextState = true
      end
    else
      if evals.brake <= brakeThreshold then
        isNextState = true
      end
    end
    if (((evals.gear == "N" and evals.gearboxMode == "realistic") or evals.gear == "R" or evals.gear == "P" or evals.gear == "S"..evals.gearIndex or evals.gear == "M"..evals.gearIndex) or tempDisabled) or isNextState then
      electrics.setIgnitionLevel(3)
      disconnectShaft()
      nextState()
    end
  elseif state[currentState] == "starting" then
    if evals.rpm >= (engine.idleAV * 9.549296596425384) - 200 then -- engine is on
      nextState()
      connectShaft()
      processRunning = false
    end
    electrics.values.startStopActive = 0
  end

  lastGear = evals.gear
end

local function zeitADASUpdateManual(dt)
  if electrics.values.fuel == 0 then return end -- system is enabled
  local evals = electrics.values
  if useSmartAdaption then dataHandling(dt) end

  timer = timer + dt
  if timer >= 6 then
    frequencyCalled = 0
    timer = 0
  end

  if frequencyCalled >= 2 or evals.brakeOverride then
    tempDisabled = true
  end

  if tempDisabled then
    if evals.wheelspeed > 5 then
      tempDisabled = false
    end
  end

  if not evals.watertemp or evals.ignitionLevel ~= 2 then return end
  if not processRunning then
    local _, pitch = obj:getRollPitchYaw()
    if (evals.watertemp <= minEngineTemp) or (pitchThreshold <= pitch) or (evals.parkingbrake ~= 0) or (evals.steering > steerThreshold) or (evals.steering < -steerThreshold) then return end
    if tempDisabled or not startStopEnabled or not startStopModeEnabled then return end
  end

  if state[currentState] == "idle" then
    if needThrottle then
      if evals.wheelspeed <= speedThreshold and evals.brake > brakeThreshold and evals.clutch == 0 then
        timeSinceStop = timeSinceStop + dt

        if evals.rpm > engine.idleAV then -- engine is on
          handleStarterSpeed(0)
        end

        if timeSinceStop >= idleTimeNeeded then
          frequencyCalled = frequencyCalled + 1
          nextState()
          controller.mainController.setEngineIgnition(false)
          electrics.values.startStopActive = 1
          processRunning = true
          timeSinceStop = 0
          handleStarterSpeed(1)
        end
      end
    else
      if evals.wheelspeed <= speedThreshold and evals.brake > brakeThreshold and evals.rpm >= 500 then
        timeSinceStop = timeSinceStop + dt
        if timeSinceStop >= idleTimeNeeded then
          frequencyCalled = frequencyCalled + 1
          nextState()
          controller.mainController.setEngineIgnition(false)
          electrics.values.startStopActive = 1
          processRunning = true
          timeSinceStop = 0
          handleStarterSpeed(1)
        end
      else
        handleStarterSpeed(0)
      end
    end
  elseif state[currentState] == "stopping" then
    if evals.rpm < 50 then
      nextState()
    end
  elseif state[currentState] == "wait" then
    local isNextState = false
    if needThrottle then
      isNextState = evals.clutch ~= 0 or tempDisabled

      if isNextState then
        electrics.setIgnitionLevel(3)
        disconnectShaft()
        nextState()
      end
    else
      isNextState = evals.brake <= brakeThreshold
      if (evals.gear == 0 and evals.gearboxMode == "realistic") or tempDisabled or isNextState then
        electrics.setIgnitionLevel(3)
        disconnectShaft()
        nextState()
      end
    end
  elseif state[currentState] == "starting" then
    if evals.rpm >= (engine.idleAV * 9.549296596425384) - 200 then -- engine is on
      handleStarterSpeed(0)
      nextState()
      connectShaft()
      processRunning = false
    end
  end

  lastGear = evals.gear
end

local function reset()
  electrics.values.startStopActive = 0
end

local function init(jbeamData)
  engine = powertrain.getDevice("mainEngine") or powertrain.getDevicesByType("combustionEngine")[1]
  if not engine then M.zeitADASUpdate = nop return end

  local curDevice = powertrain.getDevice("mainEngine")
  while curDevice.type ~= "shaft" do
      if not curDevice.children[1] then return end
      curDevice = curDevice.children[1]
  end
  shaftDevice = curDevice

  if jbeamData.gearboxType == "auto" then
    M.zeitADASUpdate = zeitADASUpdateAuto
    lastGear = "N"
  elseif jbeamData.gearboxType == "manual" then
    M.zeitADASUpdate = zeitADASUpdateManual
    lastGear = 0
  else -- not supported
    M.zeitADASUpdate = nop
  end

  speedThreshold = jbeamData.speedThreshold or 2
  pitchThreshold = jbeamData.pitchThreshold or 0.1
  brakeThreshold = jbeamData.brakeThreshold or 0
  minEngineTemp = jbeamData.minEngineTemp or 70
  steerThreshold = jbeamData.steerThreshold or 110
  idleTimeNeeded = jbeamData.idleTimeNeeded or 3
  needThrottle = jbeamData.needThrottle or false
  useSmartAdaption = jbeamData.useSmartAdaption or false

  origStarterValues = {
    starterTorque = engine.starterTorque,
    starterMaxAV = engine.starterMaxAV,
  }

  currentState = 1
  electrics.values.startStopActive = 0
end

local function setGlobalActive(bool)
  startStopEnabled = bool
end

local function setParameters(params)
  startStopModeEnabled = params.isEnabled
end

M.setParameters = setParameters
M.setGlobalActive = setGlobalActive
M.reset = reset
M.init = init
M.zeitADASUpdate = nop -- disable on first start, defined in init()

return M
