#!/usr/bin/env python

"""
Filename: rdr2_more_autosaves.py
Author: cgar
Date: 2025-08-03
Version: 1.0
Description: A simple python script to effectively convert manual save slots into more autosave slots in RDR2.
License: WTFNMFPL-1.0
"""

import os
from glob import glob
from pathlib import Path
from shutil import copy2
from time import sleep
from datetime import datetime, timedelta


# Choose a MANUAL_SLOT_OVERWRITE_COUNT from 1 to 15
# Setting 1 means there will be 1 additional autosave slot, which will be SRDR30014 (2 autosave slots total).
# Setting 15 means all 16 slots will be autosave slots, SRDR30000 to SRDR30014 plus SRDR30015 (which is always autosave).
# The default setting of 10 means the first 5 slots (SRDR30000 to SRDR30004) are reserved for manual saving.
MANUAL_SLOT_OVERWRITE_COUNT = 10

# Disables autosave copy if less than SAVE_TIME_DELTA_MINUTES has passed, 0=Always copy.
SAVE_TIME_DELTA_MINUTES = 1

# Round time delta calculation to the start of the minute, only really useful when SAVE_TIME_DELTA_MINUTES=1 to have
# the shortest possible interval that can be told apart by save name in the save menu, since it doesn't show seconds.
SAVE_TIME_MINUTE_ALIGNED = True

CUSTOM_PROFILE_PATHS = (
    # Add any custom paths to your RDR2 profile dir here. 1 per line, single or double quoted, ending in a comma.
)

SAVE_NAME_PREFIX = 'SRDR300'
AUTOSAVE_NAME = SAVE_NAME_PREFIX + '15'

SLEEP_DURATION_SEC = 1


def find_profile_path_from_defaults() -> Path | bool:
    # TODO: Check this is all the defaults.
    #       Parse non default steam library location(s) from ~/.steam.

    home = Path.home()
    wine_home = home / f'.wine/drive_c/users/{home.name}'

    vanilla_glob = 'Documents/Rockstar Games/Red Dead Redemption 2/Profiles/????????'
    proton_glob = f'.steam/steam/steamapps/compatdata/1174180/pfx/drive_c/users/{home.name}/{vanilla_glob}'
    goldberg_suffix = 'AppData/Roaming/Goldberg SocialClub Emu Saves/RDR2'
    razor_suffix    = 'AppData/Roaming/.1911/Red Dead Redemption 2/profile'

    for profile_path in (Path(p) for p in glob(f'{home}/{vanilla_glob}')):
        if check_is_profile_dir(profile_path):
            return profile_path

    for profile_path in (Path(p) for p in glob(f'{home}/{proton_glob}')):
        if check_is_profile_dir(profile_path):
            return profile_path

    for profile_path in (home/goldberg_suffix, home/razor_suffix):
        if check_is_profile_dir(profile_path):
            return profile_path

    for profile_path in (Path(p) for p in glob(f'{wine_home}/{vanilla_glob}')):
        if check_is_profile_dir(profile_path):
            return profile_path

    for profile_path in (wine_home/goldberg_suffix, wine_home/razor_suffix):
        if check_is_profile_dir(profile_path):
            return profile_path

    return False


def exit_wait_nt(errno: int = 0) -> None:
    if os.name == 'nt':
        input('Press enter to close...')
    exit(errno)


def check_is_profile_dir(path: Path) -> bool:
    if not path.exists():
        return False

    if not path.is_dir():
        print('ERROR: profile directory is not a directory:', path)
        return False

    for file in path.glob('*'):
        if file.name.upper() == 'PROFILESETTINGS':
            return True
    else:
        print('ERROR: profile directory does not contain ProfileSettings file:', path)
        return False


def parse_save_file(save_path: Path) -> tuple[str, str]:
    with open(save_path, 'rb') as f:
        name = f.read(260)

    if len(name) < 260 or not name.startswith(b'\x00\x00\x00\x04'):
        print('ERROR: Cannot parse save:')
        print(' ', save_path.name)
        exit_wait_nt(1)

    name = name.replace(b'\x00', b'').lstrip(b'\x04').decode('ascii')
    name, date = name.rsplit(' - ', maxsplit=1)
    date = f'20{date[6:8]}/{date[0:2]}/{date[3:5]} {date[9:]}'  # fix "arbitrary retarded rollercoaster" date format

    return date, name


def parse_all_saves(profile_path: Path) -> list[tuple[int, str, str, Path]]:
    reserved_slots = 15-MANUAL_SLOT_OVERWRITE_COUNT
    mtimes, dates, names, paths = [], [], [], []
    for slot_num in range(16):
        if slot_num < reserved_slots:
            continue

        save_path = profile_path / f'{SAVE_NAME_PREFIX}{slot_num:02d}'
        if save_path.exists():
            date, name = parse_save_file(save_path)
            mtime = save_path.stat().st_mtime_ns
        else:
            mtime, date, name = 0, '-'*19, '<Empty>'

        mtimes.append(mtime)
        dates.append(date)
        names.append(name)
        paths.append(save_path)
    return sorted(zip(mtimes, dates, names, paths))


def monitor_profile_dir(profile_path: Path, saves: list[tuple[int, str, str, Path]]) -> None:
    autosave_path = profile_path / AUTOSAVE_NAME

    old_autosave_mtime = [mtime for mtime, date, name, path in saves if path.name == AUTOSAVE_NAME]
    old_autosave_mtime = old_autosave_mtime[0] if old_autosave_mtime else 0
    next_save_time = datetime.now().astimezone() + timedelta(minutes=SAVE_TIME_DELTA_MINUTES)
    if SAVE_TIME_MINUTE_ALIGNED and SAVE_TIME_DELTA_MINUTES > 0:
        next_save_time -= timedelta(seconds=next_save_time.second, microseconds=next_save_time.microsecond)

    if not autosave_path.exists():
        print('No autosave found, waiting till one appears...')
        print('<Press Ctrl+C to quit>')
    while not autosave_path.exists():
        sleep(SLEEP_DURATION_SEC)

    print('AutoSave monitoring started...')
    print('<Press Ctrl+C to quit>')
    while True:
        sleep(SLEEP_DURATION_SEC)

        if autosave_path.stat().st_mtime_ns == old_autosave_mtime:
            continue

        if datetime.now().astimezone() < next_save_time:
            old_autosave_mtime = autosave_path.stat().st_mtime_ns
            continue

        _, old_date, old_name, old_path = saves[0]
        date, name = parse_save_file(autosave_path)

        print(f'Copying: {autosave_path.name[-2:]} | {date} | {name}')
        print(f'To slot: {old_path.name[-2:]} | {old_date} | {old_name}')
        print()
        copy2(autosave_path, old_path)

        saves = parse_all_saves(profile_path)
        old_autosave_mtime = autosave_path.stat().st_mtime_ns
        next_save_time = datetime.now().astimezone() + timedelta(minutes=SAVE_TIME_DELTA_MINUTES)
        if SAVE_TIME_MINUTE_ALIGNED and SAVE_TIME_DELTA_MINUTES > 0:
            next_save_time -= timedelta(seconds=next_save_time.second, microseconds=next_save_time.microsecond)


def main() -> None:
    if not isinstance(MANUAL_SLOT_OVERWRITE_COUNT, int) or not 1 <= MANUAL_SLOT_OVERWRITE_COUNT <= 15:
        print('MANUAL_SLOT_OVERWRITE_COUNT must be an integer between 1 and 15 (inclusive).')
        exit_wait_nt(1)

    for path in CUSTOM_PROFILE_PATHS:
        if check_is_profile_dir(path):
            profile_path = path
            print('Profile path:')
            print(' ', profile_path)
            break
    else:
        if not (profile_path := find_profile_path_from_defaults()):
            print('ERROR: Could not find profile directory in default paths.')
            print('       If your profile folder is in a default location please submit a bug report to have it added.')
            profile_path = input('Enter profile path: ')
            if check_is_profile_dir(Path(profile_path)):
                print('To skip this in future, add:')
                print(' ', profile_path)
                print('To the CUSTOM_PROFILE_PATHS variable at the top of this file.')
            else:
                print('ERROR: Could not get a profile path, quitting...')
                exit_wait_nt(1)

    saves = parse_all_saves(profile_path)
    if saves:
        print(f'AutoSave slot layout ({len(saves)} slots) (oldest first/overwrite order):')
        for _, date, name, path in saves:
            print(f'{path.name[-2:]} | {date} | {name}')

    monitor_profile_dir(profile_path, saves)


if __name__ == '__main__':
    try:
        main()
    except KeyboardInterrupt:
        print('\nGot KbdInterrupt, quitting...')
