#pragma once





namespace CassiopeiaEvents
{
	class CassiopeiaEventManager
	{
	public:
		struct EventData
		{
			struct RegScript
			{
				std::string               scriptName;
				std::vector<RE::TESForm*> filteredForms;
			};
			std::string            eventName;
			std::vector<RegScript> registeredScripts;
		};
		std::multimap<std::string, std::vector<EventData::RegScript>> eventsMap;
		std::vector<std::string>                                      allEventNames;


		inline bool IsRegistered(std::string s, std::string e, bool bSuppressLog = false)
		{
			if (s.empty()) {
				if (!bSuppressLog)
					LogError("scriptName empty");
				return false;
			}
			if (e.empty()) {
				if (!bSuppressLog)
					LogError("eventName empty");
				return false;
			}
			for (auto& e_it : eventsMap) {
				if (e_it.first == e) {
					for (auto& s_it : e_it.second) {
						if (s_it.scriptName == s) {
							if (!bSuppressLog)
								LogInfo("script '" + s + "' is registered for '" + e + "'");
							return true;
						}
					}
				}
			}
			if (!bSuppressLog)
				LogInfo("script '" + s + "' is not registered for '" + e + "'");
			return false;
		}

		inline bool Register(std::string s, std::string e, std::vector<RE::TESForm*> f)
		{
			if (s.empty()) {
				LogError("scriptName empty");
				return false;
			}
			if (e.empty()) {
				LogError("eventName empty");
				return false;
			}
			if (IsRegistered(s, e, true)) {
				LogError("script '" + s + "' is already registered for '" + e + "', cannot register, dropping function call");
				return false;
			}
			for (auto& e_it : eventsMap) {
				if (e_it.first == e) {
					EventData::RegScript r;
					r.filteredForms = f;
					r.scriptName = s;
					e_it.second.push_back(r);
					LogInfo("event '" + e + "' is already in eventMap, registering script '" + s + "'");
					return true;
				}
			}
			EventData::RegScript r;
			r.filteredForms = f;
			r.scriptName = s;
			std::vector<EventData::RegScript> d = { r };
			eventsMap.emplace(e, d);
			LogInfo("event '" + e + "' wasn't in eventMap, adding and registering script '" + s + "'");
			return true;
		}

		inline bool Unregister(std::string s, std::string e)
		{
			if (s.empty()) {
				LogError("scriptName empty");
				return false;
			}
			if (e.empty()) {
				LogError("eventName empty");
				return false;
			}
			if (!IsRegistered(s, e, true)) {
				LogError("script '" + s + "' is not registered for '" + e + "', cannot unregister, dropping function call");
				return false;
			}
			bool bRemoved = false;
			for (auto e_it = eventsMap.begin(); !bRemoved && e_it != eventsMap.end(); ++e_it) {
				if (e_it.operator*().first == e) {
					for (auto& s_it : e_it.operator*().second) {
						if (s_it.scriptName == s) {
							LogInfo("found script '" + s + "' on event '" + e + "', unregistering script");
							e_it = eventsMap.erase(e_it);
							bRemoved = true;
							break;
						}
					}
				}
			}
			if (!bRemoved)
				LogError("could not found script '" + s + "' on event '" + e + "', cannot unregister");
			return bRemoved;
		}

		inline bool UnregisterForAll(std::string s)
		{
			if (s.empty()) {
				LogError("scriptName empty");
				return false;
			}
			bool bRemoved = false;
			for (auto e_it = eventsMap.begin(); !bRemoved && e_it != eventsMap.end(); ++e_it) {
				for (auto& s_it : e_it.operator*().second) {
					if (s_it.scriptName == s) {
						LogInfo("found script '" + s + "' on event '" + e_it.operator*().first + "', unregistering script");
						e_it = eventsMap.erase(e_it);
						bRemoved = true;
						break;
					}
				}
			}
			if (!bRemoved)
				LogError("could not found script '" + s + "' any event, cannot unregister");
			return bRemoved;
		}

		inline bool AddFilter(std::string s, std::string e, std::vector<RE::TESForm*> f) {
			if (s.empty()) {
				LogError("scriptName empty");
				return false;
			}
			if (e.empty()) {
				LogError("eventName empty");
				return false;
			}
			if (f.size() == 0) {
				LogError("filter is empty");
				return false;
			}				
			if (!IsRegistered(s, e, true)) {
				LogError("script '" + s + "' is not registered for '" + e + "', cannot add filter, dropping function call");
				return false;
			}
			bool bAdded = false;
			for (auto& e_it : eventsMap) {
				if (e_it.first == e) {
					for (auto& s_it : e_it.second) {
						if (s_it.scriptName == s) {
							for (auto& form : f) {
								if (!form)
									continue;
								if (std::find(std::begin(s_it.filteredForms), std::end(s_it.filteredForms), form) == s_it.filteredForms.end()) {
									LogInfo("adding filter form " + GetBasicFormData(form, true, false, true) + " to script '" + s + "' on event '" + e + "'");
									bAdded = true;
									s_it.filteredForms.push_back(form);
								}								
							}
						}
					}
				}
			}
			if (!bAdded)
				LogError("could not add any filter form for script '" + s + "' on event '" + e + "', maybe the script for already filtered for the specified forms");
			return bAdded;
		}

		inline bool RemoveFilter(std::string s, std::string e, std::vector<RE::TESForm*> f)
		{
			if (s.empty()) {
				LogError("scriptName empty");
				return false;
			}
			if (e.empty()) {
				LogError("eventName empty");
				return false;
			}
			if (f.size() == 0) {
				LogError("filter is empty");
				return false;
			}
			if (!IsRegistered(s, e, true)) {
				LogError("script '" + s + "' is not registered for '" + e + "', cannot remove filter, dropping function call");
				return false;
			}
			bool bRemoved = false;
			for (auto& e_it : eventsMap) {
				if (e_it.first == e) {
					for (auto& s_it : e_it.second) {
						if (s_it.scriptName == s) {
							for (auto f_it = s_it.filteredForms.begin(); f_it != s_it.filteredForms.end();) {
								if (std::find(std::begin(f), std::end(f), f_it.operator*()) < f.end()) {
									LogInfo("removing filter form " + GetBasicFormData(f_it.operator*(), true, false, true) + " from script '" + s + "' on event '" + e + "'");
									f_it = s_it.filteredForms.erase(f_it);
									bRemoved = true;
								} else {
									++f_it;
								}									
							}
						}
					}
				}
			}
			if (!bRemoved)
				LogError("could not remove any filter form for script '" + s + "' on event '" + e + "', maybe the script wasn't filtered for the specified forms");
			return bRemoved;
		}

		inline std::vector<RE::TESForm*> GetFilter(std::string s, std::string e)
		{
			std::vector<RE::TESForm*> Result;
			if (s.empty()) {
				LogError("scriptName empty");
				return Result;
			}
			if (e.empty()) {
				LogError("eventName empty");
				return Result;
			}
			if (!IsRegistered(s, e, true)) {
				LogError("script '" + s + "' is not registered for '" + e + "', cannot get filter, dropping function call");
				return Result;
			}
			for (auto& e_it : eventsMap) {
				if (e_it.first == e) {
					for (auto& s_it : e_it.second) {
						if (s_it.scriptName == s) {
							return s_it.filteredForms;
						}
					}
				}
			}
			LogError("could not find filter for script '" + s + "' on event '" + e + "'");
			return Result;
		}

		template <class... Args>
		inline int Dispatch(std::string e, std::vector<RE::TESForm*>* filter = nullptr, Args&&... a_args)
		{
			if (e.empty()) {
				LogError("eventName empty");
				return -1;
			}
			e = ToLowerStr(e);
			auto vm = RE::GameVMCustom::GetSingleton()->virtualMechine;
			if (!vm) {
				LogError("vm is nullptr");
				return -1;
			}
			int Result = 0;
			for (auto& e_it : eventsMap) {
				if (e_it.first == e) {
					for (auto& s_it : e_it.second) {
						if (s_it.scriptName.empty())
							continue;
						if (filter && filter->size() > 0 && s_it.filteredForms.size() > 0) {		// filtering a script is not mandatory, all non filtered but registered scripts will receive the events
							bool bCanProceed = false;
							for (size_t j = 0; j < filter->size(); j++) {
								if (!filter->at(j) || std::find(std::begin(s_it.filteredForms), std::end(s_it.filteredForms), filter->at(j)) < s_it.filteredForms.end()) {
									bCanProceed = true;
									break;
								}
							}
							if (!bCanProceed)
								continue;
						}
						LogInfo("sending event '" + e + "' for script '" + s_it.scriptName + "'");
						vm->DispatchStaticCall(s_it.scriptName, e, nullptr, 0, std::forward<Args>(a_args)...);
						Result = Result + 1;
					}
				}
			}
			return Result;
		}

		inline int BuildAllEventNameData() {
			if (allEventNames.size() > 0)
				allEventNames.clear();
			allEventNames.push_back(ToLowerStr("TESLoadGameEvent"));
			allEventNames.push_back(ToLowerStr("TESPlayerFollowerWarpEvent"));
			allEventNames.push_back(ToLowerStr("DataMenu_Reopened"));
			allEventNames.push_back(ToLowerStr("PhotoMode_RefineSetting"));
			allEventNames.push_back(ToLowerStr("PhotoMode_StepperChanged"));
			allEventNames.push_back(ToLowerStr("PlayerSetWeaponStateEvent"));
			allEventNames.push_back(ToLowerStr("TESEquipEvent"));
			allEventNames.push_back(ToLowerStr("ReferenceDetach"));
			allEventNames.push_back(ToLowerStr("ReferenceSet3d"));
			allEventNames.push_back(ToLowerStr("PlayerFastTravel"));
			allEventNames.push_back(ToLowerStr("PlayerSneakingChangeEvent"));
			allEventNames.push_back(ToLowerStr("TESCombatEvent"));
			allEventNames.push_back(ToLowerStr("UpdateActivateListenerEvent"));
			allEventNames.push_back(ToLowerStr("HUDModeEvent"));
			allEventNames.push_back(ToLowerStr("FirstThirdPersonSwitch"));
			allEventNames.push_back(ToLowerStr("HideSubtitleEvent"));
			allEventNames.push_back(ToLowerStr("ShowSubtitleEvent"));
			allEventNames.push_back(ToLowerStr("LocationTextEvent"));
			allEventNames.push_back(ToLowerStr("TraitDiscoveryTextEvent"));
			allEventNames.push_back(ToLowerStr("TESHarvestEvent"));
			allEventNames.push_back(ToLowerStr("HUDNotification_MissionActiveWidgetUpdate"));
			allEventNames.push_back(ToLowerStr("LevelUp_OnWidgetShown"));
			allEventNames.push_back(ToLowerStr("ShipEditor_SystemSelected"));
			allEventNames.push_back(ToLowerStr("CellAttachDetachEvent"));
			allEventNames.push_back(ToLowerStr("PlayerKnowledgeFlagSetEvent"));
			allEventNames.push_back(ToLowerStr("PlanetTraitKnownEvent"));
			allEventNames.push_back(ToLowerStr("BGSSceneActionPlayerDialogue_ActionEndEvent"));
			allEventNames.push_back(ToLowerStr("AutoLoadDoorRolloverEvent"));
			allEventNames.push_back(ToLowerStr("ClearQuickContainerEvent"));
			allEventNames.push_back(ToLowerStr("PickRefUpdateEvent"));
			allEventNames.push_back(ToLowerStr("Workshop_CargoLinkAddedEvent"));
			allEventNames.push_back(ToLowerStr("Workshop_CargoLinkTargetChangedEvent"));
			allEventNames.push_back(ToLowerStr("Workshop_EnterOutpostBeaconModeEvent"));
			allEventNames.push_back(ToLowerStr("Workshop_ItemGrabbedEvent"));
			allEventNames.push_back(ToLowerStr("Workshop_ItemMovedEvent"));
			allEventNames.push_back(ToLowerStr("Workshop_ItemPlacedEvent"));
			allEventNames.push_back(ToLowerStr("Workshop_ItemProducedEvent"));
			allEventNames.push_back(ToLowerStr("Workshop_ItemRemovedEvent"));
			allEventNames.push_back(ToLowerStr("Workshop_ItemRepairedEvent"));
			allEventNames.push_back(ToLowerStr("Workshop_ItemScrappedEvent"));
			allEventNames.push_back(ToLowerStr("Workshop_OutpostNameChangedEvent"));
			allEventNames.push_back(ToLowerStr("Workshop_OutpostPlacedEvent"));
			allEventNames.push_back(ToLowerStr("Workshop_PlacementStatusEvent"));
			allEventNames.push_back(ToLowerStr("Workshop_PowerOffEvent"));
			allEventNames.push_back(ToLowerStr("Workshop_PowerOnEvent"));
			allEventNames.push_back(ToLowerStr("Workshop_SnapBehaviorCycledEvent"));
			allEventNames.push_back(ToLowerStr("Workshop_WorkshopFlyCameraEvent"));
			allEventNames.push_back(ToLowerStr("Workshop_WorkshopItemPlacedEvent"));
			allEventNames.push_back(ToLowerStr("Workshop_WorkshopModeEvent"));
			allEventNames.push_back(ToLowerStr("Workshop_WorkshopOutputLinkEvent"));
			allEventNames.push_back(ToLowerStr("Workshop_WorkshopStatsChangedEvent"));
			allEventNames.push_back(ToLowerStr("Workshop_WorkshopUpdateStatsEvent"));
			allEventNames.push_back(ToLowerStr("ShipCameraStateToggled"));
			allEventNames.push_back(ToLowerStr("PlayerJumpReleaseEvent"));
			allEventNames.push_back(ToLowerStr("PlayerJumpPressEvent"));
			allEventNames.push_back(ToLowerStr("PlayerZeroGSprintJustPressedEvent"));
			allEventNames.push_back(ToLowerStr("PlayerIronSightsEndEvent"));
			allEventNames.push_back(ToLowerStr("PlayerIronSightsStartEvent"));
			allEventNames.push_back(ToLowerStr("PlayerZeroGSprintReleasedEvent"));
			allEventNames.push_back(ToLowerStr("SpaceshipWeaponBindingChangedEvent"));
			allEventNames.push_back(ToLowerStr("PhotoGallery_DeletePhoto"));
			allEventNames.push_back(ToLowerStr("PowersMenu_ActivateEssence"));
			allEventNames.push_back(ToLowerStr("PowersMenu_EquipPower"));
			allEventNames.push_back(ToLowerStr("PowersMenu_FavoritePower"));
			allEventNames.push_back(ToLowerStr("ContainerMenuClosed"));
			allEventNames.push_back(ToLowerStr("BinkMovieStoppedPlayingEvent"));
			allEventNames.push_back(ToLowerStr("CraftingMenu_CraftItem"));
			allEventNames.push_back(ToLowerStr("CraftingMenu_ExitBench"));
			allEventNames.push_back(ToLowerStr("CraftingMenu_InstallMod"));
			allEventNames.push_back(ToLowerStr("CraftingMenu_RenameItem"));
			allEventNames.push_back(ToLowerStr("CraftingMenu_RevertedModdedItem"));
			allEventNames.push_back(ToLowerStr("CraftingMenu_SelectedMod"));
			allEventNames.push_back(ToLowerStr("CraftingMenu_SelectedModSlot"));
			allEventNames.push_back(ToLowerStr("CraftingMenu_SelectedModdableItem"));
			allEventNames.push_back(ToLowerStr("CraftingMenu_SelectedRecipe"));
			allEventNames.push_back(ToLowerStr("CraftingMenu_ToggleTracking"));
			allEventNames.push_back(ToLowerStr("CraftingMenu_ViewingModdableItem"));
			allEventNames.push_back(ToLowerStr("CraftingMenu_CloseMenu"));
			allEventNames.push_back(ToLowerStr("InventoryMenu_Change3DView"));
			allEventNames.push_back(ToLowerStr("InventoryMenu_PaperDollTryOn"));
			allEventNames.push_back(ToLowerStr("InventoryMenu_ResetPaperDollInv"));
			allEventNames.push_back(ToLowerStr("ShowingQuestMarketTextEvent"));
			allEventNames.push_back(ToLowerStr("TargetHitEvent"));
			allEventNames.push_back(ToLowerStr("ClearHUDMessagesEvent"));
			allEventNames.push_back(ToLowerStr("ShowHUDMessageEvent"));
			allEventNames.push_back(ToLowerStr("PlayerDetectionLevelChangeEvent"));
			allEventNames.push_back(ToLowerStr("InventoryMenu_ToggleHelmet"));
			allEventNames.push_back(ToLowerStr("InventoryMenu_ToggleSuit"));
			allEventNames.push_back(ToLowerStr("ControlsRemappedEvent"));
			allEventNames.push_back(ToLowerStr("ShowLongShipBootup"));
			allEventNames.push_back(ToLowerStr("ShipHudQuickContainer_TransferMenu"));
			allEventNames.push_back(ToLowerStr("ShipHud_Activate"));
			allEventNames.push_back(ToLowerStr("ShipHud_BodyViewMarkerDimensions"));
			allEventNames.push_back(ToLowerStr("ShipHud_ChangeComponentSelection"));
			allEventNames.push_back(ToLowerStr("ShipHud_Deselect"));
			allEventNames.push_back(ToLowerStr("ShipHud_FarTravel"));
			allEventNames.push_back(ToLowerStr("ShipHud_HailAccepted"));
			allEventNames.push_back(ToLowerStr("ShipHud_HailCancelled"));
			allEventNames.push_back(ToLowerStr("ShipHud_JumpToQuestMarker"));
			allEventNames.push_back(ToLowerStr("ShipHud_Land"));
			allEventNames.push_back(ToLowerStr("ShipHud_LandingMarkerMap"));
			allEventNames.push_back(ToLowerStr("ShipHud_Map"));
			allEventNames.push_back(ToLowerStr("ShipHud_OpenPhotoMode"));
			allEventNames.push_back(ToLowerStr("ShipHud_Repair"));
			allEventNames.push_back(ToLowerStr("ShipHud_SetTargetMode"));
			allEventNames.push_back(ToLowerStr("ShipHud_Target"));
			allEventNames.push_back(ToLowerStr("ShipHud_TargetShipSystem"));
			allEventNames.push_back(ToLowerStr("ShipHud_UntargetShipSystem"));
			allEventNames.push_back(ToLowerStr("ShipHud_UpdateComponentPower"));
			allEventNames.push_back(ToLowerStr("BGSScannerGuideEffectStatusUpdateEvent"));
			allEventNames.push_back(ToLowerStr("ExperienceMeterDisplayData"));
			allEventNames.push_back(ToLowerStr("LevelUp_AnimFinished"));
			allEventNames.push_back(ToLowerStr("LocationTextWidget_FinishedQueue"));
			allEventNames.push_back(ToLowerStr("HUDNotificationEvent"));
			allEventNames.push_back(ToLowerStr("HUDNotification_SetMissionActive"));
			allEventNames.push_back(ToLowerStr("BarterMenu_BuyItem"));
			allEventNames.push_back(ToLowerStr("BarterMenu_CloseMenu"));
			allEventNames.push_back(ToLowerStr("BarterMenu_HideModel"));
			allEventNames.push_back(ToLowerStr("BarterMenu_LoadModel"));
			allEventNames.push_back(ToLowerStr("BarterMenu_SellItem"));
			allEventNames.push_back(ToLowerStr("BarterMenu_SetMouseOverModel"));
			allEventNames.push_back(ToLowerStr("BarterMenu_ViewedItem"));
			allEventNames.push_back(ToLowerStr("MissionMenu_ClearState"));
			allEventNames.push_back(ToLowerStr("MissionMenu_PlotToLocation"));
			allEventNames.push_back(ToLowerStr("MissionMenu_RejectQuest"));
			allEventNames.push_back(ToLowerStr("MissionMenu_ShowItemLocation"));
			allEventNames.push_back(ToLowerStr("MissionMenu_ToggleTrackingQuest"));
			allEventNames.push_back(ToLowerStr("DataSlateMenu_playSFX"));
			allEventNames.push_back(ToLowerStr("DataSlateMenu_toggleAudio"));
			allEventNames.push_back(ToLowerStr("InventoryMenu_HideModel"));
			allEventNames.push_back(ToLowerStr("InventoryMenu_LoadModel"));
			allEventNames.push_back(ToLowerStr("InventoryMenu_OnEnterCategory"));
			allEventNames.push_back(ToLowerStr("InventoryMenu_OpenCargoHold"));
			allEventNames.push_back(ToLowerStr("InventoryMenu_SelectItem"));
			allEventNames.push_back(ToLowerStr("InventoryMenu_SetMouseOverModel"));
			allEventNames.push_back(ToLowerStr("InventoryMenu_ToggleFavorite"));
			allEventNames.push_back(ToLowerStr("MissionBoard_MissionEntryChanged"));
			allEventNames.push_back(ToLowerStr("MissionBoard_MissionEntryPressed"));
			allEventNames.push_back(ToLowerStr("MonocleMenu_Bioscan"));
			allEventNames.push_back(ToLowerStr("MonocleMenu_FastTravel"));
			allEventNames.push_back(ToLowerStr("MonocleMenu_Harvest"));
			allEventNames.push_back(ToLowerStr("MonocleMenu_Outpost"));
			allEventNames.push_back(ToLowerStr("MonocleMenu_SocialSpell"));
			allEventNames.push_back(ToLowerStr("MonocleMenu_SurfaceMap"));
			allEventNames.push_back(ToLowerStr("MonocleMenu_ZoomIn"));
			allEventNames.push_back(ToLowerStr("MonocleMenu_ZoomOut"));
			allEventNames.push_back(ToLowerStr("PhotoMode_InitializeCategory"));
			allEventNames.push_back(ToLowerStr("PhotoMode_ResetToDefaults"));
			allEventNames.push_back(ToLowerStr("PhotoMode_SliderChanged"));
			allEventNames.push_back(ToLowerStr("PhotoMode_TakeSnapshot"));
			allEventNames.push_back(ToLowerStr("PhotoMode_ToggleHelmet"));
			allEventNames.push_back(ToLowerStr("PhotoMode_ToggleUI"));
			allEventNames.push_back(ToLowerStr("PickpocketMenu_OnItemSelect"));
			allEventNames.push_back(ToLowerStr("ResearchMenu_AddMaterial"));
			allEventNames.push_back(ToLowerStr("ResearchMenu_CategorySelected"));
			allEventNames.push_back(ToLowerStr("ResearchMenu_HideModel"));
			allEventNames.push_back(ToLowerStr("ResearchMenu_PreviewProject"));
			allEventNames.push_back(ToLowerStr("ResearchMenu_ProjectViewed"));
			allEventNames.push_back(ToLowerStr("ResearchMenu_ToggleTrackingProject"));
			allEventNames.push_back(ToLowerStr("UnlockedTerminalElementEvent"));
			allEventNames.push_back(ToLowerStr("SecurityMenu_BackOutKey"));
			allEventNames.push_back(ToLowerStr("SecurityMenu_ConfirmExit"));
			allEventNames.push_back(ToLowerStr("SecurityMenu_EliminateUnusedKeys"));
			allEventNames.push_back(ToLowerStr("SecurityMenu_GetRingHint"));
			allEventNames.push_back(ToLowerStr("SecurityMenu_SelectNewKey"));
			allEventNames.push_back(ToLowerStr("SecurityMenu_TryUseKey"));
			allEventNames.push_back(ToLowerStr("Refuel_Accept"));
			allEventNames.push_back(ToLowerStr("Refuel_Cancel"));
			allEventNames.push_back(ToLowerStr("SkillsMenu_Accept"));
			allEventNames.push_back(ToLowerStr("CharGen_PresetNPCChangedEvent"));
			allEventNames.push_back(ToLowerStr("CharGen_BrowChange"));
			allEventNames.push_back(ToLowerStr("CharGen_BrowColorChange"));
			allEventNames.push_back(ToLowerStr("CharGen_CancelTextEntry"));
			allEventNames.push_back(ToLowerStr("CharGen_CloseMenu"));
			allEventNames.push_back(ToLowerStr("CharGen_CyclePronoun"));
			allEventNames.push_back(ToLowerStr("CharGen_DirtScarsEtcChange"));
			allEventNames.push_back(ToLowerStr("CharGen_EndBodyChange"));
			allEventNames.push_back(ToLowerStr("CharGen_EndTextEntry"));
			allEventNames.push_back(ToLowerStr("CharGen_EyeColorChange"));
			allEventNames.push_back(ToLowerStr("CharGen_FacialHairChange"));
			allEventNames.push_back(ToLowerStr("CharGen_FacialHairColorChange"));
			allEventNames.push_back(ToLowerStr("CharGen_HairChange"));
			allEventNames.push_back(ToLowerStr("CharGen_HairColorChange"));
			allEventNames.push_back(ToLowerStr("CharGen_HeadpartPlusSelectorChange"));
			allEventNames.push_back(ToLowerStr("CharGen_HeadpartPresetChange"));
			allEventNames.push_back(ToLowerStr("CharGen_JewelryChange"));
			allEventNames.push_back(ToLowerStr("CharGen_JewelryColorChange"));
			allEventNames.push_back(ToLowerStr("CharGen_MakeupChange"));
			allEventNames.push_back(ToLowerStr("CharGen_MarkingsChange"));
			allEventNames.push_back(ToLowerStr("CharGen_PostBlendColorOptionChange"));
			allEventNames.push_back(ToLowerStr("CharGen_PostBlendFaceChange"));
			allEventNames.push_back(ToLowerStr("CharGen_PostBlendIntensityChange"));
			allEventNames.push_back(ToLowerStr("CharGen_PresetChange"));
			allEventNames.push_back(ToLowerStr("CharGen_RotatePaperdoll"));
			allEventNames.push_back(ToLowerStr("CharGen_SetAdditionalSlider"));
			allEventNames.push_back(ToLowerStr("CharGen_SetBackground"));
			allEventNames.push_back(ToLowerStr("CharGen_SetBodyValues"));
			allEventNames.push_back(ToLowerStr("CharGen_SetCameraPosition"));
			allEventNames.push_back(ToLowerStr("CharGen_SetPronoun"));
			allEventNames.push_back(ToLowerStr("CharGen_SetSex"));
			allEventNames.push_back(ToLowerStr("CharGen_SetSlider"));
			allEventNames.push_back(ToLowerStr("CharGen_SetTrait"));
			allEventNames.push_back(ToLowerStr("CharGen_ShowChooseBackgroundMessage"));
			allEventNames.push_back(ToLowerStr("CharGen_ShowPlayerRenameMessage"));
			allEventNames.push_back(ToLowerStr("CharGen_SkintoneChange"));
			allEventNames.push_back(ToLowerStr("CharGen_StartBodyChange"));
			allEventNames.push_back(ToLowerStr("CharGen_StartTextEntry"));
			allEventNames.push_back(ToLowerStr("CharGen_SwitchBodyType"));
			allEventNames.push_back(ToLowerStr("CharGen_TeethChange"));
			allEventNames.push_back(ToLowerStr("CharGen_TeethRollOff"));
			allEventNames.push_back(ToLowerStr("CharGen_TeethRollOn"));
			allEventNames.push_back(ToLowerStr("CharGen_ToggleMarking"));
			allEventNames.push_back(ToLowerStr("CharGen_TogglePreviewHabSuit"));
			allEventNames.push_back(ToLowerStr("UIMenuChargenMenuDisablePaperdoll"));
			allEventNames.push_back(ToLowerStr("DataMenu_Missions"));
			allEventNames.push_back(ToLowerStr("DataMenu_PlotToLocation"));
			allEventNames.push_back(ToLowerStr("DataMenu_SetPaperDollActive"));
			allEventNames.push_back(ToLowerStr("PauseMenu_ActionCanceled"));
			allEventNames.push_back(ToLowerStr("PauseMenu_ConfirmAction"));
			allEventNames.push_back(ToLowerStr("PauseMenu_ConfirmLoad"));
			allEventNames.push_back(ToLowerStr("PauseMenu_ConfirmSave"));
			allEventNames.push_back(ToLowerStr("PauseMenu_DeleteSave"));
			allEventNames.push_back(ToLowerStr("PauseMenu_QuitToDesktop"));
			allEventNames.push_back(ToLowerStr("PauseMenu_SetCharacter"));
			allEventNames.push_back(ToLowerStr("PauseMenu_StartAction"));
			allEventNames.push_back(ToLowerStr("PauseMenu_StartLoad"));
			allEventNames.push_back(ToLowerStr("PlayBink_CloseMenu"));
			allEventNames.push_back(ToLowerStr("Reticle_OnLongAnimFinished"));
			allEventNames.push_back(ToLowerStr("ShipHudQuickContainer_TransferItem"));
			allEventNames.push_back(ToLowerStr("ShipHud_AbortJump"));
			allEventNames.push_back(ToLowerStr("ShipHud_DockRequested"));
			allEventNames.push_back(ToLowerStr("ShipHud_HailShip"));
			allEventNames.push_back(ToLowerStr("ShipHud_UpdateTargetPanelRect"));
			allEventNames.push_back(ToLowerStr("TakeoffMenu_ExitShip"));
			allEventNames.push_back(ToLowerStr("TakeoffMenu_Launch"));
			allEventNames.push_back(ToLowerStr("StarMapMenu_LandingInputInProgress"));
			allEventNames.push_back(ToLowerStr("StarMapMenu_MarkerGroupContainerVisibilityChanged"));
			allEventNames.push_back(ToLowerStr("StarMapMenu_MarkerGroupEntryClicked"));
			allEventNames.push_back(ToLowerStr("StarMapMenu_MarkerGroupEntryHoverChanged"));
			allEventNames.push_back(ToLowerStr("StarMapMenu_ScanPlanet"));
			allEventNames.push_back(ToLowerStr("StarMapMenu_SelectedLandingSite"));
			allEventNames.push_back(ToLowerStr("StarMapMenu_SelectedLandingSiteFailed"));
			allEventNames.push_back(ToLowerStr("StarMapMenu_ShowRealCursor"));
			allEventNames.push_back(ToLowerStr("StarMapMenu_QuickSelectChange"));
			allEventNames.push_back(ToLowerStr("StarMapMenu_Galaxy_FocusSystem"));
			allEventNames.push_back(ToLowerStr("StarMapMenu_OnGalaxyViewInitialized"));
			allEventNames.push_back(ToLowerStr("StarMapMenu_ExecuteRoute"));
			allEventNames.push_back(ToLowerStr("StarMapMenu_OnCancel"));
			allEventNames.push_back(ToLowerStr("StarMapMenu_OnClearRoute"));
			allEventNames.push_back(ToLowerStr("StarMapMenu_OnExitStarMap"));
			allEventNames.push_back(ToLowerStr("StarMapMenu_OnHintButtonClicked"));
			allEventNames.push_back(ToLowerStr("StarMapMenu_OnOutpostEntrySelected"));
			allEventNames.push_back(ToLowerStr("StarMapMenu_ReadyToClose"));
			allEventNames.push_back(ToLowerStr("SurfaceMapMenu_MarkerClicked"));
			allEventNames.push_back(ToLowerStr("SurfaceMapMenu_TryPlaceCustomMarker"));
			allEventNames.push_back(ToLowerStr("TerminalMenu_CancelEvent"));
			allEventNames.push_back(ToLowerStr("Terminal_CloseAllViews"));
			allEventNames.push_back(ToLowerStr("Terminal_CloseTopView"));
			allEventNames.push_back(ToLowerStr("Terminal_CloseView"));
			allEventNames.push_back(ToLowerStr("CraftingMenu_Highlight3D"));
			allEventNames.push_back(ToLowerStr("CraftingMenu_RevertHighlight"));
			allEventNames.push_back(ToLowerStr("WorkshopBuilderMenu_ChangeBuildItem"));
			allEventNames.push_back(ToLowerStr("WorkshopBuilderMenu_SelectedItem"));
			allEventNames.push_back(ToLowerStr("WorkshopBuilderMenu_ToggleTracking"));
			allEventNames.push_back(ToLowerStr("WorkshopMenu_AttemptBuild"));
			allEventNames.push_back(ToLowerStr("WorkshopMenu_ChangeVariant"));
			allEventNames.push_back(ToLowerStr("WorkshopMenu_ConnectionEvent"));
			allEventNames.push_back(ToLowerStr("WorkshopMenu_SelectedCategory"));
			allEventNames.push_back(ToLowerStr("WorkshopMenu_SelectedGridObject"));
			allEventNames.push_back(ToLowerStr("WorkshopMenu_ShowExtras"));
			allEventNames.push_back(ToLowerStr("WorkshopMenu_SwitchMode"));
			allEventNames.push_back(ToLowerStr("WorkshopMenu_ToggleDistance"));
			allEventNames.push_back(ToLowerStr("WorkshopMenu_ToggleTracking"));
			allEventNames.push_back(ToLowerStr("WorkshopMenu_ToggleView"));
			allEventNames.push_back(ToLowerStr("ModelReferenceEffectEvents_ReferenceEffectFinished"));
			allEventNames.push_back(ToLowerStr("BGSAcousticSpaceListener_StackChangedEvent"));
			allEventNames.push_back(ToLowerStr("SpaceshipBIEvents_ShipPowerAllocationBIEventSent"));
			allEventNames.push_back(ToLowerStr("ResearchProgressEvent"));
			allEventNames.push_back(ToLowerStr("ContainerMenu_HideModel"));
			allEventNames.push_back(ToLowerStr("ContainerMenu_Jettison"));
			allEventNames.push_back(ToLowerStr("ContainerMenu_LoadModel"));
			allEventNames.push_back(ToLowerStr("ContainerMenu_OpenRefuelMenu"));
			allEventNames.push_back(ToLowerStr("ContainerMenu_SetMouseOverModel"));
			allEventNames.push_back(ToLowerStr("ContainerMenu_TakeAll"));
			allEventNames.push_back(ToLowerStr("ContainerMenu_ToggleEquip"));
			allEventNames.push_back(ToLowerStr("ContainerMenu_TransferItem"));
			allEventNames.push_back(ToLowerStr("DialogueMenu_OnDialogueSelect"));
			allEventNames.push_back(ToLowerStr("DialogueMenu_OnListVisibilityChange"));
			allEventNames.push_back(ToLowerStr("DialogueMenu_OnPersuasionAutoWin"));
			allEventNames.push_back(ToLowerStr("DialogueMenu_RequestExit"));
			allEventNames.push_back(ToLowerStr("DialogueMenu_RequestSkipDialogue"));
			allEventNames.push_back(ToLowerStr("FavoritesMenu_AssignQuickkey"));
			allEventNames.push_back(ToLowerStr("FavoritesMenu_UseQuickkey"));
			allEventNames.push_back(ToLowerStr("MessageBoxMenu_OnBackOut"));
			allEventNames.push_back(ToLowerStr("MessageBoxMenu_OnButtonPress"));
			allEventNames.push_back(ToLowerStr("MessageBoxMenu_OnScriptedButtonPress"));
			allEventNames.push_back(ToLowerStr("SleepWaitMenu_InterruptRest"));
			allEventNames.push_back(ToLowerStr("SleepWaitMenu_StartRest"));
			allEventNames.push_back(ToLowerStr("TakeOffEvent"));
			allEventNames.push_back(ToLowerStr("GravJumpEvent"));
			allEventNames.push_back(ToLowerStr("LandedSetEvent"));
			allEventNames.push_back(ToLowerStr("SpaceshipBoughtEvent"));
			allEventNames.push_back(ToLowerStr("ContrabandScanWarningEvent"));
			allEventNames.push_back(ToLowerStr("DockEvent"));
			allEventNames.push_back(ToLowerStr("SpaceshipDynamicNavmeshCompleted"));
			allEventNames.push_back(ToLowerStr("SpaceshipFarTravelEvent"));
			allEventNames.push_back(ToLowerStr("LandingEvent"));
			allEventNames.push_back(ToLowerStr("SpaceshipPlanetScanEvent"));
			allEventNames.push_back(ToLowerStr("SpaceshipRampDownEvent"));
			allEventNames.push_back(ToLowerStr("SpaceshipRefueledEvent"));
			allEventNames.push_back(ToLowerStr("SpaceshipRegisteredEvent"));
			allEventNames.push_back(ToLowerStr("SpaceshipShieldEvent"));
			allEventNames.push_back(ToLowerStr("ShipAddedEvent"));
			allEventNames.push_back(ToLowerStr("ShipCollisionEvent"));
			allEventNames.push_back(ToLowerStr("ShipCustomizedEvent"));
			allEventNames.push_back(ToLowerStr("SpaceshipSoldEvent"));
			allEventNames.push_back(ToLowerStr("SpaceshipSystemDamagedEvent"));
			allEventNames.push_back(ToLowerStr("SpaceshipSystemPowerAllocationEvent"));
			allEventNames.push_back(ToLowerStr("SpaceshipSystemPowerChangeEvent"));
			allEventNames.push_back(ToLowerStr("SpaceshipSystemRepairedBIEvent"));
			allEventNames.push_back(ToLowerStr("SpaceshipSystemRepairedEvent"));
			allEventNames.push_back(ToLowerStr("SpaceshipTakeDamageEvent"));
			allEventNames.push_back(ToLowerStr("UpdateSceneRectEvent"));
			allEventNames.push_back(ToLowerStr("TESCellNavmeshGeneratedEvent"));
			allEventNames.push_back(ToLowerStr("PerkChanged"));
			allEventNames.push_back(ToLowerStr("ActivityCompletedEvent"));
			allEventNames.push_back(ToLowerStr("ChallengeCompletedEvent"));
			allEventNames.push_back(ToLowerStr("InventoryItemEvent"));
			allEventNames.push_back(ToLowerStr("BooksRead"));
			allEventNames.push_back(ToLowerStr("LocationLinked"));
			allEventNames.push_back(ToLowerStr("Activation"));
			allEventNames.push_back(ToLowerStr("ActorCellChangeEvent"));
			allEventNames.push_back(ToLowerStr("BGSAffinityEventEvent"));
			allEventNames.push_back(ToLowerStr("BGSCellGridLoadEvent"));
			allEventNames.push_back(ToLowerStr("BGSRadiationDamageEvent"));
			allEventNames.push_back(ToLowerStr("InstantReferenceInteractionEvent"));
			allEventNames.push_back(ToLowerStr("TESActivateEvent"));
			allEventNames.push_back(ToLowerStr("TESBookReadEvent"));
			allEventNames.push_back(ToLowerStr("BGSLocationLoadedEvent"));
			allEventNames.push_back(ToLowerStr("TESCellFullyLoadedEvent"));
			allEventNames.push_back(ToLowerStr("TESContainerChangedEvent"));
			allEventNames.push_back(ToLowerStr("TESFormDeleteEvent"));
			allEventNames.push_back(ToLowerStr("TESFormIDRemapEvent"));
			allEventNames.push_back(ToLowerStr("TESFurnitureEvent"));
			allEventNames.push_back(ToLowerStr("TESGrabReleaseEvent"));
			allEventNames.push_back(ToLowerStr("TESMissionAcceptedEvent"));
			allEventNames.push_back(ToLowerStr("TESObjectLoadedEvent"));
			allEventNames.push_back(ToLowerStr("TESPickNewIdleEvent"));
			allEventNames.push_back(ToLowerStr("TESResetEvent"));
			allEventNames.push_back(ToLowerStr("TESResolveNPCTemplatesEvent"));
			allEventNames.push_back(ToLowerStr("ClearShipHudTarget"));
			allEventNames.push_back(ToLowerStr("TryUpdateShipHudTarget"));
			allEventNames.push_back(ToLowerStr("ActorItemEquipped"));
			allEventNames.push_back(ToLowerStr("ReloadWeaponEvent"));
			allEventNames.push_back(ToLowerStr("BountyEvent"));
			allEventNames.push_back(ToLowerStr("CriticalHitEvent"));
			allEventNames.push_back(ToLowerStr("CustomMarkerUpdate"));
			allEventNames.push_back(ToLowerStr("EnteredUnity"));
			allEventNames.push_back(ToLowerStr("LevelIncrease"));
			allEventNames.push_back(ToLowerStr("PlayerAmmoChanged"));
			allEventNames.push_back(ToLowerStr("TerminalHacked"));
			allEventNames.push_back(ToLowerStr("MonocleMenu_PhotoMode"));
			allEventNames.push_back(ToLowerStr("BSInputEvent"));
			allEventNames.push_back(ToLowerStr("BSInputEventEx"));	// v2.4
			allEventNames.push_back(ToLowerStr("OnPlayerPlotRoute"));	// v2.6
			return allEventNames.size();
		}

		inline bool IsValidEventName(std::string e)
		{
			return (!e.empty() && std::find(std::begin(allEventNames), std::end(allEventNames), e) < allEventNames.end());
		}

		inline bool ClearAll()
		{
			eventsMap.clear();
			return (eventsMap.size() == 0);
		}

	};
	static CassiopeiaEventManager mgr;
	inline static CassiopeiaEventManager* GetEventManager()
	{
		return &mgr;
	}
	inline static bool ClearAllCassiopeiaEventRegistrations()
	{
		auto mgr = GetEventManager();
		return (mgr && mgr->ClearAll());
	}
	//inline static bool DispatchCassiopeiaEvent(std::string eventName)
	//{
	//	auto mgr = GetEventManager();
	//	return (mgr && mgr->ClearAll());
	//}

}




