#include "PCH.h"

inline static std::vector<RE::BGSMod::Attachment::Mod*> GetInstalledObjectMods(RE::BGSObjectInstanceExtra* objectInstanceExtra, bool bExcludeModColls = true)
{
	LogInfo("Called for instance extra: " + GetAddressAsHex(reinterpret_cast<unsigned long long>(objectInstanceExtra)));
	std::vector<RE::BGSMod::Attachment::Mod*> Mods;
	if (!objectInstanceExtra) {
		return Mods;
	}
	auto ObjectModArray = objectInstanceExtra->mods;
	if (!ObjectModArray)
		return Mods;
	auto ModCount = objectInstanceExtra->modCount;
	if (ModCount <= 0 || ModCount >= UINT32_MAX)
		return Mods;
	auto addr = reinterpret_cast<unsigned long long>(ObjectModArray);
	if (StarfieldBaseAddress <= 0 || addr <= 10000000000 || addr >= StarfieldBaseAddress) {
		return Mods;
	}
	int iPad = 0x10;
	int iCurrPad = 0;
	for (uint32_t i = 0; i < ModCount; i++) {
		auto loopAddr = addr + iCurrPad;
		iCurrPad = iCurrPad + iPad;
		if (loopAddr <= 10000000000 || loopAddr >= StarfieldBaseAddress)
			continue;
		auto loopOmodAddrPtr = reinterpret_cast<unsigned long long*>(loopAddr);
		if (!loopOmodAddrPtr)
			continue;
		unsigned long long loopOmodAddr = 0;
		try {
			loopOmodAddr = *loopOmodAddrPtr;
		} catch (...) {
			LogError("nullptr exception at: " + GetAddressAsHex(loopAddr));
			loopOmodAddr = 0;
		}
		if (loopOmodAddr <= 10000000000 || loopOmodAddr >= StarfieldBaseAddress)
			continue;
		auto loopOmod = reinterpret_cast<RE::BGSMod::Attachment::Mod*>(loopOmodAddr);
		if (!loopOmod)
			continue;
		if (loopOmod->formID <= 0 || loopOmod->formID > UINT32_MAX || loopOmod->GetSavedFormType() != 152)
			continue;
		auto formFlags = loopOmod->formFlags & 0xFF;
		if (formFlags == 137 && bExcludeModColls)
			continue;
		LogInfo("Adding ObjectMod " + GetBasicFormData(loopOmod, false, false, true));
		Mods.push_back(loopOmod);
	}
	if (Mods.size() > 0) {
		std::sort(Mods.begin(), Mods.end());
		Mods.erase(std::unique(Mods.begin(), Mods.end()), Mods.end());
	}
	LogInfo("Mods.size = " + std::to_string(Mods.size()));
	return Mods;
}

inline static RE::BGSObjectInstanceExtra* GetObjectInstanceExtraOfRef(RE::TESObjectREFR* Ref)
{
	if (!Ref)
		return nullptr;
	auto Extra = Ref->extraDataList.get();
	if (!Extra)
		return nullptr;
	auto InstExtra = Extra->GetByType(RE::ExtraDataType::kObjectInstance);
	if (!InstExtra)
		return nullptr;
	RE::BGSObjectInstanceExtra* ObjectInstanceExtra = (RE::BGSObjectInstanceExtra*)InstExtra;
	if (ObjectInstanceExtra != nullptr)
		return ObjectInstanceExtra;
	return nullptr;
}

inline static int HasItemForm(RE::TESObjectREFR* inventoryOwnerRef, RE::TESForm* itemForm)
{
	if (!inventoryOwnerRef || !itemForm)
		return -1;
	uint32_t ItemFormID = itemForm->formID;
	if (ItemFormID <= 0 || ItemFormID >= UINT32_MAX)
		return -1;
	auto InventoryList = inventoryOwnerRef->inventoryList.lock_read().operator->();
	if (!InventoryList)
		return -1;
	uint32_t InventoryItemCount = InventoryList->data.size();
	if (InventoryItemCount <= 0 || InventoryItemCount >= UINT32_MAX)
		return -1;
	for (uint32_t i = 0; i < InventoryList->data.size(); i++) {
		for (uint32_t j = 0; j < InventoryList->data[i].stacks.size(); j++) {
			auto& ItemBoundObj = InventoryList->data[i].object;
			//	auto& ExtraDataSmPtr = InventoryList->data[i].stacks[j].extra;
			auto formType = ItemBoundObj->GetSavedFormType();
			if (formType != 48 && formType != 34 && formType != 35 && formType != 54 && formType != 53 && formType != 57 && formType != 40 && formType != 49)  // Inventory Item Forms
				continue;
			if (ItemFormID == ItemBoundObj->formID)
				return 1;
		}
	}
	return 0;
}

inline static std::vector<RE::BGSObjectInstanceExtra*> GetObjectInstanceExtraDataForItemForm(RE::TESObjectREFR* inventoryOwnerRef, RE::TESForm* itemForm)
{
	std::vector<RE::BGSObjectInstanceExtra*> Elements;
	if (!inventoryOwnerRef || !itemForm)
		return Elements;
	uint32_t ItemFormID = itemForm->formID;
	if (ItemFormID <= 0 || ItemFormID >= UINT32_MAX)
		return Elements;
	auto InventoryList = inventoryOwnerRef->inventoryList.lock_read().operator->();
	if (!InventoryList)
		return Elements;
	uint32_t InventoryItemCount = InventoryList->data.size();
	if (InventoryItemCount <= 0 || InventoryItemCount >= UINT32_MAX)
		return Elements;
	for (uint32_t i = 0; i < InventoryList->data.size(); i++) {
		for (uint32_t j = 0; j < InventoryList->data[i].stacks.size(); j++) {
			auto& ItemBoundObj = InventoryList->data[i].object;
			auto& ExtraDataSmPtr = InventoryList->data[i].stacks[j].extra;
			auto  FormType = ItemBoundObj->GetSavedFormType();
			if (FormType != 34 && FormType != 48)  // ARMO and WEAP only
				continue;
			if (ItemFormID != ItemBoundObj->formID)
				continue;
			if (!ExtraDataSmPtr)
				continue;
			auto ExtraData = ExtraDataSmPtr.get();
			if (!ExtraData)
				continue;
			auto objInstanceExtra = ExtraData->GetByType(RE::ExtraDataType::kObjectInstance);
			if (!objInstanceExtra)
				continue;
			RE::BGSObjectInstanceExtra* BGSObjectInstanceExtraData = (RE::BGSObjectInstanceExtra*)objInstanceExtra;
			if (BGSObjectInstanceExtraData != nullptr) {
				Elements.push_back(BGSObjectInstanceExtraData);
			}
		}
	}
	return Elements;
}

inline static RE::TESForm* GetWorkbenchItemForm(RE::IMenu* WorkbenchMenu)
{
	if (!WorkbenchMenu)
		return nullptr;
	auto               MenuName = WorkbenchMenu->GetName();
	unsigned long long FormAddress1 = 0;
	if (MenuName != nullptr) {
		if (std::strcmp(MenuName, "WeaponsCraftingMenu") == 0)
			FormAddress1 = reinterpret_cast<unsigned long long>(WorkbenchMenu) + 0x668;
		else if (std::strcmp(MenuName, "ArmorCraftingMenu") == 0)
			FormAddress1 = reinterpret_cast<unsigned long long>(WorkbenchMenu) + 0xC58;
	}
	if (FormAddress1 <= 10000000000 || FormAddress1 >= StarfieldBaseAddress) {
		LogError("FormAddress1 is out of range at: " + GetAddressAsHex(FormAddress1));
		return nullptr;
	}
	auto FormAddress2 = reinterpret_cast<unsigned long long*>(FormAddress1);
	if (!FormAddress2) {
		LogError("FormAddress2 is nullptr.");
		return nullptr;
	}
	auto FormAddress3 = *FormAddress2;
	if (FormAddress3 <= 10000000000 || FormAddress3 >= StarfieldBaseAddress) {
		LogError("FormAddress3 is out of range at: " + GetAddressAsHex(FormAddress3));
		return nullptr;
	}
	auto Form = reinterpret_cast<RE::TESForm*>(FormAddress3);
	if (!Form) {
		LogError("Form is nullptr");
		return nullptr;
	}
	auto formType = Form->GetSavedFormType();
	if (formType == 34 || formType == 48) {
		return Form;
	} else {
		LogError("Form " + GetBasicFormData(Form, false, false, true) + " is not ARMO and not WEAP.");
		return nullptr;
	}
	LogError("unknown error at Form " + GetBasicFormData(Form, false, false, true));
	return nullptr;
}

inline static RE::TESObjectREFR* GetWorkbenchInventoryRendererRef(RE::IMenu* WorkbenchMenu)
{
	if (!WorkbenchMenu)
		return nullptr;
	auto MenuName = WorkbenchMenu->GetName();
	if (!MenuName)
		return nullptr;
	if (std::string(MenuName).empty())
		return nullptr;
	// v2.1
	RE::TESObjectREFR* inventoryItemRef = nullptr;
	try {
		RE::UICellRenderer_InventoryItemCustom* InventoryItem = nullptr;
		if (std::strcmp(MenuName, "WeaponsCraftingMenu") == 0) {
			RE::WeaponsCraftingMenuCustom* WPNMenu = (RE::WeaponsCraftingMenuCustom*)WorkbenchMenu;
			if (!WPNMenu)
				return nullptr;
			InventoryItem = WPNMenu->GetInventoryItem();
		} else if (std::strcmp(MenuName, "ArmorCraftingMenu") == 0) {
			RE::ArmorCraftingMenuCustom* ArmorMenu = (RE::ArmorCraftingMenuCustom*)WorkbenchMenu;
			if (!ArmorMenu)
				return nullptr;
			InventoryItem = ArmorMenu->GetInventoryItem();
		}
		if (!InventoryItem) {
			LogError("InventoryItem is nullptr.");
			return nullptr;
		}
		// LogInfo("getting inventoryitemRef...");
		inventoryItemRef = InventoryItem->GetInventoryItemRef();
		// LogInfo("processed inventoryitemRef.");
	} catch (...) {
		LogError("CraftingMenu exception.");
		return nullptr;
	}
	if (inventoryItemRef) {
		// LogInfo("found inventoryitemRef!");
		// LogInfo("inventoryitemRef is: " + GetBasicFormDataE(inventoryItemRef));
		return inventoryItemRef;
	}
	// LogError("inventoryItemRef is nullptr.");
	return nullptr;
}

///////////////////////////////////////////////////////////  PLUGIN  ///////////////////////////////////////////////////////////

inline static std::map<RE::BGSMod::Attachment::Mod*, RE::TESObjectMISC*> LooseModMap;
inline static std::vector<RE::BGSMod::Attachment::Mod*>                  StoredInstalledObjectMods;
inline static bool                                                       bBuildLooseModMapOnLoadGameEvent = true;
inline static bool                                                       bBuildLooseModMapOnDataLoadEvent = true;

inline static RE::TESForm* GetCreatedForm(RE::BGSConstructibleObject* constructibleObject)
{
	if (!constructibleObject) {
		LogError("constructibleObject is nullptr");
		return nullptr;
	}
	auto CreatedForm = constructibleObject->createdObject;
	if (!CreatedForm) {
		LogError("CreatedForm is nullptr");
		return nullptr;
	}
	return CreatedForm;
}

inline static uintptr_t GetLooseModsHashMapAddress()
{
	// REL::Relocation<RE::BSTHashMap<const RE::BGSMod::Attachment::Mod*, RE::TESObjectMISC*>*> mods
	REL::Relocation<uintptr_t> mods{ REL::ID(839041), -0x8 };  // (Starfield.exe + 0x05575CE0 in v1.88.88.0)
	return mods.address();
}

inline static bool IsLooseMod(uint32_t formFlags)
{
	return ((formFlags & 0x80) != 0);
}

inline static bool IsDigitsInStr(std::string s)
{
	bool Res = false;
	if (s.empty() == false)
		Res = s.find_first_not_of("0123456789") == std::string::npos;
	if (Res) {
		LogInfo("returning True for input String: " + s);
	} else
		LogInfo("returning False for input String: " + s);
	return Res;
}

inline static void ReloadSettings()
{
	bBuildLooseModMapOnLoadGameEvent = true;
	bBuildLooseModMapOnDataLoadEvent = true;
	bool bLoggingSetting = false;
	bLogging = true;

	if (DoesIniExist("FalloutObjectModFeatureRestored") == true) {
		std::string sValue = ReadIni("FalloutObjectModFeatureRestored", "Settings", "bLogging");
		if (sValue.empty() == false && std::strcmp(sValue.c_str(), "1") == 0)
			bLoggingSetting = true;
		else
			bLoggingSetting = false;
		sValue = "";

		sValue = ReadIni("FalloutObjectModFeatureRestored", "Settings", "bBuildLooseModMapOnLoadGameEvent");
		if (sValue.empty() == false && std::strcmp(sValue.c_str(), "1") == 0)
			bBuildLooseModMapOnLoadGameEvent = true;
		sValue = "";

		sValue = ReadIni("FalloutObjectModFeatureRestored", "Settings", "bBuildLooseModMapOnDataLoadEvent");
		if (sValue.empty() == false && std::strcmp(sValue.c_str(), "1") == 0)
			bBuildLooseModMapOnDataLoadEvent = true;

	} else {
		LogWarning("FalloutObjectModFeatureRestored.ini doesn't exist. Using default settings.");
	}

	LogInfo("Final Settings ----->");
	LogInfo("bBuildLooseModMapOnLoadGameEvent >> " + std::to_string(bBuildLooseModMapOnLoadGameEvent));
	LogInfo("bBuildLooseModMapOnDataLoadEvent >> " + std::to_string(bBuildLooseModMapOnDataLoadEvent));
	LogInfo("bLogging >> " + std::to_string(bLoggingSetting));
	LogInfo("<----- Final Settings");
	if (!bLoggingSetting) {
		LogInfo("turning off logging");
		bLogging = false;
	}
}

inline static bool IsBGSModForm(unsigned long long* unverifiedFormAddress)
{
	if (!unverifiedFormAddress)
		return false;
	unsigned long long* PtrCheck = nullptr;
	try {
		PtrCheck = ProtectPtr(unverifiedFormAddress);
	} catch (...) {
		return false;
	}
	if (!PtrCheck)
		return false;
	for (auto& i : RE::VTABLE::BGSMod__Attachment__Mod)
		if (i.address() == *unverifiedFormAddress)
			return true;
	return false;
}

inline static bool IsTESObjectREFR(unsigned long long* unverifiedFormAddress)
{
	if (!unverifiedFormAddress)
		return false;
	unsigned long long* PtrCheck = nullptr;
	try {
		PtrCheck = ProtectPtr(unverifiedFormAddress);
	} catch (...) {
		return false;
	}
	if (!PtrCheck)
		return false;
	for (auto& i : RE::VTABLE::TESObjectREFR)
		if (i.address() == *unverifiedFormAddress)
			return true;
	return false;
}

inline static bool IsBGSConstructubleObjectForm(unsigned long long* unverifiedFormAddress)
{
	if (!unverifiedFormAddress)
		return false;
	unsigned long long* PtrCheck = nullptr;
	try {
		PtrCheck = ProtectPtr(unverifiedFormAddress);
	} catch (...) {
		return false;
	}
	if (!PtrCheck)
		return false;
	for (auto& i : RE::VTABLE::BGSConstructibleObject)
		if (i.address() == *unverifiedFormAddress)
			return true;
	return false;
}

inline static bool IsMiscItemForm(unsigned long long* unverifiedFormAddress)
{
	if (!unverifiedFormAddress)
		return false;
	unsigned long long* PtrCheck = nullptr;
	try {
		PtrCheck = ProtectPtr(unverifiedFormAddress);
	} catch (...) {
		return false;
	}
	if (!PtrCheck)
		return false;
	for (auto& i : RE::VTABLE::TESObjectMISC)
		if (i.address() == *unverifiedFormAddress)
			return true;
	return false;
}

inline static size_t BuildLooseModMap()
{
	auto DataHandler = RE::TESDataHandler::GetSingleton();
	if (!DataHandler) {
		LogError("DataHandler is nullptr");
		return false;
	}
	auto AllObjectMods = GetAllFormsByType<RE::BGSMod::Attachment::Mod>(static_cast<int>(RE::FormType::kOMOD));
	if (AllObjectMods.size() == 0) {
		LogError("AllObjectMods.size == 0");
		return false;
	}
	LooseModMap.clear();
	for (auto& ObjectMod : AllObjectMods) {
		LogInfo("   --->   current ObjectMod " + GetBasicFormData(ObjectMod, false, false, true));
		if (!ObjectMod)
			continue;
		auto LooseMod = GetLooseMod(ObjectMod);
		LogInfo("   --->   current LooseMod " + GetBasicFormData(LooseMod, false, false, true));
		if (!LooseMod)
			continue;
		LogInfo("   --->   adding ObjectMod " + GetBasicFormData(ObjectMod, false, false, true) + " | LooseMod " + GetBasicFormData(LooseMod, false, false, true));
		LooseModMap.try_emplace(ObjectMod, LooseMod);
	}
	LogInfo("complete | LooseModMap.size [" + std::to_string(LooseModMap.size()) + "]");
	return LooseModMap.size();
}

inline static bool SetAS3Variable(std::string MenuName, std::string VarPath, RE::UI* UIPassThru = nullptr)
{
	if (MenuName.empty() || VarPath.empty())
		return false;
	auto theMenu = GetMenu(MenuName, UIPassThru);
	if (!theMenu) {
		LogError("theMenu is nullptr for MenuName: " + MenuName);
		return false;
	}
	auto menuRoot = GetMenuRoot(theMenu);
	if (!menuRoot) {
		LogError("menuRoot is nullptr for IMenu at address: " + GetAddressAsHex(reinterpret_cast<uintptr_t>(theMenu)));
		return false;
	}
	RE::Scaleform::GFx::Value* GFxValue = new RE::Scaleform::GFx::Value();
	if (!GFxValue) {
		LogError("GFxValue is nullptr");
		return false;
	}
	bool bVarFound = menuRoot->GetVariable(GFxValue, VarPath.c_str());
	if (!bVarFound) {
		LogError("bVarFound is nullptr");
		return false;
	}
	if (!GFxValue->IsBoolean()) {
		LogError("GFxValue is not Boolean");
		return false;
	}
	bool                       bSuccess = false;
	RE::Scaleform::GFx::Value* NewGFxValue = new RE::Scaleform::GFx::Value();
	menuRoot->CreateObject(NewGFxValue);
	NewGFxValue->operator=(true);
	bSuccess = menuRoot->SetVariableArray(RE::Scaleform::GFx::Movie::SetArrayType::kValue, VarPath.c_str(), 0, NewGFxValue, 1);
	return bSuccess;
}

namespace Thread
{
	int        iWorkbenchMode = 0;
	RE::IMenu* WorkbenchMenu = nullptr;
	bool       bQueueModSelectOnHandle = false;
	bool       bQueueModInstallOnHandle = false;
	bool       bQueueItemSelectOnHandle = false;

	inline static std::chrono::steady_clock::time_point iLastModHandleRequestTime{};

	inline static void SaveModHandleRequestTime(bool bSuppressLog = false)
	{
		auto now = std::chrono::high_resolution_clock::now();
		auto nowLog = std::chrono::system_clock::now();
		iLastModHandleRequestTime = now;
		if (!bSuppressLog) {
			std::stringstream ss;
			ss << nowLog;
			LogInfo("saved time [" + ss.str() + "]");
		}
	}

	inline static long long GetModHandleRequestTimeElapsed()
	{
		auto                                    now = std::chrono::high_resolution_clock::now();
		std::chrono::duration<long long, std::milli> elapsedTime = duration_cast<std::chrono::milliseconds>(now - iLastModHandleRequestTime);
		auto                                    elapsed = elapsedTime.count();
		return elapsed;
	}

	inline static std::chrono::steady_clock::time_point iLastiLastModHandleRequestCompleteTime{};

	inline static void SaveModHandleRequestCompleteTime(bool bSuppressLog = false)
	{
		auto now = std::chrono::high_resolution_clock::now();
		auto nowLog = std::chrono::system_clock::now();
		iLastiLastModHandleRequestCompleteTime = now;
		if (!bSuppressLog) {
			std::stringstream ss;
			ss << nowLog;
			LogInfo("saved time [" + ss.str() + "]");
		}
	}

	inline static long long GetModHandleRequestCompleteTimeElapsed()
	{
		auto                                    now = std::chrono::high_resolution_clock::now();
		std::chrono::duration<long long, std::milli> elapsedTime = duration_cast<std::chrono::milliseconds>(now - iLastiLastModHandleRequestCompleteTime);
		auto                                    elapsed = elapsedTime.count();
		return elapsed;
	}

	inline static std::chrono::steady_clock::time_point iLastModSelectedHandleTime{};

	inline static void SaveModSelectedHandleTime(bool bSuppressLog = false)
	{
		auto now = std::chrono::high_resolution_clock::now();
		auto nowLog = std::chrono::system_clock::now();
		iLastModSelectedHandleTime = now;
		if (!bSuppressLog) {
			std::stringstream ss;
			ss << nowLog;
			LogInfo("saved time [" + ss.str() + "]");
		}
	}

	inline static long long GetModSelectedHandleTimeElapsed()
	{
		auto                                    now = std::chrono::high_resolution_clock::now();
		std::chrono::duration<long long, std::milli> elapsedTime = duration_cast<std::chrono::milliseconds>(now - iLastModSelectedHandleTime);
		auto                                    elapsed = elapsedTime.count();
		return elapsed;
	}

	inline static std::chrono::steady_clock::time_point iLastModInstallHandleTime{};

	inline static void SaveModInstallHandleTime(bool bSuppressLog = false)
	{
		auto now = std::chrono::high_resolution_clock::now();
		auto nowLog = std::chrono::system_clock::now();
		iLastModInstallHandleTime = now;
		if (!bSuppressLog) {
			std::stringstream ss;
			ss << nowLog;
			LogInfo("saved time [" + ss.str() + "]");
		}
	}

	inline static long long GetModInstallHandleTimeElapsed()
	{
		auto                                    now = std::chrono::high_resolution_clock::now();
		std::chrono::duration<long long, std::milli> elapsedTime = duration_cast<std::chrono::milliseconds>(now - iLastModInstallHandleTime);
		auto                                    elapsed = elapsedTime.count();
		return elapsed;
	}

	inline static std::chrono::steady_clock::time_point iLastItemSelectHandleTime{};

	inline static void SaveItemSelectHandleTime(bool bSuppressLog = false)
	{
		auto now = std::chrono::high_resolution_clock::now();
		auto nowLog = std::chrono::system_clock::now();
		iLastItemSelectHandleTime = now;
		if (!bSuppressLog) {
			std::stringstream ss;
			ss << nowLog;
			LogInfo("saved time [" + ss.str() + "]");
		}
	}

	inline static long long GetItemSelectHandleTimeElapsed()
	{
		auto                                    now = std::chrono::high_resolution_clock::now();
		std::chrono::duration<long long, std::milli> elapsedTime = duration_cast<std::chrono::milliseconds>(now - iLastItemSelectHandleTime);
		auto                                    elapsed = elapsedTime.count();
		return elapsed;
	}

	HANDLE WorkbenchMenuWatcherHandle;

	inline static void HandleModSelect()
	{
		if (Thread::iWorkbenchMode == 0 || !Thread::WorkbenchMenu || GetForegroundWindow() != FindWindowA(NULL, "Starfield")) {
			return;
		} else {
			//    Thread::bFuncLock = true;

			// Navigate to the FormID of the selected item's Constructible Object
			auto MenuAddress = reinterpret_cast<unsigned long long>(Thread::WorkbenchMenu);
			if (MenuAddress <= 10000000000 || MenuAddress >= StarfieldBaseAddress) {
				LogError("MenuAddress is out of range at: " + GetAddressAsHex(MenuAddress));
				goto ExitFunc;
			}
			// auto selectedItemConstructibleObjectFormIDAddress = MenuAddress + 0x3C8;      // before v1.12.30
			auto selectedItemConstructibleObjectFormIDAddress = MenuAddress + 0x3D0;  // v1.12.30
			if (selectedItemConstructibleObjectFormIDAddress <= 10000000000 || selectedItemConstructibleObjectFormIDAddress >= StarfieldBaseAddress) {
				LogError("selectedItemConstructibleObjectFormIDAddress is out of range at: " + GetAddressAsHex(selectedItemConstructibleObjectFormIDAddress));
				goto ExitFunc;
			}
			auto selectedItemConstructibleObjectFormIDPtr = reinterpret_cast<uint32_t*>(selectedItemConstructibleObjectFormIDAddress);
			if (!selectedItemConstructibleObjectFormIDPtr) {
				LogError("selectedItemConstructibleObjectFormIDPtr is nullptr");
				goto ExitFunc;
			}
			uint32_t selectedItemConstructibleObjectFormID = 0;
			try {
				selectedItemConstructibleObjectFormID = *selectedItemConstructibleObjectFormIDPtr;
			} catch (...) {
				LogError("exception at selectedItemConstructibleObjectFormIDAddress: " + GetAddressAsHex(selectedItemConstructibleObjectFormIDAddress));
				goto ExitFunc;
			}

			if (selectedItemConstructibleObjectFormID <= 0 || selectedItemConstructibleObjectFormID >= UINT32_MAX) {
				LogError("selectedItemConstructibleObjectFormID is out of range at: " + GetUIntFormIDAsHex(selectedItemConstructibleObjectFormID));
				goto ExitFunc;
			}

			// Get the Constructible Object
			RE::TESForm* selectedItemConstructibleObjectForm = RE::TESForm::LookupByID(selectedItemConstructibleObjectFormID);
			if (!selectedItemConstructibleObjectForm) {
				LogError("selectedItemConstructibleObjectForm is nullptr");
				goto ExitFunc;
			}
			if (IsBGSConstructubleObjectForm(reinterpret_cast<uintptr_t*>(selectedItemConstructibleObjectForm)) == false) {
				LogError("selectedItemConstructibleObjectForm is not BGSConstructubleObjectForm");
				goto ExitFunc;
			}
			if (selectedItemConstructibleObjectForm->GetSavedFormType() != 151) {
				LogError("selectedItemConstructibleObjectForm is not COBJ");
				goto ExitFunc;
			}
			RE::BGSConstructibleObject* ConstructibleObject = (RE::BGSConstructibleObject*)selectedItemConstructibleObjectForm;
			if (!ConstructibleObject) {
				LogError("ConstructibleObject couldn't be initialized");
				goto ExitFunc;
			}
			RE::TESForm* CreatedObjectModForm = GetCreatedForm(ConstructibleObject);
			if (!CreatedObjectModForm) {
				LogError("CreatedObject of ConstructibleObject couldn't be found");
				goto ExitFunc;
			}
			if (IsBGSModForm(reinterpret_cast<uintptr_t*>(CreatedObjectModForm)) == false) {
				LogError("CreatedObjectModForm is not BGSMod");
				goto ExitFunc;
			}
			if (CreatedObjectModForm->GetSavedFormType() != 152) {
				LogError("CreatedObject of ConstructibleObject is not OMOD: " + std::to_string(CreatedObjectModForm->GetSavedFormType()));
				goto ExitFunc;
			}
			RE::BGSMod::Attachment::Mod* CreatedObjectMod = (RE::BGSMod::Attachment::Mod*)CreatedObjectModForm;
			if (!CreatedObjectMod) {
				LogError("CreatedObjectMod couldn't be initialized");
				goto ExitFunc;
			}
			// See if Player has the LooseMod of the selected Mod
			RE::TESObjectMISC* LooseMod = nullptr;
			auto               It = LooseModMap.find(CreatedObjectMod);
			if (It != LooseModMap.end()) {
				LogInfo("Selected ObjectMod " + GetBasicFormData(CreatedObjectMod, false, false, true) + " of Constructible Object " + GetBasicFormData(ConstructibleObject, false, false, true) + " has been found in LooseModMap");
				if (LooseMod == nullptr)
					LooseMod = It->second;
			} else {
				LogError("Selected ObjectMod " + GetBasicFormData(CreatedObjectMod, false, false, true) + " of Constructible Object " + GetBasicFormData(ConstructibleObject, false, false, true) + " couldn't be found in LooseModMap");
			}
			if (!LooseMod) {
				LogError("LooseMod is nullptr");
				goto ExitFunc;
			}
			int HasItem = HasItemForm(RE::PlayerCharacter::GetSingleton(), LooseMod);
			if (HasItem != 1) {
				LogWarning("Player does not have the LooseMod " + GetBasicFormData(LooseMod, false, false, true) + " of the selected Object Mod in their inventory | HasItemForm returned: " + std::to_string(HasItem));
				goto ExitFunc;
			}
			LogInfo("Player has the LooseMod " + GetBasicFormData(LooseMod, false, false, true) + " of the selected Object Mod in their inventory | HasItemForm returned: " + std::to_string(HasItem));

			// See if the selected Mod is already installed.
			auto SelectedItemRef = GetWorkbenchInventoryRendererRef(WorkbenchMenu);
			if (!SelectedItemRef) {
				LogError("SelectedItemRef is nullptr");
				goto ExitFunc;
			} else
				LogInfo("SelectedItemRef: " + GetBasicReferenceData(SelectedItemRef, false, false, true));

			if (std::find(StoredInstalledObjectMods.begin(), StoredInstalledObjectMods.end(), CreatedObjectMod) < StoredInstalledObjectMods.end()) {
				LogWarning("ObjectMod IS installed: " + GetBasicFormData(CreatedObjectMod, false, false, true));
				goto ExitFunc;
			}

			// If so, enable the 'Install' button through Scaleform
			int         iMaxSupportedModEntryCount = 20;
			int         iEntryIndex = 0;
			std::string sMenuName = Thread::WorkbenchMenu->GetName();
			for (int i = 0; i < iMaxSupportedModEntryCount; i++) {
				std::string sEntryPath = "root1.Menu_mc.ModsDirectory_mc.ModsList_mc.EntryHolder_mc.Entry" + std::to_string(iEntryIndex);
				//	int         tempInt = iEntryIndex;
				iEntryIndex = iEntryIndex + 1;
				std::string sFramePath = sEntryPath + ".currentFrame";
				std::string sFrameIndex = GetAS3VariableAsString(sMenuName, sFramePath);
				// Debug
				std::string sDebugSuffix = "";
				// sDebugSuffix = " | iEntryIndex [" + std::to_string(tempInt) + "]. | EntryText [" + GetAS3VariableAsString(sMenuName, sEntryPath + ".Component_mc.Text_tf.text") +
				// "]. IsInstalled [" + GetAS3VariableAsString(sMenuName, sEntryPath + ".Installed_mc.visible") + "].";

				// if empty, then this entry doesn't exist (normally this should only happen if the loop reached the last entry, so exit)
				if (sFrameIndex.empty()) {
					LogWarning("Exiting function because FrameIndex is empty. " + sDebugSuffix);
					goto ExitFunc;
				}
				// Check if the entry is selected by the cursor
				// UI sets these two frame indexes, both means selected.
				// Normally, only one can be selected.
				if (std::strcmp(sFrameIndex.c_str(), "2") != 0 && std::strcmp(sFrameIndex.c_str(), "5") != 0) {
					LogWarning("Skipping non-selected Entry. " + sDebugSuffix);
					continue;
				}
				std::string sInstalledMCPath = sEntryPath + ".Installed_mc.visible";
				std::string sIsInstalled = GetAS3VariableAsString(sMenuName, sInstalledMCPath);
				// GetAS3VariableAsString should return 'True' but still.
				// Check if the Mod in this entry is installed.
				// If not, skip.
				if (std::strcmp(sIsInstalled.c_str(), "True") == 0 || std::strcmp(sIsInstalled.c_str(), "true") == 0 || std::strcmp(sIsInstalled.c_str(), "1") == 0) {
					LogWarning("Skipping installed Entry. " + sDebugSuffix);
					continue;
				}
				// Enable the Install button
				bool bButtonEnabled = SetAS3Variable(sMenuName, "root1.Menu_mc.ComponentCostPanel_mc.CreateButton_mc.Enabled");
				if (bButtonEnabled) {
					// Allow free crafting so the game code won't consume resources.
					SetGameINISetting("bFreeItemCrafting:Crafting", "1", 3);
					LogInfo("Install button IS enabled. " + sDebugSuffix);
				} else {
					LogError("Install button couldn't be enabled. " + sDebugSuffix);
					// Failsafe
					SetGameINISetting("bFreeItemCrafting:Crafting", "0", 3);
				}
				goto ExitFunc;
			}

			LogError("Unexpected function end");
		}

ExitFunc:
		LogWarning("GoTo Exit called");
		return;
	}

	inline static void HandleInstallMod()
	{
		// Compare the selected Item's Mods to the stored ones.
		// Add the LooseMod of the Mod that was removed by the game code
		// upon removing the 'invalid mod' from the ModSlot.
		// Thankfully, the game code actually auto-removes a Mod's LooseMod
		// when it detects the Player's inventory contains the LooseMod
		// so no need to handle that.

		if (GetForegroundWindow() != FindWindowA(NULL, "Starfield")) {
			LogError("GetForegroundWindow error");
			return;
		}

		if (Thread::iWorkbenchMode <= 0) {
			LogError("Thread::iWorkbenchMode <= 0");
			return;
		}

		RE::BSScript::IVirtualMachine* VM = RE::GameVMCustom::GetSingleton()->virtualMechine;
		if (!VM) {
			LogError("VM is nullptr");
			return;
		}

		auto PlayerPapyrusHandle = GetHandleFromObject(RE::PlayerCharacter::GetSingleton(), static_cast<uint32_t>(RE::FormType::kACHR));
		if (PlayerPapyrusHandle == 0) {
			LogError("PlayerPapyrusHandle == 0");
			return;
		}

		// Get the Selected Item, its InstanceExtra and its Mods
		LogInfo("called.");
		auto ItemRef = GetWorkbenchInventoryRendererRef(Thread::WorkbenchMenu);
		if (!ItemRef) {
			LogError("ItemRef is nullptr");
			return;
		}
		auto ObjectInstanceExtra = GetObjectInstanceExtraOfRef(ItemRef);
		if (!ObjectInstanceExtra) {
			LogError("ObjectInstanceExtra is nullptr");
			return;
		}
		auto latestMods = GetInstalledObjectMods(ObjectInstanceExtra);

		// Error-check
		if (latestMods.empty()) {
			LogError("latestMods vector is empty");
			return;
		}

		for (size_t i = 0; i < latestMods.size(); i++) {
			auto Mod = latestMods.at(i);
			auto ModAddr = reinterpret_cast<uintptr_t*>(Mod);
			if (IsBGSModForm(ModAddr) == false) {
				LogError("Mod in std::vector latestMods is not BGSMod: " + GetAddressAsHex(reinterpret_cast<uintptr_t>(Mod)));
				continue;
			}
			if (std::find(StoredInstalledObjectMods.begin(), StoredInstalledObjectMods.end(), Mod) >= StoredInstalledObjectMods.end()) {
				LogInfo("ObjectMod is not stored, should be the newly attached Mod. " + GetBasicFormData(Mod, false, false, true));
				auto It = LooseModMap.find(Mod);
				if (It != LooseModMap.end()) {
					// logger::info("HandleInstallMod   --->   ObjectMod has been found in LooseModMap.");
				} else {
					LogError("Newly attached ObjectMod couldn't be found in LooseModMap");
				}
			}
		}

		// 'Originally attached' Mods were stored in event SelectedModdableItem
		// if latestMods doesn't contain one from the store, then it must have been removed by the game code as an 'invalidmod'.
		std::vector<RE::TESObjectMISC*> LooseModsOfRemovedMods;
		for (size_t i = 0; i < StoredInstalledObjectMods.size(); i++) {
			auto Mod = StoredInstalledObjectMods.at(i);
			auto ModAddr = reinterpret_cast<uintptr_t*>(Mod);
			if (IsBGSModForm(ModAddr) == false) {
				LogError("Mod in std::vector StoredInstalledObjectMods is not BGSMod: " + GetAddressAsHex(reinterpret_cast<uintptr_t>(Mod)));
				continue;
			}
			if (std::find(latestMods.begin(), latestMods.end(), Mod) >= latestMods.end()) {
				LogInfo("ObjectMod is in the store but currently not attached, should be the removed Mod. " + GetBasicFormData(Mod, false, false, true));
				auto It = LooseModMap.find(Mod);
				if (It != LooseModMap.end()) {
					LogInfo("ObjectMod  ->  " + GetBasicFormData(Mod, false, false, true) + " has been found in LooseModMap");
					LooseModsOfRemovedMods.push_back(It->second);
				} else {
					LogError("Removed ObjectMod couldn't be found in LooseModMap");
				}
			}
		}

		// Error-check
		if (LooseModsOfRemovedMods.empty() == false) {
			// Add the LooseMods (should be only one though)
			for (uint32_t i = 0; i < LooseModsOfRemovedMods.size(); i++) {
				//std::string asCommand = "14.CF \"ObjectReference.AddItem\" " + GetBasicFormData(LooseModsOfRemovedMods.at(i), false, false, true) + " 1 " + " False";
				//ExecuteCommand(asCommand.c_str());
				CallPapyrusFunction(PlayerPapyrusHandle, "ObjectReference", "AddItem", VM, (RE::TESForm*)LooseModsOfRemovedMods.at(i), 1, false);
				LogInfo("AddItem  ->  LooseMod: " + GetBasicFormData(LooseModsOfRemovedMods.at(i), false, false, true));
			}
		} else
			LogWarning("LooseModsOfRemovedMods vector is empty");

		// Rebuild the installed ObjectMods store as the item has a new installed mod now.
		StoredInstalledObjectMods.clear();
		for (size_t i = 0; i < latestMods.size(); i++) {
			StoredInstalledObjectMods.push_back(latestMods.at(i));
			LogInfo("Storing ObjectMod  ->  " + GetBasicFormData(latestMods.at(i), false, false, true));
		}
	}

	inline static void HandleItemSelect()
	{
		if (Thread::iWorkbenchMode == 0 || !Thread::WorkbenchMenu || GetForegroundWindow() != FindWindowA(NULL, "Starfield")) {
			return;
		}

		auto ItemRef = GetWorkbenchInventoryRendererRef(Thread::WorkbenchMenu);
		if (!ItemRef) {
			LogError("ItemRef is nullptr");
			return;
		}
		if (IsTESObjectREFR(reinterpret_cast<uintptr_t*>(ItemRef)) == false) {
			LogError("ItemRef is not TESObjectREFR");
			return;
		}

		auto ObjectInstanceExtra = GetObjectInstanceExtraOfRef(ItemRef);
		if (!ObjectInstanceExtra) {
			LogError("ObjectInstanceExtra is nullptr");
			return;
		}
		auto tempModsVec = GetInstalledObjectMods(ObjectInstanceExtra);

		// Clear the vector of dirty data then rebuild it
		StoredInstalledObjectMods.clear();
		if (tempModsVec.empty() == false) {
			for (size_t i = 0; i < tempModsVec.size(); i++) {
				StoredInstalledObjectMods.push_back(tempModsVec.at(i));
				LogInfo("Storing ObjectMod  ->  " + GetBasicFormData(tempModsVec.at(i), false, false, true));
			}
		} else {
			LogError("Item " + GetBasicReferenceData(ItemRef, false, false, true) + " has no ObjectMods ??? Could not build installed Mods store.");
		}
	}

	static DWORD Thread_WorkbenchMenuWatcher(void* param)
	{
		(void)param;
		while (Thread::iWorkbenchMode > 0 && Thread::WorkbenchMenu != nullptr) {
			Sleep(75);
			if (Thread::bQueueModSelectOnHandle || Thread::bQueueModInstallOnHandle || Thread::bQueueItemSelectOnHandle) {
				if (GetModHandleRequestTimeElapsed() > 150 && GetModHandleRequestCompleteTimeElapsed() > 150) {  // make sure frequent SelectedMod events doesn't cause a crash
					SaveModHandleRequestTime();
					if (Thread::bQueueItemSelectOnHandle && GetItemSelectHandleTimeElapsed() >= 150) {
						//SaveItemSelectHandleTime();
						Thread::bQueueItemSelectOnHandle = false;
						HandleItemSelect();
					} else if (Thread::bQueueModSelectOnHandle && GetModSelectedHandleTimeElapsed() >= 150) {
						//SaveModSelectedHandleTime();
						Thread::bQueueModSelectOnHandle = false;
						HandleModSelect();
					} else if (Thread::bQueueModInstallOnHandle && GetModInstallHandleTimeElapsed() >= 150) {
						//SaveModInstallHandleTime();
						Thread::bQueueModInstallOnHandle = false;
						HandleInstallMod();
					}
					SaveModHandleRequestCompleteTime();
				}
			}
		}
		if (Thread::WorkbenchMenuWatcherHandle != NULL) {
			CloseHandle(Thread::WorkbenchMenuWatcherHandle);
			Thread::WorkbenchMenuWatcherHandle = NULL;
		}
		return 0;
	}

}  // namespace Thread

namespace Events
{
	void HandleOnMenuOpenCloseEvent(const char* chMenuName, bool abOpening)
	{
		if (!chMenuName) {
			LogError("chMenuName is nullptr");
			return;
		}
		if (abOpening) {
			auto m1 = GetMenu(chMenuName);
			if (m1 != nullptr) {
				if (std::strcmp(chMenuName, "WeaponsCraftingMenu") == 0) {
					Thread::iWorkbenchMode = 1;
				} else if (std::strcmp(chMenuName, "ArmorCraftingMenu") == 0) {
					Thread::iWorkbenchMode = 2;
				}
				Thread::WorkbenchMenu = m1;
				LogInfo(std::string(chMenuName) + " opened at address [" + GetAddressAsHex(reinterpret_cast<unsigned long long>(Thread::WorkbenchMenu)) + "].");

				Thread::WorkbenchMenuWatcherHandle = CreateThread(NULL, 8192, Thread::Thread_WorkbenchMenuWatcher, NULL, 0, NULL);
			} else {
				LogError(std::string(chMenuName) + " opened but no address were found any open menu with this name");
				return;
			}
		} else {
			LogInfo(std::string(chMenuName) + " closed. Removing Menu data...");
			Thread::iWorkbenchMode = 0;
			Thread::WorkbenchMenu = nullptr;
			Thread::bQueueModSelectOnHandle = false;
			Thread::bQueueModInstallOnHandle = false;
			Thread::bQueueItemSelectOnHandle = false;
			SetGameINISetting("bFreeItemCrafting:Crafting", "0", 3);
		}
	}

	class SFCEMenuOpenCloseEventClass : public RE::BSTEventSink<RE::MenuOpenCloseEvent>
	{
		RE::BSEventNotifyControl ProcessEvent(const RE::MenuOpenCloseEvent& a_event, RE::BSTEventSource<RE::MenuOpenCloseEvent>*)
		{
			// Prepare/handle Menu data for the CraftingMenu events here

			const char* chMenuName = a_event.menuName.c_str();
			if (!chMenuName) {
				LogError("MenuOpenCloseEvent EVENT   --->   chMenuName is nullptr");
				return RE::BSEventNotifyControl::kContinue;
			}
			if (std::string(chMenuName).empty()) {
				LogError("MenuOpenCloseEvent EVENT   --->   chMenuName is empty");
				return RE::BSEventNotifyControl::kContinue;
			}
			if (std::strcmp(chMenuName, "WeaponsCraftingMenu") == 0 || std::strcmp(chMenuName, "ArmorCraftingMenu") == 0) {
				HandleOnMenuOpenCloseEvent(chMenuName, a_event.opening);
			}
			return RE::BSEventNotifyControl::kContinue;
		}
	};

	SFCEMenuOpenCloseEventClass SFCEMenuOpenCloseEventSink;

	class SMODCraftingMenuSelectedModClass : public RE::BSTEventSink<CraftingMenu_SelectedMod>
	{
	public:
		virtual ~SMODCraftingMenuSelectedModClass(){};

		virtual RE::BSEventNotifyControl ProcessEvent(const CraftingMenu_SelectedMod& _event, RE::BSTEventSource<CraftingMenu_SelectedMod>* _source) override  // 01 override
		{
			if (Thread::iWorkbenchMode != 1 && Thread::iWorkbenchMode != 2) {
				return RE::BSEventNotifyControl::kContinue;
			}

			// Disallow free crafting as fast as possible so the UI
			// will receive 'dimmed' entries by the game
			// (the game code frequently checks this INI setting)

			// v3.4
			Thread::SaveModSelectedHandleTime();

			SetGameINISetting("bFreeItemCrafting:Crafting", "0", 3);
			LogInfo("SelectedMod EVENT   --->   received");
			Thread::bQueueModSelectOnHandle = true;

			return RE::BSEventNotifyControl::kContinue;
		}
	};

	SMODCraftingMenuSelectedModClass SMODCraftingMenuSelectedModSink;

	class SMODCraftingMenuInstallModClass : public RE::BSTEventSink<CraftingMenu_InstallMod>
	{
	public:
		virtual ~SMODCraftingMenuInstallModClass(){};

		virtual RE::BSEventNotifyControl ProcessEvent(const CraftingMenu_InstallMod& _event, RE::BSTEventSource<CraftingMenu_InstallMod>* _source) override  // 01 override
		{
			// This event is received after the Player attempts to install a Mod on the selected Item.
			// Sent even if the game code doesn't actually perform the mod attachment
			// Create a thread for a delayed check as the 'new installed Mod' might not be attached at this time.

			LogInfo("InstallMod EVENT   --->   received");

			// v3.4
			Thread::SaveModInstallHandleTime();

			Thread::bQueueModInstallOnHandle = true;

			return RE::BSEventNotifyControl::kContinue;
		}
	};

	SMODCraftingMenuInstallModClass SMODCraftingMenuInstallModSink;

	class SMODCraftingMenuSelectedModdableItemClass : public RE::BSTEventSink<CraftingMenu_SelectedModdableItem>
	{
	public:
		virtual ~SMODCraftingMenuSelectedModdableItemClass(){};

		virtual RE::BSEventNotifyControl ProcessEvent(const CraftingMenu_SelectedModdableItem& _event,
			RE::BSTEventSource<CraftingMenu_SelectedModdableItem>*                             _source) override  // 01 override
		{
			// Store installed mods and the inventory renderer's reference.
			// This event is received after the Player selects an inventory item in the crafting menu.
			// Since the item's renderer 3D reference is created on 'CraftingMenu_ViewingModdableItem',
			// there's no need to implement a delay (i.e. by the time this is event is received the ref is fully ready)
			//      (the above might have been changed in v1.12.30: in this version, there's no TESObjectREFR* directly in the IMenu structure of the Weapons/Armor crafting menus..
			//      there are base form, instance data and UIInventoryItem)

			LogInfo("SelectedModdableItem EVENT  --->  received");

			// v3.4
			Thread::SaveItemSelectHandleTime();

			// v1.8 reset watcher variables
			Thread::bQueueModSelectOnHandle = false;
			Thread::bQueueModInstallOnHandle = false;  // why true in v2.0???

			// v2.1 queue item select func
			Thread::bQueueItemSelectOnHandle = true;

			return RE::BSEventNotifyControl::kContinue;
		}
	};

	SMODCraftingMenuSelectedModdableItemClass SMODCraftingMenuSelectedModdableItemSink;

	class SMODCraftingMenuSelectedModSlotClass : public RE::BSTEventSink<CraftingMenu_SelectedModSlot>
	{
	public:
		virtual ~SMODCraftingMenuSelectedModSlotClass(){};

		virtual RE::BSEventNotifyControl ProcessEvent(const CraftingMenu_SelectedModSlot& _event, RE::BSTEventSource<CraftingMenu_SelectedModSlot>* _source) override  // 01 override
		{
			// This event is received after the Player selects a 'ModSlot'
			return RE::BSEventNotifyControl::kContinue;
		}
	};

	SMODCraftingMenuSelectedModSlotClass SMODCraftingMenuSelectedModSlotSink;

	class SMODTESContainerChangedEventClass : public RE::BSTEventSink<TESContainerChangedEvent>
	{
	public:
		virtual ~SMODTESContainerChangedEventClass(){};

		virtual RE::BSEventNotifyControl ProcessEvent(const TESContainerChangedEvent& _event, RE::BSTEventSource<TESContainerChangedEvent>* _source) override  // 01 override
		{
			// For debugging only
			// Track Misc. Items added/removed from the Player's inventory while in the CraftingMenu.

			if (_event.targetContainerFormID != 0x14 && _event.sourceContainerFormID != 0x14)
				return RE::BSEventNotifyControl::kContinue;
			if (_event.itemFormID == 0)
				return RE::BSEventNotifyControl::kContinue;
			if (RE::UI::GetSingleton()->IsMenuOpen("WeaponsCraftingMenu") == false && RE::UI::GetSingleton()->IsMenuOpen("ArmorCraftingMenu") == false)
				return RE::BSEventNotifyControl::kContinue;
			RE::TESForm* Item = RE::TESForm::LookupByID(_event.itemFormID);
			if (!Item)
				return RE::BSEventNotifyControl::kContinue;
			if (Item->GetSavedFormType() != 40)  // MISC only
				return RE::BSEventNotifyControl::kContinue;
			RE::TESObjectMISC* Misc = (RE::TESObjectMISC*)Item;
			if (!Misc)
				return RE::BSEventNotifyControl::kContinue;
			bool bAdded = false;
			bool bFound = false;
			if (_event.targetContainerFormID == 0x14)
				bAdded = true;
			for (const auto& [key, value] : LooseModMap)
				if (value->formID == Misc->formID) {
					bFound = true;
					break;
				}
			std::string sDebug = "";
			if (bAdded)
				sDebug = "Misc was [Added] to the Player's inventory";
			else
				sDebug = "Misc was [Removed] from the Player's inventory";
			if (bFound)
				sDebug = sDebug + " and was found in the LooseModMap -> ";
			else
				sDebug = sDebug + " but WAS NOT found in the LooseModMap -> ";
			sDebug = sDebug + GetBasicFormData(Misc, false, false, true) + "  Count (" + std::to_string(_event.itemCount) + ").";
			LogInfo("TESContainerChangedEvent   --->   " + sDebug);

			return RE::BSEventNotifyControl::kContinue;
		}
	};

	SMODTESContainerChangedEventClass SMODTESContainerChangedEventSink;

	class SMODTESLoadGameEventClass : public RE::BSTEventSink<TESLoadGameEvent>
	{
	public:
		virtual ~SMODTESLoadGameEventClass(){};

		virtual RE::BSEventNotifyControl ProcessEvent(const TESLoadGameEvent& _event, RE::BSTEventSource<TESLoadGameEvent>* _source) override  // 01 override
		{
			LogInfo("TESLoadGameEvent EVENT   --->   received: Reloading settings...");

			// v1.8: reset some variables
			Thread::iWorkbenchMode = 0;
			Thread::bQueueModSelectOnHandle = false;
			Thread::bQueueModInstallOnHandle = false;
			Thread::bQueueItemSelectOnHandle = false;  // v2.1

			// Settings
			ReloadSettings();

			if (bBuildLooseModMapOnLoadGameEvent) {
				auto size = BuildLooseModMap();
				if (size == 0) {
					LogError("TESLoadGameEvent EVENT   --->   LooseModMap.size == 0, shutting down");
					std::string sDbg = { PluginVersionInfo::PROJECT.begin(), PluginVersionInfo::PROJECT.end() };
					sDbg = sDbg + ": ERROR: no Loose Mod could be found for any Object Mod. Mod has been deactivated, shutting down.";
					ShowMessageBox(sDbg, false);
				} else {
					LogInfo("TESLoadGameEvent EVENT   --->   LooseModMap.size [" + std::to_string(size) + "]");
				}
			}

			return RE::BSEventNotifyControl::kContinue;
		}
	};

	SMODTESLoadGameEventClass SMODTESLoadGameEventSink;

}  // namespace Events

/*
struct FuncStruct
{
	static uintptr_t                              Func(uintptr_t a1, uintptr_t a2, uintptr_t a3);
	inline static REL::Relocation<decltype(Func)> func;
	static uintptr_t                              Func2(uintptr_t a1);
	inline static REL::Relocation<decltype(Func2)> func2;
};

uintptr_t FuncStruct::Func(uintptr_t a1, uintptr_t a2, uintptr_t a3)
{

	auto result = func(a1, a2, a3);

	bLogging = true;

	LogInfo("called | a1 [" + GetAddressAsHex(a1) + "] | a2 [" + GetAddressAsHex(a1) + "] | a3 [" + GetAddressAsHex(a1) + "] | result [" + GetAddressAsHex(result) + "]");

	return result;

}

uintptr_t FuncStruct::Func2(uintptr_t a1)
{

	bLogging = true;

	LogInfo("called | a1 [" + GetAddressAsHex(a1) + "]");
	auto result = func2(a1);
	LogInfo("called | result [" + GetAddressAsHex(result) + "]");

	return result;
}
*/

/////////////////////////////////////////////////////////// SFSE ///////////////////////////////////////////////////////////

inline static void HandleOnPostDataLoad(SFSE::MessagingInterface::Message* Message) noexcept
{
	if (Message->type == SFSE::MessagingInterface::kPostDataLoad) {
		LogInfo("kPostDataLoad");

		// Startup

		StarfieldBaseAddress = (unsigned long long)GetModuleHandle(NULL);

		if (StarfieldBaseAddress <= 0) {
			LogError("StarfieldBaseAddress == 0, shutting down.");
			std::string sDbg = { PluginVersionInfo::PROJECT.begin(), PluginVersionInfo::PROJECT.end() };
			sDbg = sDbg + ": ERROR: address of 'Starfield.exe' couldn't found. Mod is inactive, shutting down.";
			ShowMessageBox(sDbg, true);
			return;
		}

		if (bBuildLooseModMapOnDataLoadEvent) {
			auto size = BuildLooseModMap();
			if (size == 0) {
				LogError("LooseModMap.size == 0, shutting down");
				std::string sDbg = { PluginVersionInfo::PROJECT.begin(), PluginVersionInfo::PROJECT.end() };
				sDbg = sDbg + ": ERROR: no Loose Mod could be found for any Object Mod. Mod is inactive, shutting down.";
				ShowMessageBox(sDbg, true);
			} else {
				LogInfo("LooseModMap.size [" + std::to_string(size) + "]");
			}
		}

		// v1.8
		Thread::SaveModHandleRequestTime();

		// v2.1
		Thread::SaveModHandleRequestCompleteTime();

		// v3.4
		Thread::SaveItemSelectHandleTime();
		Thread::SaveModSelectedHandleTime();
		Thread::SaveModInstallHandleTime();

		// Settings
		ReloadSettings();

		/*
		SFSE::AllocTrampoline(512);
		auto& Trampoline = SFSE::GetTrampoline();
		REL::Relocation<uintptr_t> reloc{ REL::ID(101697), 0x13 };
		FuncStruct::func = Trampoline.write_call<5>(reloc.address(), FuncStruct::Func);
		REL::Relocation<uintptr_t> reloc2{ REL::ID(147139), 0x13 };
		FuncStruct::func2 = Trampoline.write_call<5>(reloc2.address(), FuncStruct::Func2);
		REL::Relocation<uintptr_t> reloc3{ REL::ID(147139), 0x20 };
		FuncStruct::func2 = Trampoline.write_call<5>(reloc3.address(), FuncStruct::Func2);
		REL::Relocation<uintptr_t> reloc4{ REL::ID(147147), 0x127 };
		FuncStruct::func2 = Trampoline.write_call<5>(reloc4.address(), FuncStruct::Func2);
		*/

		Events::TESLoadGameEvent().GetEventSource()->RegisterSink(&Events::SMODTESLoadGameEventSink);
		RE::UI::GetSingleton()->RegisterSink(&Events::SFCEMenuOpenCloseEventSink);
		Events::CraftingMenu_SelectedMod().GetEventSource()->RegisterSink(&Events::SMODCraftingMenuSelectedModSink);
		Events::CraftingMenu_InstallMod().GetEventSource()->RegisterSink(&Events::SMODCraftingMenuInstallModSink);
		Events::CraftingMenu_SelectedModSlot().GetEventSource()->RegisterSink(&Events::SMODCraftingMenuSelectedModSlotSink);
		Events::CraftingMenu_SelectedModdableItem().GetEventSource()->RegisterSink(&Events::SMODCraftingMenuSelectedModdableItemSink);
		Events::TESContainerChangedEvent().GetEventSource()->RegisterSink(&Events::SMODTESContainerChangedEventSink);
	}
}

extern "C" DLLEXPORT constinit auto SFSEPlugin_Version = []() noexcept {
	SFSE::PluginVersionData data{};
	data.PluginVersion(PluginVersionInfo::VERSION);
	data.PluginName(PluginVersionInfo::PROJECT);
	data.AuthorName(PluginVersionInfo::AUTHORNAME);
	data.UsesAddressLibrary(true);
	data.IsLayoutDependent(true);
	data.CompatibleVersions({ SFSE::RUNTIME_SF_1_14_70, 0 });
	return data;
}();

extern "C" DLLEXPORT bool SFSEAPI SFSEPlugin_Load(const SFSE::LoadInterface* a_sfse)
{
	SFSE::Init(a_sfse);

	LogInfo(PluginVersionInfo::PROJECT.data() + " v" + PluginVersionInfo::NAME.data());

	// Startup
	StarfieldBaseAddress = (unsigned long long)GetModuleHandle(NULL);
	if (!StarfieldBaseAddress) {
		LogError("Starfield.exe handle couldn't be acquired, shutting down");
		return false;
	}

	SFSE::GetMessagingInterface()->RegisterListener(HandleOnPostDataLoad);
	return true;
}
