-- Made by Neverless @ BeamMP. Problems, Questions or requests? Feel free to ask.

--[[
	WIP
	
	Todo
		- test line overlap
]]

package.loaded[mainLevel.findLib("libs/PathC")] = nil
local PathC = require(mainLevel.findLib("libs/PathC"))

package.loaded[mainLevel.findLib("libs/PosMapC")] = nil
local PosMapC = require(mainLevel.findLib("libs/PosMapC"))

local M = {
	_VERSION = "0.2", -- 29.05.2025
}

local ROOT_GROUP = 'hazards'
local SLOW_SPEED = 60 / 3.6 -- kph
local STOPPED_SPEED = 10 / 3.6 -- kph

-- range to consider registering lights from. a light stays black if it was to far away.
-- Same time you dont want this to be to great, or it will consider lights on other bits of the track
local LIGHT_RANGE = 30

-- the amount of meters behind the hazard to swap the lights
local HAZARD_RANGE = 600

--[[
	Format
	["vehicle_id"] = table
		[is_hazard] = bool
		[hazard_kind] = int
		[lights] = table
			--[1..n] = string ("x y" as a means to have a ref to the LINE_MAP to disable the lights when far enough away again)
			["obj_id"] = int (distance)
]]
local VEHICLES = {}

--[[
	Format
	[1..n] = PosMapC
]]
local POS_MAPS = {}
local PATH_DIR = '/levels/%s/lua/hazard_pathes/'

--[[
	Format
	["obj_id"] = table
		[Hazard.Enum] = table
			["vehicle_id"] = true
]]
local LIGHTS = {}
local LIGHT_OVERWRITE = nil

local IS_ENABLED = false

-- hazard enum
local Hazard = {
	Slow = 1,
	Stopped = 2,
	Reverse = 3,
}

-- -------------------------------------------------------------------------------
-- Common
local function tableVToK(table) -- consuming
	for k, v in ipairs(table) do
		table[v] = true
		table[k] = nil
	end
	return table
end

local function vTableMerge(from, into) -- alters into
	for _, v in ipairs(from) do
		table.insert(into, v)
	end
	return into
end

local function tableMerge(from, into) -- alters into
	for k, v in pairs(from) do
		into[k] = v
	end
	return into
end

local function fileName(path)
	local str = path:sub(1):gsub("\\", "/")
	local _, pos = str:find(".*/")
	if pos == nil then return path end
	return str:sub(pos + 1, -1)
end

local function fileExtension(path)
	return path:match("[^.]+$")
end

local function getFileListWithExtensions(path, ...)
	local whitelist = tableVToK({...})
	local list = {}
	for _, file in ipairs(FS:directoryList(path) or {}) do
		if whitelist[fileExtension(file):lower()] then
			table.insert(list, file)
		end
	end
	return list
end

local function dist2d(p1, p2)
	return math.sqrt((p2.x - p1.x)^2 + (p2.y - p1.y)^2)
end

local function findAllObjectsInSimgroupOfType(sim_group, ...)
	local classes = tableVToK({...})
	local objects = {}
	for i = 0, sim_group:getCount() do
		local obj = scenetree.findObjectById(sim_group:idAt(i))
		if classes[obj:getClassName()] and obj:getName() ~= "RootGroup" then
			table.insert(objects, obj)
		end
	end
	return objects
end

local function findAllObjectsInSimgroupOfTypeRecursive(sim_group, ...)
	local classes = tableVToK({...})
	local objects = {}
	for i = 0, sim_group:getCount() do
		local obj = scenetree.findObjectById(sim_group:idAt(i))
		local class_name = obj:getClassName()
		if class_name == "Prefab" then
			obj = obj:getChildGroup()
		end
		if class_name == "SimGroup" then
			if obj:getName() ~= "RootGroup" then
				vTableMerge(findAllObjectsInSimgroupOfTypeRecursive(obj, ...), objects)
			end
		end
		if classes[class_name] then
			table.insert(objects, obj)
		end
	end
	return objects
end

local function tableHasContent(table)
	return #({next(table)}) > 0
end

-- superficial
local function tableCopy(table)
	local copy = {}
	for k, v in pairs(table) do
		copy[k] = v
	end
	return copy
end

-- -------------------------------------------------------------------------------
-- Lights
local function indexLights(sim_group)
	local lights = findAllObjectsInSimgroupOfTypeRecursive(sim_group, 'PointLight', 'SpotLight', 'TSStatic')
	for _, light in ipairs(lights) do
		local class_name = light:getClassName()
		if class_name == 'PointLight' or class_name == 'SpotLight' then
			light.isEnabled = false
			light:setField("flareType", 0, "vehicleBrakeLightFlare")
			light:setField("flareScale", 0, 0.5)
			light.animate = false
			light.color = Point4F(0, 1, 0, 1)
			
		elseif class_name == 'TSStatic' then
			light.useInstanceRenderData = true
			local color = Point4F(0, 0, 0, 0)
			light.instanceColor = color
			light.instanceColor1 = color
			light.instanceColor2 = color
			light:postApply()
		end
		
		local enum = {}
		for _, index in pairs(Hazard) do
			enum[index] = {}
		end
		LIGHTS[light:getId()] = enum
	end
	
	return lights
end

local function evalLightState(obj_id)
	local light = LIGHTS[obj_id]
	local state = 0
	for index, vehicles in pairs(light) do
		if tableHasContent(vehicles) then
			state = index
		end
	end
	
	local obj = scenetree.findObjectById(obj_id)
	if not obj then return end
	
	local class_name = obj:getClassName()
	if class_name == 'SpotLight' or class_name == 'PointLight' then
		if state == 0 then
			obj.animate = false
			obj.color = Point4F(0, 1, 0, 1)
			
		elseif state == Hazard.Slow then
			obj.color = Point4F(1, 0.9, 0, 1)
			obj.animate = false
			
		elseif state == Hazard.Stopped then
			obj.color = Point4F(1, 0.9, 0, 1)
			obj:setField("animationPeriod", 0, 1)
			obj:setField("animationType", 0, "PulseLightAnim")
			obj.animate = true
			
		elseif state == Hazard.Reverse then
			obj.color = Point4F(1, 0, 0, 1)
			obj:setField("animationPeriod", 0, 1)
			obj:setField("animationType", 0, "PulseLightAnim")
			obj.animate = true
		end
	
	elseif class_name == 'TSStatic' then
		local color = Point4F(0, 1, 0, 1)
			
		if state == Hazard.Slow then
			color = Point4F(1, 0.9, 0, 1)
			
		elseif state == Hazard.Stopped then
			color = Point4F(1, 0.9, 0, 1)
			
		elseif state == Hazard.Reverse then
			color = Point4F(1, 0, 0, 1)
			
		end
		obj.instanceColor = color
		obj.instanceColor1 = color
		obj.instanceColor2 = color
		obj:postApply()
	end
end

local function addLightState(obj_id, vehicle_id, state, previous_state)
	local light = LIGHTS[obj_id]
	if not light then return end
	
	if state > 0 then light[state][vehicle_id] = true end
	if previous_state > 0 then light[previous_state][vehicle_id] = nil end
	
	evalLightState(obj_id)
end

local function globalLightOverwrite(kind, state)
	if kind == nil then return end
	for obj_id, kinds in pairs(LIGHTS) do
		kinds[kind].overwrite = state
		
		evalLightState(obj_id)
	end
end

-- -------------------------------------------------------------------------------
-- Hazard stuff
local function checkHazard(vehicle)
	local pos = vehicle:getPosition()
	local lights = {}
	local distances = {}
	local any = false
	for index, pos_map in ipairs(POS_MAPS) do
		local point = pos_map:getPoint3d(pos)
		if point then
			tableMerge(point.value.lights, lights)
			distances[index] = point.value.distance
			any = true
		end
	end
	if not any then return end
	
	local vel = vehicle:getVelocity()
	local speed = vel:length()
	
	-- check if going reverse
	if speed > 1 then
		local pre_pos = pos + (vel:normalized() * 5)
		for index, pos_map in ipairs(POS_MAPS) do
			local point = pos_map:getPoint3d(pre_pos)
			if point and distances[index] then
				if point.value.distance < distances[index] then
					return true, Hazard.Reverse, lights
				end
			end
		end
	end
	
	if speed < STOPPED_SPEED then
		return true, Hazard.Stopped, lights
	elseif speed < SLOW_SPEED then
		return true, Hazard.Slow, lights
	end
end

local function enableHazard(vehicle_id, kind, lights)
	local vehicle = VEHICLES[vehicle_id]
	vehicle.is_hazard = true
	
	local check = tableCopy(vehicle.lights)
	
	for obj_id, distance in pairs(lights) do
		if not vehicle.lights[obj_id] or vehicle.hazard_kind ~= kind then
			vehicle.lights[obj_id] = distance
			addLightState(obj_id, vehicle_id, kind, vehicle.hazard_kind)
		end
		
		check[obj_id] = nil
	end
	
	-- auto throw away those that are out of range
	for obj_id, _ in pairs(check) do
		addLightState(obj_id, vehicle_id, 0, vehicle.hazard_kind)
		vehicle.lights[obj_id] = nil
	end
	
	vehicle.hazard_kind = kind
end

local function disableHazard(vehicle_id)
	local vehicle = VEHICLES[vehicle_id]
	if not vehicle.is_hazard then return end
	vehicle.is_hazard = false
	
	for obj_id, _ in pairs(vehicle.lights) do
		vehicle.lights[obj_id] = nil
		addLightState(obj_id, vehicle_id, 0, vehicle.hazard_kind)
	end
	vehicle.hazard_kind = 0
end

local function checkHazardJob(job)
	while IS_ENABLED do
		local vehicles = {} -- must copy veh table
		for index, vehicle in ipairs(getAllVehicles()) do
			vehicles[index] = vehicle:getId()
		end
		if #vehicles == 0 then job.yield() end
		
		if LIGHT_OVERWRITE ~= nil then
			job.yield()
		else
			for _, veh_id in ipairs(vehicles) do
				 -- must do it like this or we can run into a racing condition where any of these vehicles are removed while we are parsing them
				local vehicle = getObjectByID(veh_id)
				if vehicle then
					if not vehicle:getActive() then
						disableHazard(veh_id)
						
					else
						local is_hazard, kind, lights = checkHazard(vehicle)
						if not is_hazard then
							disableHazard(veh_id)
							
						else
							enableHazard(veh_id, kind, lights)
						end
					end
					job.yield()
				end
			end
		end
	end
end

-- -------------------------------------------------------------------------------
-- Load / Unload
local function loadPosMap(pos_map, lights)
	local pos_map = PosMapC():fromJson(jsonReadFile(pos_map))
	local path = pos_map:getPath()
	
	-- prepare light line and see which light is close to each position
	local light_line = {}
	for _, point in path:ipairs() do
		local pos = point.pos
		local distance = point.value.distance
		local new_lights = {}
		
		for _, light in ipairs(lights) do
			if dist2d(pos, light:getPosition()) < LIGHT_RANGE then
				new_lights[light:getId()] = distance
				
				local class_name = light:getClassName()
				if class_name == 'SpotLight' or class_name == 'PointLight' then
					light.isEnabled = true
					
				else
					local color = Point4F(0, 1, 0, 1)
					light.instanceColor = color
					light.instanceColor1 = color
					light.instanceColor2 = color
					light:postApply()
				end
			end
		end
		
		table.insert(light_line, {
			pos = pos,
			distance = point.value.distance,
			lights = new_lights,
			point = point
		})
	end
	
	-- sort by distance
	table.sort(light_line, function(x, y) return x.distance < y.distance end)
	
	-- merge lights so that each pos contains the lights from x meters behind and then merge that into the point value
	for index = #light_line, 1, -1 do
		local line = light_line[index]
		for index2 = index - 1, 1, -1 do
			local c_line = light_line[index2]
			if (line.distance - c_line.distance) > HAZARD_RANGE then break end
			
			-- merge light into ours
			for light, _ in pairs(c_line.lights) do
				line.lights[light] = c_line.distance
			end
		end
		
		line.point.value.lights = line.lights
	end
	
	table.insert(POS_MAPS, pos_map)
end

local function init()
	local root_group = scenetree[ROOT_GROUP]
	if root_group == nil then
		log('E', 'Hazard', 'No scentree group or prefab with the name "' .. ROOT_GROUP .. '"')
		return
	end
	
	local class_name = root_group:getClassName()
	if class_name ~= "SimGroup" then
		if class_name == "Prefab" then
			root_group = root_group:getChildGroup()
		else
			log('E', 'Hazard', 'No scenetree group or prefab with name "' .. ROOT_GROUP .. '"')
			return
		end
	end
	
	-- learn of all lights and prepare them
	local lights = indexLights(root_group)
	log('I', 'Hazard', 'Found ' .. #lights .. ' lights (SpotLight/PointLight/TSStatic)')
	
	PATH_DIR = string.format(PATH_DIR, core_levels.getLevelName(getMissionFilename()))
	if not FS:directoryExists(PATH_DIR) then FS:directoryCreate(PATH_DIR) end
	
	local pos_maps = getFileListWithExtensions(PATH_DIR, 'posmapc')
	for _, pos_map in ipairs(pos_maps) do
		log('I', 'Hazard', 'Loading pos map ' .. fileName(pos_map))
		loadPosMap(
			pos_map,
			lights
		)
	end
	
	if AddEventHandler then
		log('I', 'Hazard', 'Found BeamMP, adding remote listeners')
		AddEventHandler('hazard_lightoverwrite', M.lightOverwrite)
	end
	
	--hotreload
	for _, vehicle in ipairs(getAllVehicles()) do
		M.onVehicleSpawned(vehicle:getId())
	end
	
	core_jobsystem.create(checkHazardJob, 0)
	IS_ENABLED = true
end

local function unload()
	POS_MAPS = {}
	IS_ENABLED = false
end

-- -------------------------------------------------------------------------------
-- Api
M.lightOverwrite = function(type)
	local type = tonumber(type)
	for _, vehicle in ipairs(getAllVehicles(0)) do
		disableHazard(vehicle:getId())
	end
	if type == 1 then -- green
		globalLightOverwrite(LIGHT_OVERWRITE)
		LIGHT_OVERWRITE = nil
		
	elseif type == 2 then -- yellow
		globalLightOverwrite(LIGHT_OVERWRITE)
		globalLightOverwrite(Hazard.Slow, true)
		LIGHT_OVERWRITE = Hazard.Slow
		
	elseif type == 3 then -- b yellow
		globalLightOverwrite(LIGHT_OVERWRITE)
		globalLightOverwrite(Hazard.Stopped, true)
		LIGHT_OVERWRITE = Hazard.Stopped
	
	elseif type == 4 then -- b red
		globalLightOverwrite(LIGHT_OVERWRITE)
		globalLightOverwrite(Hazard.Reverse, true)
		LIGHT_OVERWRITE = Hazard.Reverse
	
	end
end

-- -------------------------------------------------------------------------------
-- Game Events
M.onExtensionLoaded = function()
	if worldReadyState == 2 then init() end
end

M.onWorldReadyState = function(state)
	if state == 2 then init() end
end

M.onExtensionUnloaded = unload

M.onVehicleSpawned = function(vehicle_id)
	VEHICLES[vehicle_id] = {
		is_hazard = false,
		hazard_kind = 0,
		lights = {}
	}
end

M.onVehicleDestroyed = function(vehicle_id)
	disableHazard(vehicle_id)
	VEHICLES[vehicle_id] = nil
end

M.onEditorDeactivated = function()
	unload()
	init()
end

return M
