local M = {}
M.dependencies = {"ui_imgui"}

local taillight = {
    ui = {
        windowOpen = nil,
        displayText = nil,
        fontSize = nil,
        scrollSpeed = nil,
        mirrorMode = nil,
        rainbowMode = nil,
        rainbowSpeed = nil,
        textRed = nil,
        textGreen = nil,
        textBlue = nil,
        staticMode = nil,
        textPosition = nil,
        presetName = nil,
        initialized = false
    },
    presets = {
        user = {},
        default = {
            {name = "Default", text = "RedlineAuto", fontSize = 15, mirrorMode = false, color = {255, 0, 0}, scrollSpeed = 60, staticMode = false, textPosition = 0},
            {name = "Mk4 Supra", text = "Mk4 Supra", fontSize = 18, mirrorMode = true, color = {36, 0, 0}, scrollSpeed = 60, staticMode = false, textPosition = 0},
            {name = "Subscribe", text = "Subscribe", fontSize = 23, mirrorMode = true, color = {255, 0, 0}, scrollSpeed = 0, staticMode = true, textPosition = 136},
            {
                name = "Drift Missile",
                text = "Drift   Missile",
                fontSize = 27,
                mirrorMode = false,
                color = {255, 0, 0},
                scrollSpeed = 0,
                staticMode = true,
                textPosition = 26,
                rainbowMode = true,
                rainbowSpeed = 1.0
            }
        },
        configPath = "vehicles/RLA_Supra/taillight_presets.json",
        lastStatePath = "vehicles/RLA_Supra/taillight_last_state.json"
    },
    state = {
        lastSentText = "",
        lastSentSize = 0,
        lastSentMirrorMode = false,
        lastSentRGB = {255, 0, 0},
        lastSentRainbow = false,
        lastSentRainbowSpeed = 0.5,
        lastSentStaticMode = false,
        lastSentTextPosition = 0
    }
}

local im = ui_imgui

local function initializeUI()
    taillight.ui.windowOpen = im.BoolPtr(false)
    taillight.ui.displayText = im.ArrayChar(256, "RedlineAuto")
    taillight.ui.fontSize = im.FloatPtr(15)
    taillight.ui.scrollSpeed = im.FloatPtr(60)
    taillight.ui.mirrorMode = im.BoolPtr(false)
    taillight.ui.rainbowMode = im.BoolPtr(false)
    taillight.ui.rainbowSpeed = im.FloatPtr(0.5)
    taillight.ui.textRed = im.FloatPtr(255)
    taillight.ui.textGreen = im.FloatPtr(0)
    taillight.ui.textBlue = im.FloatPtr(0)
    taillight.ui.staticMode = im.BoolPtr(false)
    taillight.ui.textPosition = im.FloatPtr(0)
    taillight.ui.presetName = im.ArrayChar(256, "My Preset")
end

local function ensureDirectoryExists()
    local directoryPath = "vehicles/RLA_Supra"
    if FS and not FS:directoryExists(directoryPath) then
        local success = FS:directoryCreate(directoryPath, true)
    end
end

local function loadPresets()
    local success, presetData = pcall(function()
        return jsonReadFile(taillight.presets.configPath)
    end)
    if success and presetData then
        taillight.presets.user = presetData
    else
        taillight.presets.user = {}
    end
end

local function savePresets()
    local success = pcall(function()
        jsonWriteFile(taillight.presets.configPath, taillight.presets.user, true)
    end)
end

local getCurrentPresetData, applyPreset

local function saveLastState()
    local lastState = getCurrentPresetData()
    local success = pcall(function()
        jsonWriteFile(taillight.presets.lastStatePath, lastState, true)
    end)
end

local function loadLastState()
    local success, lastState = pcall(function()
        return jsonReadFile(taillight.presets.lastStatePath)
    end)
    if success and lastState then
        applyPreset(lastState)
        return true
    else
        return false
    end
end

getCurrentPresetData = function()
    local currentText = ffi.string(taillight.ui.displayText)
    return {
        text = currentText,
        fontSize = taillight.ui.fontSize[0],
        mirrorMode = taillight.ui.mirrorMode[0],
        color = {
            math.floor(taillight.ui.textRed[0]),
            math.floor(taillight.ui.textGreen[0]),
            math.floor(taillight.ui.textBlue[0])
        },
        scrollSpeed = taillight.ui.scrollSpeed[0],
        staticMode = taillight.ui.staticMode[0],
        textPosition = taillight.ui.textPosition[0],
        rainbowMode = taillight.ui.rainbowMode[0],
        rainbowSpeed = taillight.ui.rainbowSpeed[0]
    }
end

applyPreset = function(preset)
    ffi.copy(taillight.ui.displayText, preset.text)
    taillight.ui.fontSize[0] = preset.fontSize
    taillight.ui.mirrorMode[0] = preset.mirrorMode
    taillight.ui.textRed[0] = preset.color[1]
    taillight.ui.textGreen[0] = preset.color[2]
    taillight.ui.textBlue[0] = preset.color[3]
    taillight.ui.scrollSpeed[0] = preset.scrollSpeed
    taillight.ui.staticMode[0] = preset.staticMode
    taillight.ui.textPosition[0] = preset.textPosition
    
    taillight.ui.rainbowMode[0] = preset.rainbowMode or false
    taillight.ui.rainbowSpeed[0] = preset.rainbowSpeed or 0.5

    local veh = be and be.getPlayerVehicle and be:getPlayerVehicle(0) or nil
    if veh then
        pcall(function()
            veh:queueLuaCommand(string.format('electrics.values.scrollSpeed = %f;', taillight.ui.scrollSpeed[0]))
            veh:queueLuaCommand(string.format('electrics.values.staticMode = %s;', tostring(taillight.ui.staticMode[0])))
            veh:queueLuaCommand(string.format('electrics.values.textPosition = %f;', taillight.ui.textPosition[0]))
            veh:queueLuaCommand(string.format('electrics.values.rainbowMode = %s;', tostring(taillight.ui.rainbowMode[0])))
            veh:queueLuaCommand(string.format('electrics.values.rainbowSpeed = %f;', taillight.ui.rainbowSpeed[0]))
        end)
    end
end

local function saveCurrentPreset()
    local newPreset = getCurrentPresetData()
    newPreset.name = ffi.string(taillight.ui.presetName)
    
    local existingIndex = nil
    for i, preset in ipairs(taillight.presets.user) do
        if preset.name == newPreset.name then
            existingIndex = i
            break
        end
    end
    
    if existingIndex then
        taillight.presets.user[existingIndex] = newPreset
    else
        table.insert(taillight.presets.user, newPreset)
    end
    
    savePresets()
end

local function deletePreset(presetName)
    for i, preset in ipairs(taillight.presets.user) do
        if preset.name == presetName then
            table.remove(taillight.presets.user, i)
            savePresets()
            return true
        end
    end
    return false
end

local function sendUpdate(text, size, mirror, red, green, blue, rainbow, rSpeed, static, position)
    local clampedRed = math.max(1, red)
    local clampedGreen = math.max(0, green)
    local clampedBlue = math.max(0, blue)

    if text == taillight.state.lastSentText and 
       size == taillight.state.lastSentSize and 
       mirror == taillight.state.lastSentMirrorMode and 
       clampedRed == taillight.state.lastSentRGB[1] and 
       clampedGreen == taillight.state.lastSentRGB[2] and 
       clampedBlue == taillight.state.lastSentRGB[3] and 
       rainbow == taillight.state.lastSentRainbow and
       rSpeed == taillight.state.lastSentRainbowSpeed and 
       static == taillight.state.lastSentStaticMode and
       position == taillight.state.lastSentTextPosition then
        return
    end

    if be and be.getPlayerVehicle then
        local veh = be:getPlayerVehicle(0)
        if veh then
            local command = 'electrics.values.taillightTextLength = ' .. #text .. '; '
            for i = 0, 5 do
                local chunk = ""
                for j = 1, 5 do
                    local charIndex = i * 5 + j
                    if charIndex <= #text then
                        chunk = chunk .. string.byte(text, charIndex) .. ","
                    else
                        chunk = chunk .. "0,"
                    end
                end
                command = command .. 'electrics.values.taillightText' .. (i + 1) .. ' = "' .. chunk:sub(1, -2) .. '"; '
            end
            command = command .. string.format(
                'electrics.values.taillightFontSize = %d; ' ..
                'electrics.values.mirrorMode = %s; ' ..
                'electrics.values.textRed = %d; ' ..
                'electrics.values.textGreen = %d; ' ..
                'electrics.values.textBlue = %d; ' ..
                'electrics.values.rainbowMode = %s; ' ..
                'electrics.values.rainbowSpeed = %f; ' ..
                'electrics.values.staticMode = %s; ' ..
                'electrics.values.textPosition = %f;',
                size, tostring(mirror), clampedRed, clampedGreen, clampedBlue, 
                tostring(rainbow), rSpeed, tostring(static), position
            )
            
            pcall(function()
                veh:queueLuaCommand(command)
            end)
            
            taillight.state.lastSentText = text
            taillight.state.lastSentSize = size
            taillight.state.lastSentMirrorMode = mirror
            taillight.state.lastSentRGB = {clampedRed, clampedGreen, clampedBlue}
            taillight.state.lastSentRainbow = rainbow
            taillight.state.lastSentRainbowSpeed = rSpeed
            taillight.state.lastSentStaticMode = static
            taillight.state.lastSentTextPosition = position
        end
    end
end

local function renderUI()
    if not taillight.ui.windowOpen[0] then return end
    
    if not taillight.ui.initialized then
        im.SetNextWindowSize(im.ImVec2(350, 0), im.Cond_FirstUseEver)
        taillight.ui.initialized = true
    end
    
    im.SetNextWindowSizeConstraints(im.ImVec2(350, 0), im.ImVec2(350, 1000))
    
    if im.Begin("Taillight Configurator (Made by RedlineAuto)", taillight.ui.windowOpen, im.WindowFlags_AlwaysAutoResize) then
        local needsUpdate = false
        
        im.PushItemWidth(230)
        
        if im.InputText("Display Text", taillight.ui.displayText, 20) then
            local currentText = ffi.string(taillight.ui.displayText)
            if #currentText == 0 then
                ffi.copy(taillight.ui.displayText, "RedlineAuto")
            end
            needsUpdate = true
        end
        if im.SliderFloat("Font Size", taillight.ui.fontSize, 8, 32, "%.0f") then
            needsUpdate = true
        end
        
        if im.Checkbox("Static Mode", taillight.ui.staticMode) then
            local veh = be and be.getPlayerVehicle and be:getPlayerVehicle(0) or nil
            if veh then
                pcall(function()
                    veh:queueLuaCommand(string.format('electrics.values.staticMode = %s;', tostring(taillight.ui.staticMode[0])))
                end)
            end
            needsUpdate = true
        end
        
        if taillight.ui.staticMode[0] then
            if im.SliderFloat("Text Position", taillight.ui.textPosition, 0, 256, "%.0f") then
                local veh = be and be.getPlayerVehicle and be:getPlayerVehicle(0) or nil
                if veh then
                    pcall(function()
                        veh:queueLuaCommand(string.format('electrics.values.textPosition = %f;', taillight.ui.textPosition[0]))
                    end)
                end
                needsUpdate = true
            end
        else
            if im.SliderFloat("Scroll Speed", taillight.ui.scrollSpeed, 0, 300, "%.0f") then
                local veh = be and be.getPlayerVehicle and be:getPlayerVehicle(0) or nil
                if veh then
                    pcall(function()
                        veh:queueLuaCommand(string.format('electrics.values.scrollSpeed = %f;', taillight.ui.scrollSpeed[0]))
                    end)
                end
            end
        end
        
        im.Dummy(im.ImVec2(0, 10))
        
        if im.Checkbox("Rainbow Mode", taillight.ui.rainbowMode) then
            needsUpdate = true
        end
        
        if taillight.ui.rainbowMode[0] then
            if im.SliderFloat("Rainbow Speed", taillight.ui.rainbowSpeed, 0.1, 2.0, "%.1f") then
                needsUpdate = true
            end
        end
        
        if not taillight.ui.rainbowMode[0] then
            im.Text("Text Color:")
            if im.SliderFloat("Red", taillight.ui.textRed, 0, 255, "%.0f") then
                needsUpdate = true
            end
            if im.SliderFloat("Green", taillight.ui.textGreen, 0, 255, "%.0f") then
                needsUpdate = true
            end
            if im.SliderFloat("Blue", taillight.ui.textBlue, 0, 255, "%.0f") then
                needsUpdate = true
            end
        end
        
        im.PopItemWidth()
        
        im.Dummy(im.ImVec2(0, 5))
        if im.Checkbox("Mirror", taillight.ui.mirrorMode) then
            local veh = be and be.getPlayerVehicle and be:getPlayerVehicle(0) or nil
            if veh then
                pcall(function()
                    veh:queueLuaCommand(string.format('electrics.values.mirrorMode = %s;', tostring(taillight.ui.mirrorMode[0])))
                end)
            end
            needsUpdate = true
        end
        
        im.Separator()
        im.Text("Save Current Settings as Preset:")
        im.PushItemWidth(200)
        im.InputText("Preset Name", taillight.ui.presetName, 30)
        im.PopItemWidth()
        
        if im.Button("Save Preset") then
            saveCurrentPreset()
        end
        
        im.Separator()
        im.Text("Default Presets:")
        
        im.Columns(2, "presetColumns", false)
        for i, preset in ipairs(taillight.presets.default) do
            if i > 1 and i % 2 == 1 then
                im.NextColumn()
            end
            
            if im.Button(preset.name .. "##default" .. i) then
                applyPreset(preset)
                needsUpdate = true
            end
            
            if i % 2 == 0 then
                im.NextColumn()
            end
        end
        im.Columns(1)
        
        if #taillight.presets.user > 0 then
            im.Separator()
            im.Text("User Presets:")
            im.Columns(2, "userPresetColumns", false)
            for i, preset in ipairs(taillight.presets.user) do
                if i > 1 and i % 2 == 1 then
                    im.NextColumn()
                end
                
                local buttonWidth = im.GetContentRegionAvailWidth() * 0.75
                if im.Button(preset.name .. "##user" .. i, im.ImVec2(buttonWidth, 0)) then
                    applyPreset(preset)
                    needsUpdate = true
                end
                
                im.SameLine()
                if im.Button("X##" .. i, im.ImVec2(20, 0)) then
                    deletePreset(preset.name)
                end
                
                if i % 2 == 0 then
                    im.NextColumn()
                end
            end
            im.Columns(1)
        end
        
        im.End()
        if needsUpdate then
            local currentText = ffi.string(taillight.ui.displayText)
            sendUpdate(currentText, taillight.ui.fontSize[0], taillight.ui.mirrorMode[0], 
                      math.floor(taillight.ui.textRed[0]), math.floor(taillight.ui.textGreen[0]), math.floor(taillight.ui.textBlue[0]),
                      taillight.ui.rainbowMode[0], taillight.ui.rainbowSpeed[0], taillight.ui.staticMode[0], taillight.ui.textPosition[0])
        end
    end
end

local function onUpdate(dtReal)
    if taillight.ui and taillight.ui.windowOpen and taillight.ui.windowOpen[0] then
        pcall(renderUI)
    end
end

local function onExtensionLoaded()
    initializeUI()
    pcall(ensureDirectoryExists)
    pcall(loadPresets)
    
    if not loadLastState() then
        local currentText = ffi.string(taillight.ui.displayText)
        sendUpdate(currentText, taillight.ui.fontSize[0], taillight.ui.mirrorMode[0], 
                  math.floor(taillight.ui.textRed[0]), math.floor(taillight.ui.textGreen[0]), math.floor(taillight.ui.textBlue[0]),
                  taillight.ui.rainbowMode[0], taillight.ui.rainbowSpeed[0], taillight.ui.staticMode[0], taillight.ui.textPosition[0])
    end
end

local function onExtensionUnloaded()
    saveLastState()
end

local function onReset()
    saveLastState()
end

local function toggle()
    taillight.ui.windowOpen[0] = not taillight.ui.windowOpen[0]
end

M.onUpdate = onUpdate
M.onExtensionLoaded = onExtensionLoaded
M.onExtensionUnloaded = onExtensionUnloaded
M.onReset = onReset
M.toggle = toggle

return M