local M = {}

local front_sfx_table = {}
local rear_sfx_table = {}
local compressor_sfx_table = {}
local current_front_sfx_playing = nil
local current_rear_sfx_playing = nil
local current_compressor_sfx_playing = nil
local current_front_sound_type = nil
local current_rear_sound_type = nil

local frontNodeName = "f18"
local rearNodeName = "rdiff"
local compressorNodeName = "rx6"

local frontNode = 0
local rearNode = 0
local compressorNode = 0

local SOUNDS = {
    up = "/vehicles/RLA_Supra/sounds/AirUp.mp3",
    down = "/vehicles/RLA_Supra/sounds/AirDown.mp3",
    compressor = "/vehicles/RLA_Supra/sounds/CompressorLoop.mp3"
}

local function resolveNodeIDs(vehicleData)
    local v = vehicleData or _G.v
    local nodeNameIdx = {}
    if v and v.data and v.data.nodes then
        for _, node in pairs(v.data.nodes) do
            if node.name then
                nodeNameIdx[node.name] = node.cid
            end
        end
    else
        log("W", "soundManager.resolveNodeIDs", "Vehicle node data not available, using default node IDs (0)")
    end

    frontNode = nodeNameIdx[frontNodeName] or 0
    rearNode = nodeNameIdx[rearNodeName] or 0
    compressorNode = nodeNameIdx[compressorNodeName] or 0
    return true
end

local function sanitizeObjectName(path)
    return path:gsub("/", "_")
end

local function loadFrontSound(sound_path)
    if sound_path and not front_sfx_table[sound_path] then
        local safe_name = sanitizeObjectName("front_" .. sound_path)
        front_sfx_table[sound_path] = obj:createSFXSource(sound_path, "AudioDefault3D", safe_name, frontNode)
        return true
    end
    return front_sfx_table[sound_path] ~= nil
end

local function loadRearSound(sound_path)
    if sound_path and not rear_sfx_table[sound_path] then
        local safe_name = sanitizeObjectName("rear_" .. sound_path)
        rear_sfx_table[sound_path] = obj:createSFXSource(sound_path, "AudioDefault3D", safe_name, rearNode)
        return true
    end
    return rear_sfx_table[sound_path] ~= nil
end

local function loadCompressorSound(sound_path)
    if sound_path and not compressor_sfx_table[sound_path] then
        local safe_name = sanitizeObjectName("compressor_" .. sound_path)
        compressor_sfx_table[sound_path] = obj:createSFXSource(sound_path, "AudioDefaultLoop3D", safe_name, compressorNode)
        return true
    end
    return compressor_sfx_table[sound_path] ~= nil
end

function M.playFrontSound(sound_path)
    if sound_path and front_sfx_table[sound_path] then
        if current_front_sfx_playing then
            obj:stopSFX(current_front_sfx_playing)
            current_front_sfx_playing = nil
            current_front_sound_type = nil
        end
        
        local sfx_to_play = front_sfx_table[sound_path]
        obj:setVolumePitch(sfx_to_play, 3, 1)
        obj:playSFX(sfx_to_play)
        current_front_sfx_playing = sfx_to_play
        current_front_sound_type = sound_path
    end
end

function M.playRearSound(sound_path)
    if sound_path and rear_sfx_table[sound_path] then
        if current_rear_sfx_playing then
            obj:stopSFX(current_rear_sfx_playing)
            current_rear_sfx_playing = nil
            current_rear_sound_type = nil
        end
        
        local sfx_to_play = rear_sfx_table[sound_path]
        obj:setVolumePitch(sfx_to_play, 3, 1)
        obj:playSFX(sfx_to_play)
        current_rear_sfx_playing = sfx_to_play
        current_rear_sound_type = sound_path
    end
end

function M.playCompressorSound(sound_path)
    if sound_path and compressor_sfx_table[sound_path] then
        if not current_compressor_sfx_playing then
            local sfx_to_play = compressor_sfx_table[sound_path]
            obj:setVolumePitch(sfx_to_play, 0.5, 1)
            obj:playSFX(sfx_to_play)
            current_compressor_sfx_playing = sfx_to_play
        end
    end
end

function M.stopFrontSound()
    if current_front_sfx_playing then
        obj:stopSFX(current_front_sfx_playing)
        current_front_sfx_playing = nil
        current_front_sound_type = nil
    end
end

function M.stopRearSound()
    if current_rear_sfx_playing then
        obj:stopSFX(current_rear_sfx_playing)
        current_rear_sfx_playing = nil
        current_rear_sound_type = nil
    end
end

function M.stopCompressorSound()
    if current_compressor_sfx_playing then
        obj:stopSFX(current_compressor_sfx_playing)
        current_compressor_sfx_playing = nil
    end
end

function M.stopAllSounds()
    M.stopFrontSound()
    M.stopRearSound()
    M.stopCompressorSound()
end

function M.onExtensionLoaded()
end

function M.init(vehicleData)
    if not resolveNodeIDs(vehicleData) then
        log("E", "soundManager.init", "Failed to resolve node IDs, using defaults")
    end

    loadFrontSound(SOUNDS.up)
    loadFrontSound(SOUNDS.down)
    loadRearSound(SOUNDS.up)
    loadRearSound(SOUNDS.down)
    loadCompressorSound(SOUNDS.compressor)
end

function M.reset(vehicleData)
    M.stopAllSounds()
    
    for _, sfx in pairs(front_sfx_table) do
        obj:deleteSFXSource(sfx)
    end
    front_sfx_table = {}
    
    for _, sfx in pairs(rear_sfx_table) do
        obj:deleteSFXSource(sfx)
    end
    rear_sfx_table = {}
    
    for _, sfx in pairs(compressor_sfx_table) do
        obj:deleteSFXSource(sfx)
    end
    compressor_sfx_table = {}
    
    M.init(vehicleData)
end

function M.destroy()
    M.stopAllSounds()
    
    for _, sfx in pairs(front_sfx_table) do
        obj:deleteSFXSource(sfx)
    end
    front_sfx_table = {}
    
    for _, sfx in pairs(rear_sfx_table) do
        obj:deleteSFXSource(sfx)
    end
    rear_sfx_table = {}
    
    for _, sfx in pairs(compressor_sfx_table) do
        obj:deleteSFXSource(sfx)
    end
    compressor_sfx_table = {}
end

M.SOUNDS = SOUNDS

return M