local M = {}

local soundManager = require('baggedSuspension/soundManager')

local TANK_PSI = {
    MIN = 30,
    MAX = 110,
    MAX_TANK = 185,
    MIN_TANK = 130,
    COMPRESSOR_RATE = 0.1
}

local tankState = {
    currentPsi = TANK_PSI.MAX_TANK,
    compressorActive = false,
    previousPositions = {
        fl = 0,
        fr = 0,
        rl = 0,
        rr = 0
    },
    lastMessageTime = 0
}

local function positionToPsi(position)
    return math.floor((position * (TANK_PSI.MAX - TANK_PSI.MIN)) + TANK_PSI.MIN + 0.5)
end

local function psiToPosition(psi)
    return (psi - TANK_PSI.MIN) / (TANK_PSI.MAX - TANK_PSI.MIN)
end

function M.getFillRateModifier()
    return math.max(0, math.min(1.0, tankState.currentPsi / TANK_PSI.MAX_TANK))
end

function M.getMaxPosition()
    return math.min(1.0, psiToPosition(tankState.currentPsi))
end

function M.updateTankPressure(positions)
    local tankDelta = 0
    
    for pos, values in pairs(positions) do
        local currentPsi = positionToPsi(values.current)
        local prevPsi = positionToPsi(tankState.previousPositions[pos])
        
        if currentPsi > prevPsi then
            tankDelta = tankDelta - (currentPsi - prevPsi) * 0.2
        end
        
        tankState.previousPositions[pos] = values.current
    end
    
    tankState.currentPsi = math.min(TANK_PSI.MAX_TANK, math.max(0, tankState.currentPsi + tankDelta))
    
    local wasCompressorActive = tankState.compressorActive
    
    if tankState.currentPsi < TANK_PSI.MIN_TANK then
        tankState.compressorActive = true
    end
    
    if tankState.compressorActive then
        tankState.currentPsi = math.min(TANK_PSI.MAX_TANK, tankState.currentPsi + TANK_PSI.COMPRESSOR_RATE)
        if tankState.currentPsi >= TANK_PSI.MAX_TANK then
            tankState.compressorActive = false
        end
    end
    
    if tankState.compressorActive and not wasCompressorActive then
        soundManager.playCompressorSound(soundManager.SOUNDS.compressor)
    elseif not tankState.compressorActive and wasCompressorActive then
        soundManager.stopCompressorSound()
    end
    
    electrics.values.tankPsi = tankState.currentPsi
end

function M.checkTankPressure(positions, isFillingAny)
    local currentTime = os.clock()
    if isFillingAny and currentTime - tankState.lastMessageTime > 60 then
        local maxPosition = M.getMaxPosition()
        local tryingToFillFl = positions.fl.current >= maxPosition and positions.fl.current < 1
        local tryingToFillFr = positions.fr.current >= maxPosition and positions.fr.current < 1
        local tryingToFillRl = positions.rl.current >= maxPosition and positions.rl.current < 1
        local tryingToFillRr = positions.rr.current >= maxPosition and positions.rr.current < 1
    end
end

function M.reset()
    tankState.currentPsi = TANK_PSI.MAX_TANK
    tankState.compressorActive = false
    tankState.lastMessageTime = 0
    electrics.values.tankPsi = tankState.currentPsi
    
    for pos in pairs(tankState.previousPositions) do
        tankState.previousPositions[pos] = 0
    end
end

return M