
local DEFAULT_ORB_COLOR = ColorF(1, 1, 1, 1)
local DEFAULT_ORB_OVERLAP_COLOR = ColorF(0, 1, 1, 1)
local DEFAULT_LINE_COLOR = ColorF(0, 1, 1, 1)

local function dist2d(p1, p2)
	return math.sqrt((p2.x - p1.x)^2 + (p2.y - p1.y)^2)
end

local function defaultRender(center, cam_pos, render_distance)
	local overlap = {}
	local side_lines = {l = {}, r = {}}
	for index, path in ipairs(center.path) do
		if overlap[tostring(path.pos)] then
			debugDrawer:drawSphere(path.pos, 0.2, DEFAULT_ORB_OVERLAP_COLOR)
		
		else
			debugDrawer:drawSphere(path.pos, 0.2, DEFAULT_ORB_COLOR)
			overlap[tostring(path.pos)] = true
		
			if center.path[index - 1] then
				local dir = (center.path[index - 1].pos - path.pos):normalized()
				dir:set(-dir.y, dir.x, dir.z)
				table.insert(side_lines.r, path.pos + (dir * path.size))
				table.insert(side_lines.l, path.pos + (-dir * path.size))
				
			elseif center.path[index + 1] then
				local dir = (path.pos - center.path[index + 1].pos):normalized()
				dir:set(-dir.y, dir.x, dir.z)
				table.insert(side_lines.r, path.pos + (dir * path.size))
				table.insert(side_lines.l, path.pos + (-dir * path.size))
			end
		end
	end
	
	
	for index, pos in ipairs(side_lines.l) do
		local p2 = side_lines.l[index - 1]
		if p2 then
			debugDrawer:drawLine(pos, p2, DEFAULT_LINE_COLOR)
		end
	end
	for index, pos in ipairs(side_lines.r) do
		local p2 = side_lines.r[index - 1]
		if p2 then
			debugDrawer:drawLine(pos, p2, DEFAULT_LINE_COLOR)
		end
	end
end

local function build(self)
	local path = self._path
	if not path:get(1) then return end
	
	-- Split the path apart to create render points
	--[[
		x->->->->->->x->->->->->->x->->->->->->x->->->->->->
	]]
	local pre_points = {}
	local center = {pos = path:get(1).pos, path = {}}
	for _, point in path:ipairs() do
		if dist2d(center.pos, point.pos) > self._render_spacing then
			table.insert(pre_points, center)
			center = {pos = point.pos, path = {}}
		end
		
		table.insert(center.path, point)
	end
	
	if #center.path > 0 then
		table.insert(pre_points, center)
	end
	
	-- Merge the points together
	--[[
		<-<-<-<-<-<-x->->->->->-><-<-<-<-<-<-x->->->->->->
	]]
	local points = {}
	for i = 1, #pre_points, 2 do
		local p2 = pre_points[i + 1]
		if p2 == nil then -- happens when uneven
			table.insert(points, pre_points[i])
		else
			local new_p = {pos = pre_points[i].pos, path = pre_points[i].path}
			for _, point in ipairs(p2.path) do
				table.insert(new_p.path, point)
			end
			table.insert(points, new_p)
		end
	end
	
	self._points = points
end

-- this should rather work with a append mode, where it saves the last path._next and only works through the new instead of rebuilding the entire path over and over
local function buildJob(job, self)
	while self._render_job do
		job.yield()
		local path = self._path
		if path:get(1) then
			-- Split the path apart to create render points
			--[[
				x->->->->->->x->->->->->->x->->->->->->x->->->->->->
			]]
			local pre_points = {}
			local center = {pos = path:get(1).pos, path = {}}
			for _, point in path:ipairs() do
				if dist2d(center.pos, point.pos) > self._render_spacing then
					table.insert(pre_points, center)
					center = {pos = point.pos, path = {}}
					
					if (#pre_points % 10) == 0 then job.yield() end
				end
				
				table.insert(center.path, point)
			end
			
			if #center.path > 0 then
				table.insert(pre_points, center)
			end
			
			-- Merge the points together
			--[[
				<-<-<-<-<-<-x->->->->->-><-<-<-<-<-<-x->->->->->->
			]]
			local points = {}
			for i = 1, #pre_points, 2 do
				local p2 = pre_points[i + 1]
				if p2 == nil then -- happens when uneven
					table.insert(points, pre_points[i])
				else
					local new_p = {pos = pre_points[i].pos, path = pre_points[i].path}
					for _, point in ipairs(p2.path) do
						table.insert(new_p.path, point)
					end
					table.insert(points, new_p)
				end
			end
			
			self._points = points
		end
	end
end

return function()
	local render = {
		_render_distance = 0,
		_render_spacing = 0,
		_render = defaultRender,
		_render_job = false,
		_path = {},
		_points = {}
	}
	
	function render:setPath(path)
		self._path = path
		return self
	end
	
	function render:setRender(func)
		self._render = func
		return self
	end
	
	function render:renderDistance(render_distance)
		self._render_distance = render_distance
		self._render_spacing = math.floor(render_distance / 4) -- meters
		return self
	end
	
	function render:build()
		build(self)
		return self
	end
	
	function render:buildJob(state)
		if state == self._render_job then return self end
		
		self._render_job = state
		if state then
			core_jobsystem.create(buildJob, 0, self)
		else
			self:build()
		end
		return self
	end
	
	function render:getPointsInRange() -- for manual render
		local cam_pos = core_camera:getPosition()
		if not cam_pos then return end
		
		local points = {}
		local render_distance = self._render_distance
		for _, center in ipairs(self._points) do
			if dist2d(cam_pos, center.pos) < render_distance then
				table.insert(points, center)
			end
		end
		return points
	end
	
	function render:render()
		local cam_pos = core_camera:getPosition()
		if not cam_pos then return end
		
		local render = self._render
		local render_distance = self._render_distance
		for _, center in ipairs(self._points) do
			if dist2d(cam_pos, center.pos) < render_distance then
				render(center, cam_pos, render_distance)
			end
		end
	end
	
	return render
end
