-- Made by Neverless @ BeamMP. Problems, Questions or requests? Feel free to ask.
-- WIP
local M = {}

-- -------------------------------------------------------------------------------
-- Common
local function compareVec(vec1, vec2)
	if vec1.x ~= vec2.x then return end
	if vec1.y ~= vec2.y then return end
	if vec1.z ~= vec2.z then return end
	return true
end

local function dist2d(p1, p2)
	return math.sqrt((p2.x - p1.x)^2 + (p2.y - p1.y)^2)
end

local function tableVToK(table) -- consuming
	for k, v in ipairs(table) do
		table[v] = true
		table[k] = nil
	end
	return table
end

local function valueCleanse(table, blacklist)
	if type(table) ~= "table" then return table end
	tableVToK(blacklist)
	local copy = {}
	for k, v in pairs(table) do
		if not blacklist[k] then
			copy[k] = v
		end
	end
	return copy
end

-- -------------------------------------------------------------------------------
-- Class
local function pathIter(path, index)
	index = index + 1
	if index < path._next then
		return index, path._points[index]
	end
end

local function new()
	--[[
		Format
		[_points] = table
			[1..n] = table
				[pos] = vec3
				[size] = float
				[value] = nil/any
		[_next] = int
	]]
	local path = {
		_points = {},
		_next = 1,
		_locked = false
	}
	
	function path:add(pos, size, opt_value)
		if self._locked then return self end
		local next = self._next
		self._points[next] = {
			pos = pos,
			size = size or 1,
			value = opt_value
		}
		self._next = next + 1
		return self
	end
	
	function path:remove(index, pos)
		if self._locked then return self end
		local index = index or self:find(pos)
		if not index then return self end
		
		self._points[index] = nil
		
		if index == (self._next - 1) then self._next = index else
			self = self:fix()
		end
		return self
	end
	
	-- ---------------------------------------------------------------------------------
	-- Lock/Unlock. Will only lock add and removals
	function path:setLock(state)
		self._locked = state
		return self
	end
	
	-- ---------------------------------------------------------------------------------
	-- Setters
	function path:setSize(index, size) self._points[index] = size end
	function path:setValue(index, value) self._points[inde] = value end
	function path:reset()
		self._points = {}
		self._next = 1
		return self
	end
	
	-- ---------------------------------------------------------------------------------
	-- Getters
	function path:get(index) return self._points[index] end -- editable
	function path:getLast() return self._points[self._next - 1] end
	function path:getPos(index) return vec3(self._points[index].pos) end
	function path:getPosAsRef(index) return self._points[index].pos end -- editable
	function path:getSize(index) return self._points[index].size end
	function path:getValue(index) return self._points[index].value end -- editable
	function path:getCount() return self._next - 1 end
	
	-- ---------------------------------------------------------------------------------
	-- File serialization
	function path:toJson()
		local data = {}
		for index, point in self:ipairs() do
			data[index] = {
				{
					point.pos.x,
					point.pos.y,
					point.pos.z
				},
				point.size,
				point.value
			}
		end
		return jsonEncode(data)
	end
	
	function path:fromJson(data)
		if self._locked then return self end
		if type(data) ~= "table" then data = jsonDecode(data) end
		for _, point in ipairs(data) do
			self:add(
				vec3(
					point[1][1],
					point[1][2],
					point[1][3]
				),
				point[2],
				point[3]
			)
		end
		return self
	end
	
	-- ---------------------------------------------------------------------------------
	-- VM crossing
	function path:serialize() -- this can properly deserialize the point values, something that the file serializer ignores
		return serialize(self._points)
	end
	
	function path:fromSerialized(serialized)
		if self._locked then return self end
		self._points = deserialize(serialized)
		self._next = #self._points
		return self
	end

	-- ---------------------------------------------------------------------------------
	-- Util
	function path:find(pos) -- slow
		for index, point in self:ipairs() do
			if compareVec(point.pos, pos) then return index end
		end
		return
	end
	
	function path:contains(pos)
		return self:find(pos) ~= nil
	end
	
	function path:smooth(spacing)
		if self._locked then return self end
		local points = self._points
		local size = self._next - 1
		self:reset()
		local start_pos = points[1].pos
		local index = 2
		while true do
			-- find all points close to start pos
			local point_vecs = {}
			local c_index = index
			while c_index <= size do
				local point = points[c_index]
				
				if dist2d(start_pos, point.pos) < spacing then
					table.insert(point_vecs, point)
				else
					break
				end
				c_index = c_index + 1
			end
			index = c_index + 1
			
			if index >= size then
				break
			end
			
			-- pick furthest and create dir
			local f_point = point_vecs[#point_vecs] or points[index]
			local f_dir = (f_point.pos - start_pos):normalized()
			
			-- create next path point
			start_pos = start_pos + (f_dir * spacing)
			self:add(start_pos, f_point.size, f_point.value)
		end
		
		return self
	end
	
	function path:overlapC(func) -- overlap as in, add points from the start to the end until condition is met
		if self._locked then return self end
		local final_point = self._points[self._next - 1]
		for index, point in self:ipairs() do
			if func(point, final_point) then
				self:add(point.pos, point.size, point.value)
			else
				break
			end
		end
	end
	
	function path:overlap(meters)
		if self._locked then return self end
		local final_point = self._points[self._next - 1]
		for index, point in self:ipairs() do
			if dist2d(point.pos, final_point.pos) < meters then
				self:add(point.pos, point.size, point.value)
			else
				break
			end
		end
	end
	
	function path:collectC(func)
		local collect = {}
		for _, point in self:ipairs() do
			if func(point) then
				table.insert(collect, point)
			end
		end
		return collect
	end
	
	--[[
		To relate directly attached points to each other
			func(previous_point, current_point, next_point)
		previous_point is nil on first point
		next_point is nil on last point.
		
		Eg. to measure the distance to each other, or to calc the dir between them
	]]
	function path:relateC(func)
		for index = 1, self._next - 1 do
			func(self._points[index - 1], self._points[index], self._points[index + 1])
		end
		return self
	end
	
	-- adds to the filtered whatever your func returns
	function path:filterC(func)
		local filtered = {}
		for _, point in self:ipairs() do
			local r = func(point)
			if r then table.insert(filtered, r) end
		end
		return filtered
	end
	
	-- eg for checking of value or a element of it exists
	function path:valueC(func)
		for _, point in self:ipairs() do
			point.value = func(point) or point.value
		end
		return self
	end
	
	function path:pointC(func)
		for _, point in self:ipairs() do
			func(point)
		end
		return self
	end
	
	-- for pos edit / convert. eg float to int or 3d to 2d. (pos must stay vec3!)
	function path:posC(func)
		for _, point in self:ipairs() do
			func(point.pos)
		end
		return self
	end
	
	function path:ipairs(start) -- for index, pos in path:ipairs() do
		return pathIter, self, (start or 1) - 1
	end
	
	function path:fix() -- fixes a holed path without creating a copy
		if self._locked then return self end
		local index = 0
		local size = self._next - 1
		while index < size do
			index = index + 1
			if self._points[index] == nil then -- found a hole
				for index = index, size + 1 do
					self._points[index] = self._points[index + 1]
				end
				size = size - 1
			end
		end
		self._next = size + 1
		return self
	end
	
	function path:copy()
		local path = new()
		for index = 1, self._next do
			local point = self._points[index]
			if point then
				path:add(point.pos, point.size, point.value)
			end
		end
		return path
	end
	
	function path:deepcopy()
		local path = new()
		for index = 1, self._next do
			local point = self._points[index]
			if point then
				path:add(vec3Copy(point.pos), point.size, deepcopy(point.value))
			end
		end
		return path
	end
	
	return path
end

setmetatable(M, {
	__call = new
})

return M
