-- Sergio 'Sopze' del Pino @ 2024 -- The Junker radar screen

local M = {}

local htmlTexture = require("htmlTexture")

M.type = "sopzeHTMLtexture"
M.relevantDevice = nil

local _READYSTATE_NIL= 0
local _READYSTATE_OFF= 1
local _READYSTATE_SETUP= 2
local _READYSTATE_RUN= 3

local _BROKENSTATE_NORMAL= 0
local _BROKENSTATE_DAMAGED= 1
local _BROKENSTATE_BROKEN= 2

local radarData = {
  readyState= _READYSTATE_NIL,
  brokenState= _BROKENSTATE_NORMAL,
  texSize= 512,
  spawnPoint= {},
}

local updateData = {
  animProgress= 0.0,
  playerCoords= {},
}

local jbeamConnections = {}
local radarLight= nil
local radarLightFailure= nil

local fps= 30
local fUpdate = 1 / fps
local fTime = 0

local hTime = 0
local vuUpdate = 1 / 5

local animSpeed= 2.25

local function tablesize(T)
  local s = 0
  for _ in pairs(T) do s = s + 1 end
  return s
end

local function updateLightState()
  local broken= radarData.brokenState
  local running= radarData.readyState > _READYSTATE_OFF
  if radarLight then electrics.values[radarLight]= (broken == _BROKENSTATE_NORMAL and running) and 1 or 0 end
  if radarLightFailure then electrics.values[radarLightFailure]= (broken == _BROKENSTATE_DAMAGED and running) and 1 or 0 end
end

local function setRadarState(newReady, newBroken)

  if radarData.brokenState ~= newBroken and newBroken == _BROKENSTATE_BROKEN then
    gui.message({txt = "Radar broken", context = {}}, 4, "info")
  end

  radarData.readyState= newReady
  radarData.brokenState= newBroken
  updateLightState()
  htmlTexture.call("@junker_radar", "radar.updateState", {newReady, newBroken})
end

local function updateHealthBeams()
  for _,c in pairs(jbeamConnections) do
    if tablesize(c)== 0 then
      setRadarState(radarData.readyState, _BROKENSTATE_BROKEN)
      return
    elseif radarData.brokenState < _BROKENSTATE_BROKEN then
      for i,bid in pairs(c) do
        if obj:beamIsBroken(bid) then
          table.remove(c,i)
          setRadarState(radarData.readyState, _BROKENSTATE_DAMAGED)
          return
        end
      end
    end
  end
end

local function update(dt)

  if radarData.readyState == _READYSTATE_NIL then return end

  -- update heavy tasks
  hTime = hTime + dt*bullettime.get()
  if hTime > vuUpdate and radarData.readyState == _READYSTATE_RUN then

    obj:queueGameEngineLua("extensions.junkerUtils.getAllObjectLocations2D("..tostring(objectId)..")")
    
    if radarData.brokenState < _BROKENSTATE_BROKEN then
      updateHealthBeams()
    end
    
    hTime = 0
  end

  -- update radar
  fTime = fTime + dt*bullettime.get()
  if fTime > fUpdate then

    if radarData.readyState == _READYSTATE_RUN then

      -- update player position
      local position= obj:getPosition()
      updateData.playerCoords= {x= position.x, y= position.y, rot= math.deg(obj:getDirection())}

      -- animation things
      updateData.animProgress= updateData.animProgress + dt*animSpeed
      if updateData.animProgress > 1.0 then
        updateData.animProgress= updateData.animProgress - 1.0
      end

      if electrics.values.ignitionLevel< 2 then setRadarState(_READYSTATE_OFF, radarData.brokenState) end

    elseif radarData.readyState == _READYSTATE_SETUP then

      local spawn= obj:getPosition()
      radarData.spawnPoint= {x= spawn.x, y= spawn.y}
      htmlTexture.call("@junker_radar", "radar.init", radarData)
      setRadarState(_READYSTATE_RUN, radarData.brokenState)
      
    elseif radarData.readyState == _READYSTATE_OFF then
      if electrics.values.ignitionLevel== 2 then setRadarState(_READYSTATE_SETUP, radarData.brokenState) end
    end

    fTime = 0
    htmlTexture.call("@junker_radar", "radar.update", updateData)
  end
end

local function initRadarData()
  local jbeamData= v.data["junkerRadar"] or {}

  --print("--------- tests")
  --for match in string.gmatch(serialize(beamstate),'[^,]+') do
  --  if string.find(match, "=nil") then print(match) end
  --end

  -- connections
  jbeamConnections= {}
  if jbeamData.connections then
    for i,v in pairs(jbeamData.connections) do
      jbeamConnections[v]= {}
    end
  end

  for _,beam in pairs(v.data.beams) do
    if beam.name and beam.name ~= "" and jbeamConnections[beam.name] ~= nil then
      table.insert(jbeamConnections[beam.name], beam.cid)
    end
  end
  
  --print("--------- tests")
  --for match in string.gmatch(serialize(jbeamConnections),'[^,]+') do
  --  print(match)
  --end

  -- lights
  if jbeamData.lights then
    radarLight= jbeamData.lights[1]
    radarLightFailure= jbeamData.lights[2]
  end

  setRadarState(_READYSTATE_OFF, _BROKENSTATE_NORMAL)
end

local function init()

  if radarData.readyState == _READYSTATE_NIL then
    htmlTexture.create("@junker_radar", "local://local/vehicles/common/spz_hl2_junker_radar/radar_screen.html", radarData.texSize, radarData.texSize, fps, nil)
  end

  initRadarData()
end

local function reset()
  initRadarData()
end

M.init      = init
M.reset     = reset
M.updateGFX = update

function updateObjectLocations(objectsListStr)
  htmlTexture.call("@junker_radar", "radar.updateObjects", objectsListStr)
end

return M