-- Sergio 'Sopze' del Pino @ 2024 -- The Junker dynamic engine props

local M = {}
M.type = "sopzeController"
M.defaultOrder = 1500
M.relevantDevice = nil

local powertrain = require("powertrain")

local junkerBeamsRaw= {}
local junkerData = {}

local updateCount = 0
local defaultFPS= 8
local fastFPS= 16
local updateFPS = defaultFPS
local stepUpdate = 1 / updateFPS

local gaugesCount = 0
local gaugesFPS = 15
local stepGauges = 1 / gaugesFPS

local isEven= false

local lastThrottle= 0
local lastOutputAV= 0

local oilPressureSmoother = newTemporalSigmoidSmoothing(4, 20, 12, 2, 0)
local fuelPressureSmoother = newTemporalSigmoidSmoothing(3, 30, 19, 3, 0)

local function clamp(num, min, max)
  return math.max(math.min(max, num), min)
end

local function clamp01(num)
  return clamp(num, 0.0, 1.0)
end

local function setUpdateRate(fps)
  updateFPS = fps or defaultFPS
  stepUpdate = 1 / updateFPS
end

local function getMoanFactor(av)
  if av < junkerData.failureRangeAV[2] then
    return clamp01((av-junkerData.failureRangeAV[1]) * junkerData.failureRangeAV[4])
  else
    return clamp01((junkerData.failureRangeAV[3] - av) * junkerData.failureRangeAV[5])
  end
end

-- fake measurement, just for visual purposes (i didnt wanted them to be static)
local function updateStandardGauges(dt)

  if junkerData.arcGaugesBroken and (electrics.values.junker_wheelspeed > 0.0 or electrics.values.junker_rpm > 0.0) then
    electrics.values.junker_wheelspeed= electrics.values.junker_wheelspeed * 0.98
    electrics.values.junker_rpm= electrics.values.junker_rpm * 0.98
  else
    electrics.values.junker_wheelspeed= electrics.values.wheelspeed or 0
    electrics.values.junker_rpm= electrics.values.rpm or 0
  end
end

-- fake measurement, just for visual purposes (i didnt wanted them to be static)
local function updatePressureGauges(dt)

  local fuelraw= 0
  local oilraw= 0

  if electrics.values.rpm > 50 then

    local running= junkerData.device.outputAV1 > 25
    local tEnv = obj:getEnvTemperature() - 273.15

    fuelraw= (running and 18 or 6) + electrics.values.throttle * 1.5 + tEnv * .08
    oilraw= (running and 35 or 27) - electrics.values.oiltemp * .2 + tEnv * .08 + electrics.values.rpm * .0135
  end

  electrics.values.junker_fuelpressure = fuelPressureSmoother:get(fuelraw, electrics.values.junker_fuelpressure >= 12 and dt or dt*4)
  electrics.values.junker_oilpressure = oilPressureSmoother:get(oilraw, electrics.values.junker_oilpressure >= 35 and dt or dt*4)
end

local function updateAirFilter()

  if not junkerData.airFilterBroken then

    if junkerData.device and junkerData.device.outputAV1 > 10 then

      local force= 0

      if junkerData.device.outputAV1 < junkerData.ignitionShakeAV then
        force= (math.random()*1100 + junkerData.device.outputAV1*1.2) * junkerData.ignitionShake
      else
        force= (350*electrics.values.throttle+math.random()*250+150) * junkerData.runningShake
      end
      if math.random() > 0.5 then
        for _, v in pairs(junkerData.beams.filterShake.r) do
          obj:applyForce(v.id2, v.id1, force)
        end
      else
        for _, v in pairs(junkerData.beams.filterShake.l) do
          obj:applyForce(v.id2, v.id1, force)
        end
      end
    end
  end
end

local function updateEngine()
  if junkerData.device and junkerData.device.outputAV1 > 10 then
    
    local force= 0

    if junkerData.device.outputAV1 < junkerData.ignitionShakeAV then

      if updateFPS ~= fastFPS then setUpdateRate(fastFPS) end

      -- ignition shake
      force= (30000 + math.random()*14000 + 4000 * junkerData.device.outputAV1) * junkerData.ignitionShake
      if math.random() > 0.5 then
        for _, v in pairs(junkerData.beams.engineShake) do
          obj:applyForce(v.id1, v.id2, force)
        end
      else
        for _, v in pairs(junkerData.beams.engineShake) do
          obj:applyForce(v.id1, v.id2, -force)
        end
      end
    else

      if updateFPS ~= defaultFPS then setUpdateRate(defaultFPS) end
      
      -- engine moaning
      if junkerData.device.outputAV1 > junkerData.failureRangeAV[1] and junkerData.device.outputAV1 < junkerData.failureRangeAV[3] then
        
        local av= junkerData.device.outputAV1
        local moanFactor= getMoanFactor(av) * .78 * junkerData.moaningFactor

        if lastOutputAV > av then
          moanFactor= moanFactor * .125
        end

        local softFactor= 1.0 - clamp01(((av-lastOutputAV) * 0.0325))
        local value= clamp01(1.0 - moanFactor * softFactor)

        junkerData.device.outputAV1= av * value
      end

      -- running shake
      if electrics.values.throttle - 0.16 >= lastThrottle then
        force= (30000 + math.random()*6000 + 30000 * electrics.values.throttle) * junkerData.runningShake
        for _, v in pairs(junkerData.beams.engineShake) do
          obj:applyForce(v.id1, v.id2, force)
        end
      end

      for _, v in pairs(junkerData.beams.engineShake) do
        obj:applyForce(v.id1, v.id2, -40000*junkerData.device.engineLoad * junkerData.runningShake)
      end

      if math.random() < 0.35 then
        local value = math.random() * junkerData.runningShake
        for _, v in pairs(junkerData.beams.engineShake) do
          obj:applyForce(v.id1, v.id2, 20000 + 10000 * value)
        end
      end
    end
  end
end

local function reset()
  electrics.values.junker_wheelspeed = 0
  electrics.values.junker_rpm = 0
  electrics.values.junker_fuelpressure = 0
  electrics.values.junker_oilpressure = 0

  junkerData.airFilterBroken= false
  lastThrottle = 0
  setUpdateRate(defaultFPS)
end

local function update(dt)

  updateCount = updateCount + dt
  if updateCount > stepUpdate then

    if electrics.values.ignitionLevel> 1 then
      updateAirFilter()
      updateEngine()
    end
    
    lastThrottle= electrics.values.throttle
    lastOutputAV= junkerData.device.outputAV1
    isEven = not isEven
    
    updateCount = 0
  end
  
  updateStandardGauges()

  gaugesCount = gaugesCount + dt
  if gaugesCount > stepGauges then

    updatePressureGauges(dt)
    gaugesCount= 0
  end
end

local function init(jbeamData)

  if jbeamData then
    
    local device = powertrain.getDevice(jbeamData.inputName or v.data.powertrain[0].name or "mainEngine")
    
    if not device then return end

    local rpmToAV = 0.104719755

    junkerBeamsRaw = {
      arcGaugesBeam= jbeamData.arcGaugesBeam,
      airFilterBeam= jbeamData.airFilterBeam,
      engineShakeBeam= jbeamData.engineShakeBeam or {},
      airFilterShakeBeamLeft= jbeamData.airFilterShakeBeamLeft or {},
      airFilterShakeBeamRight= jbeamData.airFilterShakeBeamRight or {}
    }

    junkerData = {
      device= device,
      runningShake= jbeamData.runningShake or 1,
      ignitionShake= jbeamData.ignitionShake or 1,
      moaningFactor= jbeamData.moaningFactor or 1,
      idleAV= device.idleRPM*rpmToAV,
      ignitionShakeAV= device.idleRPM*rpmToAV*0.85,
      failureRangeAV= {1500*rpmToAV, 2200*rpmToAV, 3100*rpmToAV, 1.5/(700*rpmToAV), 1.25/(900*rpmToAV)},
      airFilterBroken= false,
      arcGaugesBroken= false,
      beams= {}
    }

    -- using junker as prefix to prevent conflicts in case they add those measurements to the base game
    electrics.values.junker_oilpressure= 0
    electrics.values.junker_fuelpressure= 0

    --print("----------------------")
    --for match in string.gmatch(serialize(device),'[^,]+') do
    --  print(match)
    --end
  end
end

local function initLastStage()

  local engine = {}
  local filter_right = {}
  local filter_left = {}

  for _, beam in pairs(v.data.beams) do
    if beam.name and beam.name ~="" then
      if junkerBeamsRaw.engineShakeBeam == beam.name then
        table.insert(engine, beam)
      elseif junkerBeamsRaw.airFilterShakeBeamLeft == beam.name then
        table.insert(filter_left, beam)
      elseif junkerBeamsRaw.airFilterShakeBeamRight == beam.name then
        table.insert(filter_right, beam)
      end
    end
  end

  junkerData.beams = {
    engineShake= engine or {},
    filterShake= { r= filter_right or {}, l=filter_left or {} },
    airFilter= junkerBeamsRaw.airFilterBeam,
    arcGauges= junkerBeamsRaw.arcGaugesBeam,
  }

end


local function beamBroke(cid)
  local bname= v.data.beams[cid].name

  if bname then
    if bname == junkerData.beams.airFilter then
      junkerData.airFilterBroken= true
    elseif bname == junkerData.beams.arcGauges then
      junkerData.arcGaugesBroken= true
    end
  end
end


M.init = init
M.initLastStage = initLastStage
M.reset= reset
M.updateGFX = update

M.beamBroken = beamBroke

return M