-- This Source Code Form is subject to the terms of the bCDDL, v. 1.1.
-- If a copy of the bCDDL was not distributed with this
-- file, You can obtain one at http://beamng.com/bCDDL-1.1.txt

local M = {}

M.outputPorts = {[1] = true}
M.deviceCategories = {clutchlike = true, viscouscoupling = true}
M.requiredExternalInertiaOutputs = {1}

local max = math.max
local min = math.min
local abs = math.abs
local sqrt = math.sqrt
local guardZero = guardZero

-- Update velocity function
local function updateVelocity(device, dt)
  device.inputAV = device.parent.outputAV1
end

-- Goose Dump Valve torque update function (GCDV)
local function updateTorque_gooseDV(device, dt)
    local lockupClutchRatio = electrics.values[device.lockupClutchRatioName] or 0
    local stallTorqueRatio = device.stallTorqueRatio

    local inputAV = guardZero(device.inputAV)
    local outputAV1 = guardZero(device.outputAV1)
    local avRatio = outputAV1 / inputAV
    local avDiff = device.inputAV - device.outputAV1
    local maxLockupClutchAngle = device.maxLockupClutchAngle

    device.lockupClutchAngle = min(max(device.lockupClutchAngle + avDiff * dt * 0.25,
                                       -maxLockupClutchAngle * lockupClutchRatio),
                                   maxLockupClutchAngle * lockupClutchRatio)
    local lockupClutchTorque = device.lockupClutchTorque * device.damageLockupClutchTorqueCoef *
                               device.wearLockupClutchTorqueCoef
    local lockupTorque = min(max(device.lockupClutchAngle * device.lockupClutchSpring +
                                 device.lockupClutchDamp * avDiff * lockupClutchRatio,
                                 -lockupClutchTorque), lockupClutchTorque)

    -- Initialize with base stiffness and dump valve offset
    local baseStiffness = device.gooseTable[1] or 0.3
    local dumpValveOffset = device.gooseTable[2] or 0 -- Negative or zero value

    -- Calculate current speed and stiffness
    local currentSpeed = (electrics.values.airspeed or 0) * 2.23694 -- Convert m/s to MPH
    local dumpValveOffSpeed = device.speedThresholds[1] or 10

    local gooseConverterDiameter
    if currentSpeed < dumpValveOffSpeed then
        local interpolationFactor = currentSpeed / dumpValveOffSpeed
        gooseConverterDiameter = baseStiffness + dumpValveOffset * (1 - interpolationFactor)
    else
        gooseConverterDiameter = baseStiffness
    end

    -- Update kFactor calculation
    device.kFactorCoef = device.fluidDensity * gooseConverterDiameter^5
    local kFactor = device.kFactorSmoother:get(
                      -0.004 * device.converterStiffness * (avRatio - 1) /
                      (1 + device.converterStiffness * abs(avRatio - 1)))

    local inputTorque = min(max(kFactor * device.kFactorCoef * inputAV * inputAV *
                                sign(inputAV), -device.converterTorque), device.converterTorque)

    local torqueRatio = min(max(stallTorqueRatio - (stallTorqueRatio - 1) * avRatio /
                                device.couplingAVRatio, 1), stallTorqueRatio)

    device.outputTorque1 = inputTorque * torqueRatio + lockupTorque
    device.torqueDiff = inputTorque + lockupTorque
end

-- Goose by Gear torque update function (GCBG)
local function updateTorque_gooseBG(device, dt)
    local lockupClutchRatio = electrics.values[device.lockupClutchRatioName] or 0
    local stallTorqueRatio = device.stallTorqueRatio

    local inputAV = guardZero(device.inputAV)
    local outputAV1 = guardZero(device.outputAV1)
    local avRatio = outputAV1 / inputAV
    local avDiff = device.inputAV - device.outputAV1
    local maxLockupClutchAngle = device.maxLockupClutchAngle

    device.lockupClutchAngle = min(max(device.lockupClutchAngle + avDiff * dt * 0.25,
                                       -maxLockupClutchAngle * lockupClutchRatio),
                                   maxLockupClutchAngle * lockupClutchRatio)
    local lockupClutchTorque = device.lockupClutchTorque * device.damageLockupClutchTorqueCoef *
                               device.wearLockupClutchTorqueCoef
    local lockupTorque = min(max(device.lockupClutchAngle * device.lockupClutchSpring +
                                 device.lockupClutchDamp * avDiff * lockupClutchRatio,
                                 -lockupClutchTorque), lockupClutchTorque)

    local gooseTable = device.gooseTable
    local currentGear = abs(electrics.values.gearIndex or 1)

    local gooseConverterDiameter = gooseTable[currentGear] or 0.3

    device.kFactorCoef = device.fluidDensity * gooseConverterDiameter^5
    local kFactor = device.kFactorSmoother:get(
                      -0.004 * device.converterStiffness * (avRatio - 1) /
                      (1 + device.converterStiffness * abs(avRatio - 1)))

    local inputTorque = min(max(kFactor * device.kFactorCoef * inputAV * inputAV *
                                sign(inputAV), -device.converterTorque), device.converterTorque)

    local torqueRatio = min(max(stallTorqueRatio - (stallTorqueRatio - 1) * avRatio /
                                device.couplingAVRatio, 1), stallTorqueRatio)

    device.outputTorque1 = inputTorque * torqueRatio + lockupTorque
    device.torqueDiff = inputTorque + lockupTorque
end

-- Goose by Speed torque update function (GCBS)
local function updateTorque_gooseBS(device, dt)
    local lockupClutchRatio = electrics.values[device.lockupClutchRatioName] or 0
    local stallTorqueRatio = device.stallTorqueRatio

    local inputAV = guardZero(device.inputAV)
    local outputAV1 = guardZero(device.outputAV1)
    local avRatio = outputAV1 / inputAV
    local avDiff = device.inputAV - device.outputAV1
    local maxLockupClutchAngle = device.maxLockupClutchAngle

    device.lockupClutchAngle = min(max(device.lockupClutchAngle + avDiff * dt * 0.25,
                                       -maxLockupClutchAngle * lockupClutchRatio),
                                   maxLockupClutchAngle * lockupClutchRatio)
    local lockupClutchTorque = device.lockupClutchTorque * device.damageLockupClutchTorqueCoef *
                               device.wearLockupClutchTorqueCoef
    local lockupTorque = min(max(device.lockupClutchAngle * device.lockupClutchSpring +
                                 device.lockupClutchDamp * avDiff * lockupClutchRatio,
                                 -lockupClutchTorque), lockupClutchTorque)

    local gooseConverterDiameter = device.gooseTable[1] or 0.3

    local currentSpeed = (electrics.values.airspeed or 0) * 2.23694 -- Convert m/s to MPH

    if currentSpeed < device.speedThresholds[1] then
        local nextDiameter = device.gooseTable[2] or gooseConverterDiameter
        local t = currentSpeed / device.speedThresholds[1]
        gooseConverterDiameter = gooseConverterDiameter + (nextDiameter - gooseConverterDiameter) * t
    else
        for i, threshold in ipairs(device.speedThresholds) do
            if currentSpeed >= threshold then
                if i < #device.speedThresholds and currentSpeed < device.speedThresholds[i + 1] then
                    local nextThreshold = device.speedThresholds[i + 1]
                    local nextDiameter = device.gooseTable[i + 2] or device.gooseTable[i + 1]
                    local t = (currentSpeed - threshold) / (nextThreshold - threshold)
                    gooseConverterDiameter = gooseConverterDiameter +
                                             (nextDiameter - gooseConverterDiameter) * t
                else
                    gooseConverterDiameter = device.gooseTable[i + 1] or gooseConverterDiameter
                end
            end
        end
    end

    device.kFactorCoef = device.fluidDensity * gooseConverterDiameter^5
    local kFactor = device.kFactorSmoother:get(
                      -0.004 * device.converterStiffness * (avRatio - 1) /
                      (1 + device.converterStiffness * abs(avRatio - 1)))

    local inputTorque = min(max(kFactor * device.kFactorCoef * inputAV * inputAV *
                                sign(inputAV), -device.converterTorque), device.converterTorque)

    local torqueRatio = min(max(stallTorqueRatio - (stallTorqueRatio - 1) * avRatio /
                                device.couplingAVRatio, 1), stallTorqueRatio)

    device.outputTorque1 = inputTorque * torqueRatio + lockupTorque
    device.torqueDiff = inputTorque + lockupTorque
end

-- Default torque update function (for non-goose converters)
local function updateTorque(device, dt)
    local lockupClutchRatio = electrics.values[device.lockupClutchRatioName] or 0
    local stallTorqueRatio = device.stallTorqueRatio

    local inputAV = guardZero(device.inputAV)
    local outputAV1 = guardZero(device.outputAV1)
    local avRatio = outputAV1 / inputAV
    local avDiff = device.inputAV - device.outputAV1
    local maxLockupClutchAngle = device.maxLockupClutchAngle

    device.lockupClutchAngle = min(max(device.lockupClutchAngle + avDiff * dt * 0.25,
                                       -maxLockupClutchAngle * lockupClutchRatio),
                                   maxLockupClutchAngle * lockupClutchRatio)
    local lockupClutchTorque = device.lockupClutchTorque * device.damageLockupClutchTorqueCoef *
                               device.wearLockupClutchTorqueCoef
    local lockupTorque = min(max(device.lockupClutchAngle * device.lockupClutchSpring +
                                 device.lockupClutchDamp * avDiff * lockupClutchRatio,
                                 -lockupClutchTorque), lockupClutchTorque)

    local kFactor = device.kFactorSmoother:get(
                      -0.004 * device.converterStiffness * (avRatio - 1) /
                      (1 + device.converterStiffness * abs(avRatio - 1)))

    local inputTorque = min(max(kFactor * device.kFactorCoef * inputAV * inputAV *
                                sign(inputAV), -device.converterTorque), device.converterTorque)

    local torqueRatio = min(max(stallTorqueRatio - (stallTorqueRatio - 1) * avRatio /
                                device.couplingAVRatio, 1), stallTorqueRatio)

    device.outputTorque1 = inputTorque * torqueRatio + lockupTorque
    device.torqueDiff = inputTorque + lockupTorque
end

-- Selection of updates for GCDV (Dump Valve)
local function selectUpdates_gooseDV(device)
  device.velocityUpdate = updateVelocity
  device.torqueUpdate = updateTorque_gooseDV
end

-- Selection of updates for GCBG (By Gear)
local function selectUpdates_gooseBG(device)
  device.velocityUpdate = updateVelocity
  device.torqueUpdate = updateTorque_gooseBG
end

-- Selection of updates for GCBS (By Speed)
local function selectUpdates_gooseBS(device)
  device.velocityUpdate = updateVelocity
  device.torqueUpdate = updateTorque_gooseBS
end

-- Selection of updates for default converters (non-goose)
local function selectUpdates(device)
  device.velocityUpdate = updateVelocity
  device.torqueUpdate = updateTorque
end

-- Apply deform group damage
local function applyDeformGroupDamage(device, damageAmount)
  device.damageLockupClutchTorqueCoef = device.damageLockupClutchTorqueCoef -
                                        linearScale(damageAmount, 0, 0.01, 0, 0.05)
  device:calculateInertia()
end

-- Set part condition
local function setPartCondition(device, subSystem, odometer, integrity, visual)
  device.wearLockupClutchTorqueCoef = linearScale(odometer, 30000000, 500000000, 1, 0.2)
  local integrityState = integrity
  if type(integrity) == "number" then
    local integrityValue = integrity
    integrityState = {
      damageLockupClutchTorqueCoef = linearScale(integrityValue, 1, 0, 1, 0)
    }
  end

  device.damageLockupClutchTorqueCoef = integrityState.damageLockupClutchTorqueCoef or 1
  device:calculateInertia()
end

-- Get part condition
local function getPartCondition(device)
  local integrityState = {damageLockupClutchTorqueCoef = device.damageLockupClutchTorqueCoef}
  local integrityValue = linearScale(device.damageLockupClutchTorqueCoef, 1, 0, 1, 0)
  return integrityValue, integrityState
end

-- Validate function
local function validate(device)
  if not device.parent.deviceCategories.engine then
    log("E", "torqueConverter.validate", "Parent device is not an engine device...")
    log("E", "torqueConverter.validate", "Actual parent:")
    log("E", "torqueConverter.validate", powertrain.dumpsDeviceData(device.parent))
    return false
  end

  device.converterTorque = device.converterTorque or
                           (device.parent.torqueData.maxTorque * 1.25 +
                            device.parent.maxRPM * device.parent.inertia * math.pi / 30)
  return true
end

-- Set mode function
local function setMode(device, mode)
  device.mode = mode
  selectUpdates(device)
end

-- Calculate inertia function
local function calculateInertia(device)
  local outputInertia = 0
  local cumulativeGearRatio = 1
  local maxCumulativeGearRatio = 1
  if device.children and #device.children > 0 then
    local child = device.children[1]
    outputInertia = child.cumulativeInertia
    cumulativeGearRatio = child.cumulativeGearRatio
    maxCumulativeGearRatio = child.maxCumulativeGearRatio
  end

  device.cumulativeInertia = outputInertia / device.stallTorqueRatio
  device.lockupClutchSpring = device.lockupClutchSpringBase or
                              (powertrain.stabilityCoef * powertrain.stabilityCoef *
                               device.cumulativeInertia)
  device.lockupClutchDamp = device.lockupClutchDampRatio *
                            sqrt(device.lockupClutchSpring * device.cumulativeInertia)
  device.maxLockupClutchAngle = device.lockupClutchTorque / device.lockupClutchSpring -- rad

  device.cumulativeGearRatio = cumulativeGearRatio * device.stallTorqueRatio
  device.maxCumulativeGearRatio = maxCumulativeGearRatio * device.stallTorqueRatio
end

-- Reset function that reselects the correct update function based on the type of converter
local function reset(device, jbeamData)
  device.cumulativeGearRatio = 1
  device.maxCumulativeGearRatio = 1

  device.outputAV1 = 0
  device.inputAV = 0
  device.outputTorque1 = 0
  device.isBroken = false

  device.lockupClutchAngle = 0
  device.damageLockupClutchTorqueCoef = 1
  device.wearLockupClutchTorqueCoef = 1

  device.isGCDV = jbeamData.isGCDV
  device.isGCBG = jbeamData.isGCBG
  device.isGCBS = jbeamData.isGCBS

  if device.isGCBS then
    device.gooseTable = jbeamData.diam
    device.speedThresholds = jbeamData.speedThresholds
    selectUpdates_gooseBS(device)
    print("Goose Converter By Speed Enabled")
  elseif device.isGCBG then
    device.gooseTable = jbeamData.diam
    selectUpdates_gooseBG(device)
    print("Goose Converter By Gear Enabled")
  elseif device.isGCDV then
    device.gooseTable = jbeamData.diam
    device.speedThresholds = jbeamData.speedThresholds
    selectUpdates_gooseDV(device)
    print("Goose Converter By Dump Valve Enabled")
  else
    selectUpdates(device) -- Default vanilla converter behavior
  end

  return device
end

-- Create a new device based on jbeam data
local function new(jbeamData)
  local device = {
    deviceCategories = shallowcopy(M.deviceCategories),
    requiredExternalInertiaOutputs = shallowcopy(M.requiredExternalInertiaOutputs),
    outputPorts = shallowcopy(M.outputPorts),
    name = jbeamData.name,
    type = jbeamData.type,
    inputName = jbeamData.inputName,
    inputIndex = jbeamData.inputIndex,
    gearRatio = 1,
    additionalEngineInertia = jbeamData.additionalEngineInertia or 0,
    cumulativeGearRatio = 1,
    maxCumulativeGearRatio = 1,
    isPhysicallyDisconnected = true,
    outputAV1 = 0,
    inputAV = 0,
    outputTorque1 = 0,
    torqueDiff = 0,
    damageLockupClutchTorqueCoef = 1,
    wearLockupClutchTorqueCoef = 1,
    isBroken = false,
    reset = reset,
    setMode = setMode,
    validate = validate,
    calculateInertia = calculateInertia,
    applyDeformGroupDamage = applyDeformGroupDamage,
    setPartCondition = setPartCondition,
    getPartCondition = getPartCondition
  }

  device.lockupClutchRatioName = jbeamData.lockupClutchRatioName or "lockupClutchRatio"
  device.lockupClutchAngle = 0

  device.lockupClutchTorque = jbeamData.lockupClutchTorque or 100
  device.lockupClutchSpringBase = jbeamData.lockupClutchSpring
  device.lockupClutchDampRatio = jbeamData.lockupClutchDampRatio or 0.15

  device.couplingAVRatio = jbeamData.couplingAVRatio or 0.85
  device.stallTorqueRatio = jbeamData.stallTorqueRatio or 2
  device.converterStiffness = jbeamData.converterStiffness or 10
  device.converterDiameter = jbeamData.converterDiameter or 0.30
  device.converterTorque = jbeamData.converterTorque
  device.fluidDensity = 844
  device.kFactorSmoother = newExponentialSmoothing(jbeamData.kFactorSmoothing or 75)
  device.kFactorCoef = device.fluidDensity * device.converterDiameter^5

  device.breakTriggerBeam = jbeamData.breakTriggerBeam
  if device.breakTriggerBeam and device.breakTriggerBeam == "" then
    device.breakTriggerBeam = nil
  end

  -- Determine converter type and select appropriate updates
  device.isGCDV = jbeamData.isGCDV
  device.isGCBG = jbeamData.isGCBG
  device.isGCBS = jbeamData.isGCBS

  if device.isGCBS then
    device.gooseTable = jbeamData.diam
    device.speedThresholds = jbeamData.speedThresholds
    selectUpdates_gooseBS(device)
    print("Goose Converter By Speed Enabled")
  elseif device.isGCBG then
    device.gooseTable = jbeamData.diam
    selectUpdates_gooseBG(device)
    print("Goose Converter By Gear Enabled")
  elseif device.isGCDV then
    device.gooseTable = jbeamData.diam
    device.speedThresholds = jbeamData.speedThresholds
    selectUpdates_gooseDV(device)
    print("Goose Converter By Dump Valve Enabled")
  else
    selectUpdates(device) -- Default vanilla converter behavior
  end

  return device
end

M.new = new

return M
