-- This Source Code Form is subject to the terms of the bCDDL, v. 1.1.
-- If a copy of the bCDDL was not distributed with this
-- file, You can obtain one at http://beamng.com/bCDDL-1.1.txt

local M = {}
local logTag = 'chapter_2_4_delivery'
local playerInstance = 'scenario_player0'
local cargoInstance = 'scenario_cargo'
local running = false
local cargoDelivered = false
local cargoFellOff = false
local trailerAttached = false
local displayedRouteInfo = false
local displayedZoomInfo = false
local displayedCouplingInfo = false;
local messageDelayTimer = 0
local autoConnectTrailer = true
local attemptedAutoAttach = false
local attemptTimer = 0

local function reset()
  running = false
  cargoDelivered = false
  cargoFellOff = false
  trailerAttached = false
  displayedRouteInfo = false
  displayedZoomInfo = false
  displayedCouplingInfo = false
  messageDelayTimer = 0
end

local function fail(reason)
  scenario_scenarios.finish({failed = reason})
  reset()
end

local function success(reason)
  scenario_scenarios.finish({msg = reason})
  reset()
end

local function onRaceStart()
  -- log('I', logTag,'onRaceStart called')
  reset()
  scenario_scenarios.trackVehicleMovementAfterDamage(playerInstance, {waitTimerLimit=2})
  running = true

end

local function onRaceResult()
  if cargoDelivered then
    success('scenarios.utah.chapter_2.chapter_2_4_delivery.pass.msg')
  else
    if cargoFellOff then
      fail('scenarios.utah.chapter_2.chapter_2_4_delivery.cargoFellOff.fail.msg')
    elseif not trailerAttached then
      fail('scenarios.utah.chapter_2.chapter_2_4_delivery.trailerNotAttached.fail.msg')
    else
      fail('scenarios.utah.chapter_2.chapter_2_4_delivery.fail.msg')
    end
  end
end

local function onPreRender(dt)
  if not running then
    return
  end

  local playerVehicle = scenetree.findObject(playerInstance)
  local trailer = scenetree.findObject('scenario_trailer')
  local cargo = scenetree.findObject(cargoInstance)

  if playerVehicle and trailer and cargo then
    messageDelayTimer = messageDelayTimer + dt
    if autoConnectTrailer and messageDelayTimer >= 1 then
      ui_message("scenarios.utah.chapter_2.chapter_2_4_delivery.drive_in_reverse.msg", 1, 'CHAPTER_2_4_DELIVERY', nil)
      messageDelayTimer = messageDelayTimer - 1
    elseif not autoConnectTrailer and not displayedCouplingInfo and messageDelayTimer >= 1 then
      ui_message("scenarios.utah.chapter_2.chapter_2_4_delivery.trailer_controls.msg", 4, 'CHAPTER_2_4_DELIVERY', nil)
      messageDelayTimer = messageDelayTimer - 1
      displayedCouplingInfo = true
    elseif not trailerAttached and not autoConnectTrailer and messageDelayTimer >= 1 then
      ui_message("scenarios.utah.chapter_2.chapter_2_4_delivery.attach_trailer.msg", 1, 'CHAPTER_2_4_DELIVERY', nil)
      messageDelayTimer = messageDelayTimer - 1
    elseif not displayedRouteInfo and messageDelayTimer >= 4 then
      ui_message("scenarios.utah.chapter_2.chapter_2_4_delivery.multiple_routes.msg", 8, 'CHAPTER_2_4_DELIVERY', nil)
      displayedRouteInfo = true
    elseif displayedRouteInfo and not displayedZoomInfo and messageDelayTimer >= 8 then
      ui_message("scenarios.utah.chapter_2.chapter_2_4_delivery.zoom_controls.msg", 10, 'CHAPTER_2_4_DELIVERY_ZOOM', nil)
      displayedZoomInfo = true
    end


    local trailerPos = trailer:getPosition()
    local trailerBSphereRadius = trailer:getBSphereRadius()
    local cargoPos = cargo:getPosition()
    -- local cargoRadius = cargo:getBSphereRadius()
    local cargoDistance = (cargoPos - trailerPos):len()
    if cargoDistance > trailerBSphereRadius then
      if not cargoDelivered then
        cargoFellOff = true
        running = false
        scenario_scenarios.endRace()
      end
    end

    local distanceToTrailer = (playerVehicle:getPosition() - trailerPos):len()
    -- log('I', logTag, 'distanceToTrailer: '..distanceToTrailer)
    if autoConnectTrailer and not attemptedAutoAttach and not trailerAttached and distanceToTrailer <= 8.5 then
      -- log('D', logTag,'Auto attaching couplters...')
      playerVehicle:queueLuaCommand('beamstate.toggleCouplers()')
      attemptedAutoAttach = true
      attemptTimer = 0
    end

    if attemptedAutoAttach then
      attemptTimer = attemptTimer + dt
      if not trailerAttached and attemptTimer >= 0.25 and distanceToTrailer >= 11 then
        log('D', logTag,'Auto deattaching couplters...')
        playerVehicle:queueLuaCommand('beamstate.toggleCouplers()')
        attemptedAutoAttach = false
      end
    end

    -- debugDrawer:drawSphere(vec3(trailerPos), 0.05, ColorF(1.0,1.0,1.0,1.0))
    -- debugDrawer:drawSphere(vec3(trailerPos), trailerBSphereRadius, ColorF(1.0,0.0,0.0,0.4))
    -- debugDrawer:drawSphere(vec3(cargoPos), 0.05, ColorF(1.0,1.0,1.0,1.0))
    -- debugDrawer:drawSphere(vec3(cargoPos), cargoRadius, ColorF(0.0,1.0,0.0,0.4))
  end
end

local function onBeamNGTrigger(data)
  local cargo = scenetree.findObject(cargoInstance)
  -- log('I', logTag,'onBeamNGTrigger called...'..dumps(data))

  if data.event == 'enter' then
    if cargo and data.triggerName == 'delivery_zone' then
      if data.subjectID and data.subjectID == cargo:getID() then
        cargoDelivered = true
      end
    end

    if data.triggerName == 'first_junction' then
      messageDelayTimer = 0
      displayedRouteInfo = false
    end
  end
end

local function onRaceWaypointReached(data)
  -- log('I', logTag,'onRaceWaypointReached called - trailerAttached = '..tostring(trailerAttached)..'  '..dumps(data))

  if data.cur == 1 then
    if not trailerAttached then
      running = false
      scenario_scenarios.endRace()
    end
  end
end

local function onVehicleStoppedMoving(vehicleID, damaged)
  if running then
    local playerVehicleID = scenetree.findObject(playerInstance):getID()
    if vehicleID == playerVehicleID and damaged then
      if not cargoDelivered then
       running = false
       scenario_scenarios.endRace()
      end
    end
  end
end

local function onCouplerAttached(objId1, objId2)
  --log('I', logTag,'onCouplerAttached called... objId1: '..objId1..' objId2: '..objId2)
  Engine.Audio.playOnce('AudioGui','event:>Vehicle>Coupler_Attach_2D')

  trailerAttached = true
  messageDelayTimer = 0
  autoConnectTrailer = false
  attemptedAutoAttach = false
end

local function onCouplerDetached(objId1, objId2)
  --log('I', logTag,'onCouplerDetached called... objId1: '..objId1..' objId2: '..objId2)
  Engine.Audio.playOnce('AudioGui','event:>Vehicle>Coupler_Detach_2D')
  trailerAttached = false
  messageDelayTimer = 1
end

M.onRaceStart = onRaceStart
M.onRaceResult = onRaceResult
M.onBeamNGTrigger = onBeamNGTrigger
M.onRaceWaypointReached = onRaceWaypointReached
M.onVehicleStoppedMoving = onVehicleStoppedMoving
M.onPreRender = onPreRender
M.onCouplerAttached = onCouplerAttached
M.onCouplerDetached = onCouplerDetached

return M