-- This Source Code Form is subject to the terms of the bCDDL, v. 1.1.
-- If a copy of the bCDDL was not distributed with this
-- file, You can obtain one at http://beamng.com/bCDDL-1.1.txt

local M = {}
local logTag = 'chapter_2_5_roll'
local playerInstance = 'scenario_player0'
local targetInstance = 'clone0'
local running = false
local startPosition = nil
local lastPosition = nil
local targetDistance = 0
local playerDistance = 0
local finalPlayerDistance = 0
local lastRoad = nil
local playerPath = {}
local playerVisited = {}
local pathToTarget ={}
local maxPoints = 0

local function reset()
  running = false
  startPosition = nil
  lastPosition = nil
  playerDistance = 0
  targetDistance = 0
  finalPlayerDistance = 0
  lastRoad = nil
  playerPath = {}
  pathToTarget = {}
  maxPoints = 0
end

local function fail(reason)
  scenario_scenarios.finish({failed = reason})
  reset()
end

local function success(reason)
  scenario_scenarios.finish({msg = reason})
  reset()
end

local function getClosestRoadNode(position)
  local mapData = map.getMap()
  local first, second, distance = map.findClosestRoad(position)
  local best = nil

  if first and second then
    local node1 = mapData.nodes[first]
    local node2 = mapData.nodes[second]
    local sqrDist1 = (position - node1.pos):squaredLength()
    local sqrDist2 = (position - node2.pos):squaredLength()
    if sqrDist1 < sqrDist2 then
      best = first
    else
      best = second
    end
  end

  return best
end

local function getPathDistance(path)
  local distance = 0
  local pathSize = tableSize(path)

  if pathSize > 1 then
    local mapData = map.getMap()
    for i=1,pathSize - 1 do
      local node1 = mapData.nodes[path[i]]
      local node2 = mapData.nodes[path[i + 1]]
      distance = distance + (node2.pos - node1.pos):length()
    end
  end
  return distance
end

local function distanceAlongLineSeg(lineStartPos, lineEndPos, pos)
  -- dot(dirVec)
  local lineVec = lineEndPos - lineStartPos
  local vecLength = lineVec:dot(lineVec)
  -- log('I', logTag,'vecLength is '..tostring(vecLength))
  if vecLength > 0 then
    local t = (pos - lineStartPos):dot(lineVec) / vecLength
    if t < 0.0 then t = 0.0 end
    if t > 1.0 then t = 1.0 end
    local posOnLine = lineStartPos + t * lineVec
    local distance = (posOnLine - lineStartPos):length()
    return distance, t
  end
  return 0,0
end

local function onScenarioChange(scenario)
  -- log('I', logTag,'onScenarioChange called')
  if scenario.state == 'pre-running' then
    local vehicle = scenetree.findObject(playerInstance)

    if not vehicle then
      return
    end

    local pos = vehicle:getPosition()
    local first, second, distance = map.findClosestRoad(pos)
    lastRoad = nil
    if first then
      lastRoad = {node1=first, node2=second, distance=distance}
    end
  end
end

local function onVehicleStoppedMoving(vehicleID)
  if running then
    log('I', logTag,'onVehicleStoppedMoving called '..vehicleID)
    local playerVehicleID = scenetree.findObject(playerInstance):getID()
    if vehicleID == playerVehicleID then
     --log('I', logTag,'player path: '..finalPlayerDistance..' targetDistance: '..targetDistance)
      -- dump(playerPath)
      scenario_scenarios.endScenario(0)
    end
  end
end

local function getNodeInfront(vehicle, roadNode1, roadNode2)
  local vehicleData = map.objects[vehicle:getID()]
  if not vehicleData then
    return nil
  end

  local vehiclePos = vehicle:getPosition()
  local mapData = map.getMap()
  local posNode1 = mapData.nodes[roadNode1].pos
  local posNode2 = mapData.nodes[roadNode2].pos
  local dirVec = vehicle:getDirectionVector()
  local node1Dot = (posNode1 - vehiclePos):dot(dirVec)
  local node2Dot = (posNode2 - vehiclePos):dot(dirVec)

  if node1Dot > 0 and node2Dot > 0 then
    if node1Dot < node2Dot then
      return roadNode1
    else
      return roadNode2
    end
  elseif node1Dot > 0 then
    return roadNode1
  elseif node2Dot > 0 then
    return roadNode2
  end

  return nil
end

local function getNodeBehind(vehicle, roadNode1, roadNode2)
  local nodeInfront = getNodeInfront(vehicle, roadNode1, roadNode2)
  if nodeInfront then
    if nodeInfront == roadNode1 then
      return roadNode2
    else
      return roadNode1
    end
  end

  return nil
end

local function onRaceStart()
  -- log('I', logTag,'onRaceStart called')
  reset()
  local mapData = map.getMap()

  local targetVehicle = scenetree.findObject(targetInstance)
  local targetPosition  = vec3(targetVehicle:getPosition())
  targetVehicle:queueLuaCommand('controller.setFreeze(1)')

  local playerVehicle = scenetree.findObject(playerInstance)
  startPosition   = vec3(playerVehicle:getPosition())

  local pathNode1 = getClosestRoadNode(startPosition)
  local first, second, distance = map.findClosestRoad(targetPosition)
  local pathNode2 = getNodeBehind(targetVehicle, first, second)
  pathToTarget = map.getPath(pathNode1, pathNode2)

  -- dump(mapData.nodes)
  local behindPos = mapData.nodes[pathNode2].pos
  local infrontPos
  if pathNode2 == first then
    infrontPos = mapData.nodes[second].pos
  else
    infrontPos = mapData.nodes[first].pos
  end

  local lineDelta = distanceAlongLineSeg(behindPos, infrontPos, targetPosition)
  targetDistance = math.floor(getPathDistance(pathToTarget) + lineDelta)
  -- log('I', logTag,'targetDistance: '..targetDistance)

  scenario_scenarios.trackVehicleMovementAfterDamage(playerInstance)
  playerVehicle:queueLuaCommand('wheels.scaleBrakeTorque(0)')

  maxPoints = 1000
  local data = {
      enabled = false,
      maxPoints = maxPoints
    }

  statistics_statistics.setStatProgress(playerVehicle:getID(), 'distance', playerInstance, data)
  running = true
end

local function onRaceResult()
  if running == false then
    return
  end
  log('I', logTag,'onRaceResult called...')
  local playerVehicle = scenetree.findObject(playerInstance)
  local playerVehId = playerVehicle:getID()
  if finalPlayerDistance < targetDistance then
    statistics_statistics.setGoalProgress(playerVehId, "Gravity race", playerInstance, {status='failed', maxPoints=nil})
    fail('scenarios.utah.chapter_2.chapter_2_5_roll.fail.msg')
  else
    local maxDistance = 1931
    local range  = maxDistance - targetDistance
    local playerDist = finalPlayerDistance - targetDistance
    local points = math.floor((playerDist / range) * maxPoints)
    if points > maxPoints then
      points = maxPoints
    end
    local data = {value=finalPlayerDistance, points=points}
    statistics_statistics.setStatProgress(playerVehId, 'distance', playerInstance, data)
    success('scenarios.utah.chapter_2.chapter_2_5_roll.pass.msg')
  end

  reset()
end

-- local lastBehindNode = nil
-- local lastInfronNode = nil

local function onPreRender(dt)
  if not running then
    return
  end

  local vehicle = scenetree.findObject(playerInstance)

  if not vehicle then
    return
  end

  local vehicleData = map.objects[vehicle:getID()]
  if not vehicleData then
    return
  end

  -- log('I', logTag,'onPreRender called...')
  local mapData = map.getMap()

  local posVec = vehicle:getPosition()
  local first = nil
  local second = nil
  local distance = nil

  if not lastRoad then
    -- log('I', logTag,'updating last road...')
    first, second, distance = map.findClosestRoad(posVec)
    if first and second then
      lastRoad = {node1=first, node2=second, distance=distance}
    end
    -- dump(lastRoad)
  end

  local nodeInfront = getNodeInfront(vehicle, lastRoad.node1, lastRoad.node2)
  if not nodeInfront then
    -- log('I', logTag,'setting last road to nil')
    if lastRoad.nodeInfront and not playerVisited[lastRoad.nodeInfront] then
      table.insert(playerPath, lastRoad.nodeInfront)
      playerVisited[lastRoad.nodeInfront] = true
      playerDistance = getPathDistance(playerPath)
      guihooks.trigger('odometerDistance', {distance=playerDistance, targetDistance=targetDistance,decimalPlaces=2})
      finalPlayerDistance = playerDistance
    end
    lastRoad = nil
  else
    lastRoad.nodeInfront = nodeInfront
    local infrontPos = mapData.nodes[nodeInfront].pos
    local pathSize = tableSize(playerPath)
    if pathSize > 0 then
      lastRoad.behindNode = playerPath[pathSize]
      local behindPos = mapData.nodes[lastRoad.behindNode].pos
      local lineDelta = distanceAlongLineSeg(behindPos, infrontPos, posVec)
      guihooks.trigger('odometerDistance', {distance=playerDistance + lineDelta, targetDistance=targetDistance,decimalPlaces=2})
      finalPlayerDistance = playerDistance + lineDelta
    end
  end
end

M.onRaceStart             = onRaceStart
M.onRaceResult            = onRaceResult
M.onScenarioChange        = onScenarioChange
M.onVehicleStoppedMoving  = onVehicleStoppedMoving
M.onPreRender             = onPreRender

return M
