local M = {}

local sunriseGroup = nil
local nightGroup = nil
local checkInterval = 1.0  -- check every 1 second
local fadeSpeed = 0.3      -- volume change per second
local maxVolume = 1.0      -- max volume
local minVolume = 0.0

local sunriseEmitters = {}
local nightEmitters = {}

local timer = 0

-- Get current in-game time
local function getTime()
  local sun = scenetree.findObject("theSun")
  if sun then return sun:getTimeOfDay() end
  return 12 -- fallback
end

-- Time range logic
local function isInTimeRange(currentTime, startHour, endHour)
  if startHour < endHour then
    return currentTime >= startHour and currentTime < endHour
  else
    return currentTime >= startHour or currentTime < endHour
  end
end

-- Recursively collect all SFXEmitters inside a group
local function collectEmitters(group, outList)
  if not group then return end
  for _, obj in ipairs(group:getChildren()) do
    if obj:isSubClassOf("SFXEmitter") then
      table.insert(outList, obj)
      obj:setVolume(0)
      obj:setActive(true) -- start active to allow fading
    elseif obj:getClassName() == "SimGroup" then
      collectEmitters(obj, outList)
    end
  end
end

-- Smooth fade for each emitter
local function updateEmitterVolumes(emitters, targetVolume, dt)
  for _, emitter in ipairs(emitters) do
    if emitter then
      local currentVol = emitter:getVolume() or 0
      local diff = targetVolume - currentVol
      if math.abs(diff) > 0.01 then
        local newVol = currentVol + math.min(fadeSpeed * dt, math.abs(diff)) * (diff > 0 and 1 or -1)
        newVol = math.min(math.max(newVol, minVolume), maxVolume)
        emitter:setVolume(newVol)
      else
        emitter:setVolume(targetVolume)
      end
    end
  end
end

-- Init groups and emitters once
local function initialize()
  sunriseGroup = scenetree.findObject("SunriseSound")
  nightGroup = scenetree.findObject("NightAmbientSound")
  if sunriseGroup and #sunriseEmitters == 0 then
    collectEmitters(sunriseGroup, sunriseEmitters)
  end
  if nightGroup and #nightEmitters == 0 then
    collectEmitters(nightGroup, nightEmitters)
  end
end

-- Main update loop
M.onUpdate = function(dt)
  timer = timer + dt
  initialize()

  if timer >= checkInterval then
    local time = getTime()
    local isMorning = isInTimeRange(time, 5, 11)
    local isNight = isInTimeRange(time, 19, 4)

    -- Decide fade targets
    local targetSunrise = isMorning and maxVolume or minVolume
    local targetNight = isNight and maxVolume or minVolume

    updateEmitterVolumes(sunriseEmitters, targetSunrise, timer)
    updateEmitterVolumes(nightEmitters, targetNight, timer)

    timer = 0
  end
end

return M
