#!/usr/bin/env python3
import re
from pathlib import Path
from datetime import datetime

# =========================
#  MENU (toggle + FOV set)
# =========================
MENU = [
    ("UnlockInteractCamera",  "1"),
    ("RemoveReticleSway",     "2"),
    ("NoKillCamFilter",       "3"),
    ("RemoveBlackBars",       "4"),
    ("Set NON-aiming FOV",    "5"),
    ("Set AIMING FOV",        "6"),
]

# =========================
#  HARD-CODED TOGGLE TARGETS
# =========================
HARDCODED_PAIR_VALUE = {
    "UnlockInteractCamera": [
        (9541,  "UNK_MEMBER_0xBB742862", "0.00000000",  "1000.00000000"),
        (13699, "UNK_MEMBER_0x791B55E6", "0.00000000",  "0.50000000"),
        (13700, "UNK_MEMBER_0xFB0803FA", "0.00000000",  "0.55000000"),
        (13707, "SpringConstant",         "3000.00000000","15.00000000"),
        (13865, "UNK_MEMBER_0xBB742862", "0.00000000",  "90.00000000"),
    ],
    "RemoveReticleSway": [],
    "NoKillCamFilter": [],
    "RemoveBlackBars": [
        (46341, "UNK_MEMBER_0xAE00384B", "1.77800000", "2.35000000"),
    ],
}

HARDCODED_SIMPLE = {
    "UnlockInteractCamera": [],
    "RemoveReticleSway": [
        (5764,  "<Item>0xC9B22720</Item>"),
        (15806, "<Item>0xC9B22720</Item>"),
        (16603, "<Item>0xC9B22720</Item>"),
        (16936, "<Item>0xC9B22720</Item>"),
        (21417, "<Item>0xC9B22720</Item>"),
        (21643, "<Item>0xC9B22720</Item>"),
        (21869, "<Item>0xC9B22720</Item>"),
        (22094, "<Item>0xC9B22720</Item>"),
        (23327, "<Item>0xC9B22720</Item>"),
        (23553, "<Item>0xC9B22720</Item>"),
        (23777, "<Item>0xC9B22720</Item>"),
        (24014, "<Item>0xC9B22720</Item>"),
        (24251, "<Item>0xC9B22720</Item>"),
        (25398, "<Item>0xC9B22720</Item>"),
        (25740, "<Item>0xC9B22720</Item>"),
        (26075, "<Item>0xC9B22720</Item>"),
        (26297, "<Item>0xC9B22720</Item>"),
        (26699, "<Item>0xC9B22720</Item>"),
        (26922, "<Item>0xC9B22720</Item>"),
        (27838, "<Item>0xC9B22720</Item>"),
        (28180, "<Item>0xC9B22720</Item>"),
    ],
    "NoKillCamFilter": [],
    "RemoveBlackBars": [],
}

HARDCODED_SWAP = {
    "UnlockInteractCamera": [],
    "RemoveReticleSway": [],
    "NoKillCamFilter": [
        (
            46197,
            "<UNK_MEMBER_0x55AAD59B/>",
            "<UNK_MEMBER_0x55AAD59B>killcamhonorchange</UNK_MEMBER_0x55AAD59B>",
        )
    ],
    "RemoveBlackBars": [],
}

# =========================
#  REGEX HELPERS (toggle)
# =========================
def build_pair_value_regex(tag: str, active_val: str, default_val: str, marker: str, active_first: bool):
    t  = re.escape(tag)
    av = re.escape(active_val)
    dv = re.escape(default_val)
    m  = re.escape(marker)
    tag_active  = rf'<\s*{t}\s+value\s*=\s*"{av}"\s*/\s*>'
    tag_default = rf'<\s*{t}\s+value\s*=\s*"{dv}"\s*/\s*>'
    c_open, c_close = r'<!--\s*', r'\s*-->'
    marker_cmt = rf'<!--{m}-->'
    if active_first:
        pat = rf'^(?P<indent>\s*){tag_active}\s*{c_open}{tag_default}{c_close}\s*{marker_cmt}\s*$'
    else:
        pat = rf'^(?P<indent>\s*){c_open}{tag_default}{c_close}\s*{tag_active}\s*{marker_cmt}\s*$'
    return re.compile(pat)

def detect_pair_value_state(line: str, tag: str, active_val: str, default_val: str, marker: str):
    if build_pair_value_regex(tag, active_val, default_val, marker, True).match(line):  return "enabled"
    if build_pair_value_regex(tag, active_val, default_val, marker, False).match(line): return "disabled"
    return "unknown"

def rewrite_line_pair_value(line: str, tag: str, active_val: str, default_val: str, marker: str):
    pat_on  = build_pair_value_regex(tag, active_val, default_val, marker, active_first=True)
    pat_off = build_pair_value_regex(tag, active_val, default_val, marker, active_first=False)
    m = pat_on.match(line)
    if m:
        indent = m.group("indent")
        return f'{indent}<{tag} value="{default_val}" /><!--<{tag} value="{active_val}" />--><!--{marker}-->\n', True
    m = pat_off.match(line)
    if m:
        indent = m.group("indent")
        return f'{indent}<{tag} value="{active_val}" /><!--<{tag} value="{default_val}" />--><!--{marker}-->\n', True
    return line, False

def build_simple_regex(inner: str, marker: str, commented_first: bool):
    i = re.escape(inner.strip())
    m = re.escape(marker)
    if commented_first:
        return re.compile(rf'^(?P<indent>\s*)<!--\s*(?P<inner>{i})\s*-->\s*<!--{m}-->\s*$')
    return re.compile(rf'^(?P<indent>\s*)(?P<inner>{i})\s*<!--{m}-->\s*$')

def detect_simple_state(line: str, inner: str, marker: str):
    if build_simple_regex(inner, marker, True).match(line):  return "enabled"
    if build_simple_regex(inner, marker, False).match(line): return "disabled"
    return "unknown"

def rewrite_line_simple(line: str, inner: str, marker: str):
    pat_commented = build_simple_regex(inner, marker, commented_first=True)
    pat_plain     = build_simple_regex(inner, marker, commented_first=False)
    m = pat_commented.match(line)
    if m:
        indent = m.group("indent")
        return f"{indent}{inner.strip()}<!--{marker}-->\n", True
    m = pat_plain.match(line)
    if m:
        indent = m.group("indent")
        return f"{indent}<!--{inner.strip()}--><!--{marker}-->\n", True
    return line, False

def build_swap_regex(active_xml: str, default_xml: str, marker: str, active_first: bool):
    a = re.escape(active_xml.strip())
    b = re.escape(default_xml.strip())
    m = re.escape(marker)
    c_open, c_close = r'<!--\s*', r'\s*-->'
    marker_cmt = rf'<!--{m}-->'
    if active_first:
        return re.compile(rf'^(?P<indent>\s*)(?P<a>{a})\s*{c_open}(?P<b>{b}){c_close}\s*{marker_cmt}\s*$')
    else:
        return re.compile(rf'^(?P<indent>\s*){c_open}(?P<b>{b}){c_close}\s*(?P<a>{a})\s*{marker_cmt}\s*$')

def detect_swap_state(line: str, active_xml: str, default_xml: str, marker: str):
    if build_swap_regex(active_xml, default_xml, marker, True).match(line):  return "enabled"
    if build_swap_regex(active_xml, default_xml, marker, False).match(line): return "disabled"
    return "unknown"

def rewrite_line_swap(line: str, active_xml: str, default_xml: str, marker: str):
    pat_on  = build_swap_regex(active_xml, default_xml, marker, active_first=True)
    pat_off = build_swap_regex(active_xml, default_xml, marker, active_first=False)
    m = pat_on.match(line)
    if m:
        indent = m.group("indent")
        return f"{indent}{default_xml}<!--{active_xml}--><!--{marker}-->\n", True
    m = pat_off.match(line)
    if m:
        indent = m.group("indent")
        return f"{indent}{active_xml}<!--{default_xml}--><!--{marker}-->\n", True
    return line, False

# =========================
#  STATUS SUMMARY (toggles)
# =========================
def summarize_toggle_state(lines: list[str]) -> dict:
    summary = {}
    for marker, _num in MENU[:4]:
        en = dis = unk = 0
        for (line_no, tag, mod_val, def_val) in HARDCODED_PAIR_VALUE.get(marker, []):
            idx = line_no - 1
            if 0 <= idx < len(lines):
                s = detect_pair_value_state(lines[idx], tag, mod_val, def_val, marker)
                en += (s == "enabled"); dis += (s == "disabled"); unk += (s == "unknown")
        for (line_no, inner) in HARDCODED_SIMPLE.get(marker, []):
            idx = line_no - 1
            if 0 <= idx < len(lines):
                s = detect_simple_state(lines[idx], inner, marker)
                en += (s == "enabled"); dis += (s == "disabled"); unk += (s == "unknown")
        for (line_no, a_xml, b_xml) in HARDCODED_SWAP.get(marker, []):
            idx = line_no - 1
            if 0 <= idx < len(lines):
                s = detect_swap_state(lines[idx], a_xml, b_xml, marker)
                en += (s == "enabled"); dis += (s == "disabled"); unk += (s == "unknown")
        dis += unk
        summary[marker] = (en, dis)
    return summary

# =========================
#  FOV EDIT (5=non-aim, 6=aim)
# =========================
PAIR_ANY_TAG = re.compile(
    r'^(?P<indent>\s*)'
    r'(?P<active><\s*(?P<tag>[A-Za-z0-9_:.]+)\s+value\s*=\s*"(?P<val_active>[0-9.]+)"\s*/\s*>)'
    r'\s*<!--\s*'
    r'(?P<default><\s*(?P=tag)\s+value\s*=\s*"(?P<val_default>[0-9.]+)"\s*/\s*>)'
    r'\s*-->\s*$'
)

def is_fov_tag(tag_name: str) -> bool:
    t = tag_name.lower()
    return (
        t == "fov" or
        t == "basefov" or
        t == "defaultfov" or
        t == "maxfov" or
        t == "maxfovtoconsiderformaxnearclip" or
        t == "aimfov" or
        t.endswith(":fov")
    )

def classify_aiming(tag_name: str, file_suffix: str) -> bool:
    if 'aim' in tag_name.lower():
        return True
    if file_suffix.lower() == '.meta':
        return True
    return False

def format_value_like(active_str: str, default_str: str, new_value: float) -> str:
    """
    Preserve style:
      - If active has decimals, mirror its precision.
      - Else if default has decimals, mirror default precision (e.g., '45.0').
      - Else write integer if possible, else plain float.
    """
    def dec_places(s: str) -> int | None:
        return len(s.split('.', 1)[1]) if '.' in s else None

    ap = dec_places(active_str)
    dp = dec_places(default_str)
    if ap is not None:
        return f"{new_value:.{ap}f}".rstrip('0').rstrip('.') if ap == 0 else f"{new_value:.{ap}f}"
    if dp is not None:
        return f"{new_value:.{dp}f}".rstrip('0').rstrip('.') if dp == 0 else f"{new_value:.{dp}f}"
    if float(new_value).is_integer():
        return str(int(round(new_value)))
    return str(new_value)

def set_fov_across_tree(root: Path, new_value: float, aiming: bool, backup_enabled: bool) -> tuple[int,int]:
    files_changed = 0
    lines_changed = 0
    for f in root.rglob("*"):
        if not f.is_file():
            continue
        suffix = f.suffix.lower()
        if suffix not in ('.ymt', '.meta'):
            continue

        text = f.read_text(encoding="utf-8", errors="ignore")
        changed_in_file = 0
        out_lines = []
        backed_up = False

        for raw in text.splitlines(keepends=True):
            line = raw.rstrip("\n")
            m = PAIR_ANY_TAG.match(line)
            if not m:
                out_lines.append(raw); continue

            tag = m.group("tag")
            if not is_fov_tag(tag):
                out_lines.append(raw); continue

            is_aim = classify_aiming(tag, suffix)
            if is_aim != aiming:
                out_lines.append(raw); continue

            indent      = m.group("indent")
            active_str  = m.group("val_active")
            default_str = m.group("val_default")

            new_val_str = format_value_like(active_str, default_str, new_value)

            new_active = re.sub(
                r'value\s*=\s*"[0-9.]+"',
                f'value="{new_val_str}"',
                m.group("active")
            )
            new_line = f"{indent}{new_active}<!--{m.group('default')}-->\n"

            if new_line != raw:
                if backup_enabled and not backed_up:
                    ts = datetime.now().strftime("%Y%m%d_%H%M%S")
                    backup = f.with_suffix(f.suffix + f".fovbak_{ts}")
                    backup.write_text(text, encoding="utf-8")
                    backed_up = True
                out_lines.append(new_line)
                changed_in_file += 1
            else:
                out_lines.append(raw)

        if changed_in_file:
            f.write_text("".join(out_lines), encoding="utf-8")
            files_changed += 1
            lines_changed += changed_in_file

    return files_changed, lines_changed

# =========================
#  UI HELPERS
# =========================
def parse_selection(prompt: str, max_n: int) -> list[int]:
    while True:
        s = input(prompt).strip()
        if not s:
            print("Enter numbers 1-6 (commas/ranges ok, e.g. 1,3 or 1-4).")
            continue
        chosen: set[int] = set()
        ok = True
        for part in s.split(","):
            part = part.strip()
            if "-" in part:
                a, _, b = part.partition("-")
                if a.isdigit() and b.isdigit():
                    ai, bi = int(a), int(b)
                    if 1 <= ai <= max_n and 1 <= bi <= max_n and ai <= bi:
                        chosen.update(range(ai, bi+1))
                    else:
                        ok = False; break
                else:
                    ok = False; break
            else:
                if part.isdigit():
                    n = int(part)
                    if 1 <= n <= max_n:
                        chosen.add(n)
                    else:
                        ok = False; break
                else:
                    ok = False; break
        if ok and chosen:
            return sorted(chosen)
        print("Invalid selection. Use numbers 1-6, e.g. 2,5 or 1-6.")

def prompt_float(msg: str) -> float:
    while True:
        raw = input(msg).strip()
        try:
            return float(raw)
        except ValueError:
            print("Please enter a number (e.g., 73, 70.0, 51.30000000).")

# =========================
#  MAIN
# =========================
def main():
    cwd = Path.cwd()
    cameras = cwd / "cameras.ymt"

    # --- Backup toggle prompt (disabled by default) ---
    ans = input("Enable backups (.bak / .fovbak)? [y/N]: ").strip().lower()
    backup_enabled = ans.startswith('y')

    if cameras.exists():
        text = cameras.read_text(encoding="utf-8", errors="ignore")
        lines = text.splitlines(keepends=True)
        summary = summarize_toggle_state(lines)
        print("Select which to TOGGLE (from current state):")
        for name, num in MENU[:4]:
            en, dis = summary[name]
            print(f"  {num}. {name}  [{en} enabled, {dis} disabled]")
    else:
        print("Select which to TOGGLE (from current state):")

    print("  5. Set NON-aiming FOV (Fov in .ymt files)")
    print("  6. Set AIMING FOV     (Fov in .meta files)")
    selected_idx = parse_selection("> ", len(MENU))
    selected_labels = [MENU[i-1][0] for i in selected_idx]

    # Handle toggles 1–4 strictly on cameras.ymt
    if cameras.exists():
        text = cameras.read_text(encoding="utf-8", errors="ignore")
        lines = text.splitlines(keepends=True)

        needs_backup = backup_enabled and any(
            lbl in ("UnlockInteractCamera","RemoveReticleSway","NoKillCamFilter","RemoveBlackBars")
            for lbl in selected_labels
        )
        if needs_backup:
            ts = datetime.now().strftime("%Y%m%d_%H%M%S")
            backup = cameras.with_suffix(cameras.suffix + f".bak_{ts}")
            backup.write_text(text, encoding="utf-8")
            print(f"\nBackup written: {backup.name}")

        changes = 0
        per_marker = {m: 0 for m, _ in MENU[:4]}

        for marker in selected_labels:
            if marker not in per_marker:  # skip FOV options
                continue
            for (line_no, tag, mod_val, def_val) in HARDCODED_PAIR_VALUE.get(marker, []):
                idx = line_no - 1
                if 0 <= idx < len(lines):
                    new_line, changed = rewrite_line_pair_value(lines[idx], tag, mod_val, def_val, marker)
                    if changed:
                        lines[idx] = new_line; changes += 1; per_marker[marker] += 1
            for (line_no, inner) in HARDCODED_SIMPLE.get(marker, []):
                idx = line_no - 1
                if 0 <= idx < len(lines):
                    new_line, changed = rewrite_line_simple(lines[idx], inner, marker)
                    if changed:
                        lines[idx] = new_line; changes += 1; per_marker[marker] += 1
            for (line_no, a_xml, b_xml) in HARDCODED_SWAP.get(marker, []):
                idx = line_no - 1
                if 0 <= idx < len(lines):
                    new_line, changed = rewrite_line_swap(lines[idx], a_xml, b_xml, marker)
                    if changed:
                        lines[idx] = new_line; changes += 1; per_marker[marker] += 1

        if changes:
            cameras.write_text("".join(lines), encoding="utf-8")
            print(f"\nSaved toggle changes to {cameras.name} (lines changed: {changes})")
            summary2 = summarize_toggle_state("".join(lines).splitlines(keepends=True))
            for name, num in MENU[:4]:
                en, dis = summary2[name]
                print(f"  {num}. {name}  [now {en} enabled, {dis} disabled]")

    # Handle FOV options (5 / 6) across the whole tree
    if "Set NON-aiming FOV" in selected_labels or "Set AIMING FOV" in selected_labels:
        if "Set NON-aiming FOV" in selected_labels:
            new_fov = prompt_float("\nEnter NON-aiming FOV (e.g., 73, 70, 51.30000000): ")
            fchg, lchg = set_fov_across_tree(cwd, new_fov, aiming=False, backup_enabled=backup_enabled)
            print(f"NON-aiming FOV updated in {fchg} file(s), {lchg} line(s).")
        if "Set AIMING FOV" in selected_labels:
            new_aim = prompt_float("\nEnter AIMING FOV (e.g., 45.0, 42, 30.0): ")
            fchg, lchg = set_fov_across_tree(cwd, new_aim, aiming=True, backup_enabled=backup_enabled)
            print(f"AIMING FOV updated in {fchg} file(s), {lchg} line(s).")

    input("\nDone. Press Enter to exit...")

if __name__ == "__main__":
    main()
