﻿#include "script.h"
#include "keyboard.h"
#include "natives.h"
#include <vector>
#include <string>
#include "global.h"

// ================== GLOBALS ==================
std::vector<Ped> herd;           // Active herd
std::vector<Ped> spawnedCows;    // Cows spawned with debug keybinds
std::vector<int> cowBlips;       // Blips for cows
std::vector<Ped> stragglerCows; // straggler cows needing herding

// AI Cowboy System
std::vector<Ped> cowboys;        // Active cowboys
std::vector<Ped> cowboyHorses;   // Horses for cowboys
std::vector<int> cowboyBlips;    // Blips for cowboys
bool cowboysEnabled = true;
int maxCowboys = 3;
int cowboyHirePrice = 2500;
float cowboyEfficiency = 0.8f;   // How well they herd (0.0-1.0)

Ped herdLeader = 0;
bool herdingActive = false;
bool showHerdHUD = true; // default ON
float herdSpeed = 1.0f; // default normal speed
int hudToggleMessageTime = 10000; // when we last pressed K
int hudToggleFadeDuration = 3000; // ms to stay visible
int lastEventTime = 0;
int eventCooldown = 900000; // 15 minutes between random events
int randomEventChance = 30; // 30% chance every check
int notifyStart = 0;
std::string notifyText = "";
int notifyDuration = 3000; // show for 3 seconds
int playerPosX = 0, playerPosY = 0; // for HUD display

// Progression system
int ranchReputation = 0; // -100 (bad) to +100 (respected)
int herdingSkill = 1; // starts at 1, improves with practice
float dailyMarketModifier = 1.0f;
int lastMarketDay = -1;
bool rustlersEnabled = false; // set to false to disable rustler event
const int MAX_WORLD_PEDS = 1024;
Ped worldPeds[MAX_WORLD_PEDS];

int lastUpdateTime = 0;
const int updateInterval = 1000;
const float MAX_HERDING_DISTANCE = 50.0f;

// Blips / Prompts
int sellBlip, buyBlip;
Hash buyPrompt, sellPrompt;
Hash MONEY_STAT_BASE;
Hash MONEY_STAT_PERMUTATION;

// Herding prompts
Hash SpeedUpHerdPrompt;
Hash SlowDownHerdPrompt;
Hash stopHerdPrompt;
Hash herdPromptGroup;

// Cowboy prompts
Hash hireCowboyPrompt;
Hash dismissCowboyPrompt;

// Example buy/sell locations
Vector3 sellLocationValentine = { -258.027618, 669.947327f, 113.267586f };
Vector3 sellLocationBlackwater = { -754.0f, -1291.0f, 43.0f };

Vector3 buyLocationEmerald = { 1422.697266, 295.063171f, 88.962830f };
Vector3 buyLocationMcFarlane = { -2373.0f, -2410.0f, 61.5f };

// Cow prices
int cowPrice = 2000;
int cowSellPrice = 2500;

// ================== UTILS ==================
float CalculateDistance(Vector3 p1, Vector3 p2) {
    float dx = p2.x - p1.x;
    float dy = p2.y - p1.y;
    float dz = p2.z - p1.z;
    return sqrt(dx * dx + dy * dy + dz * dz);
}

// Custom notification system
void Notify(const char* message) {
    notifyText = message;
    notifyStart = GetGameTimer();
}

Vector3 NormalizeVector(Vector3 vec) {
    float length = sqrt(vec.x * vec.x + vec.y * vec.y + vec.z * vec.z);
    if (length > 0) {
        return { vec.x / length, vec.y / length, vec.z / length };
    }
    return { 0,0,0 };
}

Vector3 CalculateHerdCenter() {
    Vector3 center = { 0,0,0 };
    int validCount = 0;

    for (Ped cow : herd) {
        if (ENTITY::DOES_ENTITY_EXIST(cow) && !ENTITY::IS_ENTITY_DEAD(cow)) {
            Vector3 pos = ENTITY::GET_ENTITY_COORDS(cow, true, true);
            center.x += pos.x;
            center.y += pos.y;
            center.z += pos.z;
            validCount++;
        }
    }
    if (validCount > 0) {
        center.x /= validCount;
        center.y /= validCount;
        center.z /= validCount;
    }
    return center;
}

// Camera forward direction helper (returns normalized forward vector)
Vector3 GetCamDirection() {
    Vector3 rot = CAM::GET_GAMEPLAY_CAM_ROT(2); // returns pitch(x), roll(y), yaw(z)
    const float DEG2RAD = 0.0174532924f;
    float pitch = rot.x * DEG2RAD;
    float yaw = rot.z * DEG2RAD;
    float cosPitch = cosf(pitch);
    return { -sinf(yaw) * cosPitch, cosf(yaw) * cosPitch, sinf(pitch) };
}

// Helper: random float based on GetRandomIntInRange
float GetRandomFloatInRange(float min, float max) {
    int scale = 1000; // precision
    int rnd = GetRandomIntInRange((int)(min * scale), (int)(max * scale));
    return (float)rnd / scale;
}

void SetHerdSpeed(float speed) {
    herdSpeed = speed;

    for (Ped cow : herd) {
        if (ENTITY::DOES_ENTITY_EXIST(cow) && !ENTITY::IS_ENTITY_DEAD(cow)) {
            Vector3 pos = ENTITY::GET_ENTITY_COORDS(cow, true, true);
            TASK::TASK_FOLLOW_NAV_MESH_TO_COORD(cow, pos.x, pos.y, pos.z, herdSpeed, -1, 0.5f, 0, 0.f);
            if (herdSpeed == 0) {
                // Don't move herd — just idle
            }
            else {
                // existing movement logic
            }

        }
    }
}

bool IsCowOrBull(Ped ped, Hash& modelOut) {
    if (!ENTITY::DOES_ENTITY_EXIST(ped)) return false;
    modelOut = ENTITY::GET_ENTITY_MODEL(ped);
    return (modelOut == joaat("A_C_COW") || modelOut == joaat("A_C_BULL_01"));
}

// Update reputation
void ChangeReputation(int amount) {
    ranchReputation += amount;
    if (ranchReputation > 1) ranchReputation = 5;
    if (ranchReputation < -1) ranchReputation = -5;

    if (amount > 1) {
        Notify("Your reputation as a rancher has improved");
    }
    else if (amount < 1) {
        Notify("Your reputation has suffered");
    }
}

// Update herding skill
void ImproveHerding(int amount) {
    herdingSkill += amount;
    if (herdingSkill > 1) herdingSkill = 2, 3, 4, 5, 6, 7, 8, 9, 10; // cap at 10
    if (herdingSkill < 1) herdingSkill = -1;   // floor at 1
}

// Daily market modifier (random once per in-game day)
void UpdateMarketModifier() {
    int day = CLOCK::GET_CLOCK_DAY_OF_MONTH(); // natives.h
    if (day != lastMarketDay) {
        dailyMarketModifier = 0.8f + ((float)(rand() % 41) / 100.0f); // 0.8–1.2
        lastMarketDay = day;
    }
}

// ================== AI COWBOY SYSTEM ==================

struct CowboyData {
    Ped ped;
    Ped horse;
    int skillLevel;       // 1-5 skill level
    float wages;         // Daily wages
    int hiredTime;       // When they were hired
    bool isWorking;      // Currently herding
    Vector3 lastPosition;
    std::string name;
};

std::vector<CowboyData> activeCowboys;

// Cowboy names for variety
std::vector<std::string> cowboyNames = {
    "Jake", "Buck", "Tex", "Cole", "Wade", "Clay", "Luke", "Hank",
    "Rusty", "Dutch", "Jim", "Jesse", "Colt", "Clint", "Shane"
};

CowboyData CreateCowboy(Vector3 position) {
    CowboyData cowboy;

    // Create horse first
    std::vector<Hash> kentuckyModels = {
    joaat("A_C_HORSE_KENTUCKYSADDLE_BLACK"),
    joaat("A_C_HORSE_KENTUCKYSADDLE_CHESTNUTPINTO"),
    joaat("A_C_HORSE_KENTUCKYSADDLE_GREY"),
    joaat("A_C_HORSE_KENTUCKYSADDLE_SILVERBAY")
    };
    Hash horseModel = kentuckyModels[GetRandomIntInRange(0, (int)kentuckyModels.size() - 1)];
    STREAMING::REQUEST_MODEL(horseModel, 0);
    while (!STREAMING::HAS_MODEL_LOADED(horseModel)) WAIT(0);

    cowboy.horse = CreatePed(horseModel, position, 0.0f);
    ApplyRandomSaddle(cowboy.horse);

    // Create cowboy
    Hash cowboyModel = joaat("A_M_M_RANCHER_01");
    STREAMING::REQUEST_MODEL(cowboyModel, 0);
    while (!STREAMING::HAS_MODEL_LOADED(cowboyModel)) WAIT(0);

    Vector3 cowboyPos = { position.x + 1.0f, position.y, position.z };
    cowboy.ped = CreatePed(cowboyModel, cowboyPos, 0.0f);

    // Apply random outfit to cowboy
    SetRandomOutfitPreset(cowboy.ped);

    // Mount cowboy on horse
    PED::SET_PED_ONTO_MOUNT(cowboy.ped, cowboy.horse, -1, true);

    // Set stats
    cowboy.skillLevel = GetRandomIntInRange(1, 5);
    cowboy.wages = 5.0f + (cowboy.skillLevel * 5.0f);
    cowboy.hiredTime = GetGameTimer();
    cowboy.isWorking = false;
    cowboy.lastPosition = position;

    // Assign random name
    int nameIndex = GetRandomIntInRange(0, (int)cowboyNames.size() - 1);
    cowboy.name = cowboyNames[nameIndex];

    // Give appropriate weapons
    AddWeaponToPed(cowboy.ped, joaat("WEAPON_LASSO"), 1, 0, false);
    if (cowboy.skillLevel >= 3) {
        AddWeaponToPed(cowboy.ped, joaat("WEAPON_REVOLVER_CATTLEMAN"), 30, 0, false);
    }

    // Set behavior
    PED::SET_BLOCKING_OF_NON_TEMPORARY_EVENTS(cowboy.ped, false);
    PED::SET_PED_FLEE_ATTRIBUTES(cowboy.ped, 0, false);
    PED::SET_PED_COMBAT_ATTRIBUTES(cowboy.ped, 46, true); // Can fight

    // Add blip
    int blip = MAP::BLIP_ADD_FOR_ENTITY(joaat("BLIP_STYLE_PLAYER"), cowboy.ped);
    MAP::SET_BLIP_SPRITE(blip, joaat("blip_ambient_companion"), true);
    cowboyBlips.push_back(blip);

    STREAMING::SET_MODEL_AS_NO_LONGER_NEEDED(horseModel);
    STREAMING::SET_MODEL_AS_NO_LONGER_NEEDED(cowboyModel);

    return cowboy;
}

void HireCowboy() {
    if ((int)activeCowboys.size() >= maxCowboys) {
        Notify("You already have the maximum number of cowboys hired!");
        return;
    }

    int cash = MONEY::_MONEY_GET_CASH_BALANCE();
    if (cash < cowboyHirePrice) {
        Notify("Not enough money to hire a cowboy!");
        return;
    }

    MONEY::_MONEY_DECREMENT_CASH_BALANCE(cowboyHirePrice);

    Ped playerPed = PLAYER::PLAYER_PED_ID();
    Vector3 playerPos = GetEntityCoords(playerPed);
    Vector3 spawnPos = GetRandomOffsetFromCoords(playerPos, 10.0f, 15.0f);

    CowboyData newCowboy = CreateCowboy(spawnPos);
    activeCowboys.push_back(newCowboy);

    std::string message = "Hired cowboy " + newCowboy.name + " (Skill: " + std::to_string(newCowboy.skillLevel) + "/5)";
    Notify(message.c_str());

    ChangeReputation(1);
}

void DismissCowboy(Ped target = 0) {
    if (activeCowboys.empty()) {
        Notify("No cowboys to dismiss!");
        return;
    }

    int closestIndex = -1;

    // If we were given a target (aimed at), find that cowboy
    if (target != 0) {
        for (size_t i = 0; i < activeCowboys.size(); i++) {
            if (activeCowboys[i].ped == target) {
                closestIndex = (int)i;
                break;
            }
        }
    }
    else {
        // fallback: nearest cowboy within range
        Ped playerPed = PLAYER::PLAYER_PED_ID();
        Vector3 playerPos = GetEntityCoords(playerPed);
        bool isPlayerMounted = PED::IS_PED_ON_MOUNT(playerPed);
        float closestDistance = 999999.0f;
        float maxRange = isPlayerMounted ? 50.0f : 10.0f;

        for (size_t i = 0; i < activeCowboys.size(); i++) {
            if (!ENTITY::DOES_ENTITY_EXIST(activeCowboys[i].ped)) continue;
            float distance = CalculateDistance(playerPos, GetEntityCoords(activeCowboys[i].ped));
            if (distance < closestDistance && distance < maxRange) {
                closestDistance = distance;
                closestIndex = (int)i;
            }
        }
    }

    if (closestIndex != -1) {
        std::string message = "Dismissed cowboy " + activeCowboys[closestIndex].name;
        if (ENTITY::DOES_ENTITY_EXIST(activeCowboys[closestIndex].ped)) DeleteEntity(&activeCowboys[closestIndex].ped);
        if (ENTITY::DOES_ENTITY_EXIST(activeCowboys[closestIndex].horse)) DeleteEntity(&activeCowboys[closestIndex].horse);
        activeCowboys.erase(activeCowboys.begin() + closestIndex);
        Notify(message.c_str());
    }
    else {
        Notify("No cowboy to dismiss!");
    }
}

void UpdateCowboyAI() {
    if (!herdingActive || herd.empty()) {
        Ped playerPed = PLAYER::PLAYER_PED_ID();
        Vector3 playerPos = GetEntityCoords(playerPed);

        for (auto& cowboy : activeCowboys) {
            if (!ENTITY::DOES_ENTITY_EXIST(cowboy.ped) || ENTITY::IS_ENTITY_DEAD(cowboy.ped)) continue;

            Vector3 cowboyPos = GetEntityCoords(cowboy.ped);
            float distToPlayer = CalculateDistance(cowboyPos, playerPos);

            // Offset formation position (spread them out around player)
            int index = &cowboy - &activeCowboys[0];
            float angleOffset = (index * 120.0f) * 3.14159f / 180.0f;
            Vector3 followPos = {
                playerPos.x + cos(angleOffset) * 12.0f,
                playerPos.y + sin(angleOffset) * 12.0f,
                playerPos.z
            };

            // If too far, always move
            if (distToPlayer > 15.0f) {
                float speed = (distToPlayer > 30.0f) ? 2.5f : 1.5f; // run if far
                FollowNavMeshToCoord(cowboy.ped, followPos, speed, -1, 4.0f, 0, 0.0f);
                cowboy.isWorking = false;
            }

        }
    }

    Ped playerPed = PLAYER::PLAYER_PED_ID();
    Vector3 playerPos = GetEntityCoords(playerPed);
    Vector3 herdCenter = CalculateHerdCenter();

    // Calculate herd movement direction
    Vector3 herdDirection = NormalizeVector({
        herdCenter.x - playerPos.x,
        herdCenter.y - playerPos.y,
        0
        });

    for (auto& cowboy : activeCowboys) {
        if (!ENTITY::DOES_ENTITY_EXIST(cowboy.ped) || ENTITY::IS_ENTITY_DEAD(cowboy.ped)) continue;

        Vector3 cowboyPos = GetEntityCoords(cowboy.ped);
        float distToPlayer = CalculateDistance(cowboyPos, playerPos);

        // Only work if reasonably close to the herding action
        if (distToPlayer > MAX_HERDING_DISTANCE * 2.5f) {
            cowboy.isWorking = false;
            continue;
        }

        cowboy.isWorking = true;

        // Find cattle that need herding
        Ped targetCow = 0;
        float maxStrayDistance = 0.0f;
        Vector3 strayPos;

        for (Ped cow : herd) {
            if (!ENTITY::DOES_ENTITY_EXIST(cow) || ENTITY::IS_ENTITY_DEAD(cow)) continue;

            Vector3 cowPos = GetEntityCoords(cow);
            float distFromCenter = CalculateDistance(cowPos, herdCenter);
            float distFromPlayer = CalculateDistance(cowPos, playerPos);

            // Prioritize cattle that are far from both herd center and player
            float priority = distFromCenter + (distFromPlayer * 0.5f);

            if (priority > maxStrayDistance && distFromCenter > 15.0f) {
                maxStrayDistance = priority;
                targetCow = cow;
                strayPos = cowPos;
            }
        }

        if (targetCow != 0) {
            // Position cowboy to effectively push cow toward herd
            Vector3 pushDirection = NormalizeVector({
                herdCenter.x - strayPos.x,
                herdCenter.y - strayPos.y,
                0
                });

            // Position cowboy behind the cow relative to where we want it to go
            Vector3 herderPos = {
                strayPos.x - pushDirection.x * 12.0f,
                strayPos.y - pushDirection.y * 12.0f,
                strayPos.z
            };

            // Match herd speed - cowboys move faster when herd is running
            float cowboySpeed = herdSpeed;
            if (herdSpeed > 1.5f) {
                cowboySpeed = herdSpeed * 1.1f; // Slightly faster to catch up
            }
            else if (herdSpeed == 0.0f) {
                cowboySpeed = herdSpeed; // Slow approach when herd is stopped
            }

            // Skill affects success rate
            float skillModifier = cowboy.skillLevel / 5.0f;
            if (GetRandomIntInRange(0, 100) < (int)(80 * skillModifier)) {
                FollowNavMeshToCoord(cowboy.ped, herderPos, cowboySpeed, -1, 3.0f, 0, 0.0f);

                // Apply pressure to the cow to move it
                float distToCow = CalculateDistance(cowboyPos, strayPos);
                if (distToCow < 20.0f) {
                    // Make the cow aware of the cowboy and move away
                    Vector3 cowMoveDirection = NormalizeVector({
                        pushDirection.x + GetRandomFloatInRange(-0.3f, 0.3f),
                        pushDirection.y + GetRandomFloatInRange(-0.3f, 0.3f),
                        0
                        });

                    Vector3 cowTarget = {
                        strayPos.x + cowMoveDirection.x * 15.0f,
                        strayPos.y + cowMoveDirection.y * 15.0f,
                        strayPos.z
                    };

                    // Make cow move at herd speed
                    TASK::TASK_FOLLOW_NAV_MESH_TO_COORD(targetCow, cowTarget.x, cowTarget.y, cowTarget.z, herdSpeed, -1, 1.0f, 0, 0.0f);

                    // Higher skill cowboys occasionally use lasso
                    if (cowboy.skillLevel >= 4 && GetRandomIntInRange(0, 100) < 15) {
                        TaskAimGunAtEntity(cowboy.ped, targetCow, 1500);
                    }
                }
            }
        }
        else {
            // No stray cattle - take flanking positions around the herd
            int cowboyIndex = 0;
            for (size_t i = 0; i < activeCowboys.size(); i++) {
                if (&activeCowboys[i] == &cowboy) {
                    cowboyIndex = (int)i;
                    break;
                }
            }

            // Position cowboys at different angles around the herd
            float baseAngle = atan2(herdDirection.y, herdDirection.x);
            float cowboyAngle = baseAngle + (cowboyIndex * 120.0f * 3.14159f / 180.0f); // 120 degrees apart

            Vector3 flankPos = {
                herdCenter.x + cos(cowboyAngle) * 30.0f,
                herdCenter.y + sin(cowboyAngle) * 30.0f,
                herdCenter.z
            };

            // Move at herd speed to stay in formation
            float moveSpeed = (herdSpeed > 0.0f) ? herdSpeed : 0.8f;
            FollowNavMeshToCoord(cowboy.ped, flankPos, moveSpeed, -1, 8.0f, 0, 0.0f);
        }

        cowboy.lastPosition = cowboyPos;
    }
}

void HandleCowboyWages() {
    static int lastWageTime = 0;
    int currentTime = GetGameTimer();

    // Pay wages every 5 minutes of real time (represents a day)
    if (currentTime - lastWageTime > 300000) { // 5 minutes
        lastWageTime = currentTime;

        float totalWages = 0;
        for (const auto& cowboy : activeCowboys) {
            if (ENTITY::DOES_ENTITY_EXIST(cowboy.ped) && !ENTITY::IS_ENTITY_DEAD(cowboy.ped)) {
                totalWages += cowboy.wages;
            }
        }

        if (totalWages > 0) {
            int cash = MONEY::_MONEY_GET_CASH_BALANCE();
            if (cash >= (int)totalWages) {
                MONEY::_MONEY_DECREMENT_CASH_BALANCE((int)totalWages);
                std::string message = "Paid $" + std::to_string((int)totalWages) + " in cowboy wages";
                Notify(message.c_str());
            }
            else {
                // Can't afford wages - cowboys might leave
                for (auto it = activeCowboys.begin(); it != activeCowboys.end();) {
                    if (GetRandomIntInRange(0, 100) < 30) { // 30% chance to leave
                        std::string message = "Cowboy " + it->name + " left due to unpaid wages!";
                        DeleteEntity(&it->ped);
                        DeleteEntity(&it->horse);
                        it = activeCowboys.erase(it);
                        Notify(message.c_str());
                        ChangeReputation(-2);
                    }
                    else {
                        ++it;
                    }
                }
            }
        }
    }
}

void CleanupCowboys() {
    // Remove dead or missing cowboys
    for (auto it = activeCowboys.begin(); it != activeCowboys.end();) {
        if (!ENTITY::DOES_ENTITY_EXIST(it->ped) || ENTITY::IS_ENTITY_DEAD(it->ped)) {
            std::string message = "Cowboy " + it->name + " is no longer available";
            Notify(message.c_str());

            if (ENTITY::DOES_ENTITY_EXIST(it->horse)) {
                DeleteEntity(&it->horse);
            }

            it = activeCowboys.erase(it);
        }
        else {
            ++it;
        }
    }

    // Clean up orphaned blips
    for (size_t i = 0; i < cowboyBlips.size(); i++) {
        if (MAP::DOES_BLIP_EXIST(cowboyBlips[i])) {
            MAP::REMOVE_BLIP(&cowboyBlips[i]);
        }
    }
    cowboyBlips.clear();

    // Recreate blips for active cowboys
    for (const auto& cowboy : activeCowboys) {
        if (ENTITY::DOES_ENTITY_EXIST(cowboy.ped)) {
            int blip = MAP::BLIP_ADD_FOR_ENTITY(joaat("BLIP_STYLE_PLAYER"), cowboy.ped);
            MAP::SET_BLIP_SPRITE(blip, joaat("blip_ambient_companion"), true);
            cowboyBlips.push_back(blip);
        }
    }
}

// ================== ECONOMY ==================
void SpawnCow(Vector3 pos) {
    // Array of different cow models for variety
    std::vector<Hash> cowModels = {
        joaat("A_C_COW")
    };

    // Randomly select a cow model
    Hash model = cowModels[GetRandomIntInRange(0, (int)cowModels.size() - 1)];

    STREAMING::REQUEST_MODEL(model, 0);
    while (!STREAMING::HAS_MODEL_LOADED(model)) WAIT(0);

    Ped cow = CreatePed(model, pos, 0.0f);
    if (DoesEntityExist(cow)) {
        herd.push_back(cow);

        // 20% chance this cow is a straggler
        if (GetRandomIntInRange(0, 100) < 20) {
            stragglerCows.push_back(cow);
        }

        // Add blip for this cow
        int blip = MAP::BLIP_ADD_FOR_ENTITY(joaat("BLIP_STYLE_OBJECTIVE"), cow);
        MAP::SET_BLIP_NAME_FROM_TEXT_FILE(blip, "cow");
        cowBlips.push_back(blip);
    }

    STREAMING::SET_MODEL_AS_NO_LONGER_NEEDED(model);
}

void BuyCow()
{
    int cash = MONEY::_MONEY_GET_CASH_BALANCE();

    if (cash >= cowPrice)
    {
        MONEY::_MONEY_DECREMENT_CASH_BALANCE(cowPrice);

        // Determine spawn location based on player proximity
        Ped playerPed = PLAYER::PLAYER_PED_ID();
        Vector3 playerPos = GetEntityCoords(playerPed);

        Vector3 spawnPos;
        float distToEmerald = CalculateDistance(playerPos, buyLocationEmerald);
        float distToMcFarlane = CalculateDistance(playerPos, buyLocationMcFarlane);

        if (distToEmerald < distToMcFarlane) {
            // Spawn near Emerald Ranch
            spawnPos = { buyLocationEmerald.x + 2.0f, buyLocationEmerald.y + 2.0f, buyLocationEmerald.z };
        }
        else {
            // Spawn near McFarlane Ranch
            spawnPos = { buyLocationMcFarlane.x + 2.0f, buyLocationMcFarlane.y + 2.0f, buyLocationMcFarlane.z };
        }

        SpawnCow(spawnPos);
        Notify("You bought a Cow!");
    }
    else
    {
        Notify("Not enough money to buy a Cow!");
    }
}

void AutoSellHerd() {
    for (auto it = herd.begin(); it != herd.end();) {
        Ped cow = *it;
        if (ENTITY::DOES_ENTITY_EXIST(cow)) {
            Vector3 cowPos = ENTITY::GET_ENTITY_COORDS(cow, true, true);

            // Check distance to both sell locations
            float distToValentine = CalculateDistance(cowPos, sellLocationValentine);
            float distToBlackwater = CalculateDistance(cowPos, sellLocationBlackwater);

            // Sell if cow is within range of either location
            if (distToValentine < 40.0f || distToBlackwater < 40.0f) {
                MONEY::_MONEY_INCREMENT_CASH_BALANCE(cowSellPrice, 4000);
                DeleteEntity(&cow);
                it = herd.erase(it);
                continue;
            }
        }
        ++it;
    }

    // Clean up blips after selling
    for (size_t i = 0; i < cowBlips.size(); i++) {
        if (MAP::DOES_BLIP_EXIST(cowBlips[i])) {
            MAP::REMOVE_BLIP(&cowBlips[i]);
        }
    }
    cowBlips.clear();
}

void TriggerRandomEvent(Ped playerPed) {
    if (herd.empty()) return;

    if (rustlersEnabled) {
        Vector3 playerPos = GetEntityCoords(playerPed);

        int numRustlers = 5;
        for (int i = 0; i < numRustlers; i++) {
            float angle = (float)i * (360.0f / numRustlers) * 3.14159f / 10.0f;
            float distance = GetRandomFloatInRange(180.0f, 200.0f);

            Vector3 spawnPos = {
                playerPos.x + cos(angle) * distance,
                playerPos.y + sin(angle) * distance,
                playerPos.z
            };

            // Get the ground Z coordinate for the spawn position
            float groundZ;
            MISC::GET_GROUND_Z_FOR_3D_COORD(spawnPos.x, spawnPos.y, 1000.0f, &groundZ, 1);
            spawnPos.z = groundZ;

            Hash rustlerModel = joaat("A_M_M_RANCHER_01");
            Hash horseModel = joaat("A_C_HORSE_KENTUCKYSADDLE_BLACK");
            STREAMING::REQUEST_MODEL(rustlerModel, 0);
            STREAMING::REQUEST_MODEL(horseModel, 0);

            while (!STREAMING::HAS_MODEL_LOADED(rustlerModel)) {
                WAIT(0);
            }
            while (!STREAMING::HAS_MODEL_LOADED(horseModel)) {
                WAIT(0);
            }

            Ped horse = CreatePed(horseModel, spawnPos.x, spawnPos.y, spawnPos.z, 0.0f, true);
            Ped rustler = CreatePed(rustlerModel, spawnPos.x, spawnPos.y, spawnPos.z, 0.0f, true);

            if (DoesEntityExist(horse) && DoesEntityExist(rustler)) {
                WEAPON::REMOVE_ALL_PED_WEAPONS(rustler, true, true);

                if (GetRandomIntInRange(0, 1) == 0) {
                    AddWeaponToPed(rustler, joaat("WEAPON_REVOLVER_CATTLEMAN"), 50, 0, true);
                }
                else {
                    AddWeaponToPed(rustler, joaat("WEAPON_REPEATER_CARBINE"), 50, 0, true);
                }
                SetRandomOutfitPreset(rustler);
                SetRandomOutfitPreset(horse);
                GiveSaddleToHorse(horse);
                PED::SET_PED_ONTO_MOUNT(rustler, horse, -1, true);

                if (GetRandomIntInRange(0, 100) < 50) {
                    TaskCombatPed(rustler, playerPed, 0);
                }
                else {
                    if (!herd.empty()) {
                        Ped targetCow = herd[GetRandomIntInRange(0, (int)herd.size() - 1)];
                        TaskCombatPed(rustler, targetCow, 0);
                    }
                }
            }

            STREAMING::SET_MODEL_AS_NO_LONGER_NEEDED(rustlerModel);
            STREAMING::SET_MODEL_AS_NO_LONGER_NEEDED(horseModel);
        }
        Notify("Rustlers Spotted! They are coming for you and your herd!");

        // Make cowboys defend against rustlers
        for (auto& cowboy : activeCowboys) {
            if (ENTITY::DOES_ENTITY_EXIST(cowboy.ped) && !ENTITY::IS_ENTITY_DEAD(cowboy.ped)) {
                PED::SET_PED_COMBAT_ATTRIBUTES(cowboy.ped, 5, true); // Can use cover
                PED::SET_PED_COMBAT_ATTRIBUTES(cowboy.ped, 46, true); // Will fight
                PED::SET_PED_COMBAT_ATTRIBUTES(cowboy.ped, 17, false); // Disable flee

                // Cowboys will automatically engage hostile peds nearby
                PED::SET_PED_RELATIONSHIP_GROUP_HASH(cowboy.ped, joaat("PLAYER"));
            }
        }
    }
}

// ================== HERDING ==================
void UpdateHerdMovement(Ped playerPed) {
    Vector3 playerPos = ENTITY::GET_ENTITY_COORDS(playerPed, true, true);
    Vector3 herdCenter = CalculateHerdCenter();

    Vector3 dirFromPlayer = { herdCenter.x - playerPos.x, herdCenter.y - playerPos.y, 0 };
    Vector3 normDir = NormalizeVector(dirFromPlayer);

    Vector3 herdTarget = { herdCenter.x + normDir.x * 20.0f, herdCenter.y + normDir.y * 20.0f, herdCenter.z };

    for (size_t i = 0; i < herd.size(); ++i) {
        Ped cow = herd[i];
        if (cow == herdLeader) continue;
        if (!ENTITY::DOES_ENTITY_EXIST(cow)) continue;

        Vector3 cowPos = ENTITY::GET_ENTITY_COORDS(cow, true, true);
        float distToPlayer = CalculateDistance(playerPos, cowPos);

        bool isStraggler = std::find(stragglerCows.begin(), stragglerCows.end(), cow) != stragglerCows.end();

        if (isStraggler) {
            // Stragglers sometimes wander off
            if (GetRandomIntInRange(0, 100) < 10) { // 10% chance per update
                Vector3 wanderTarget = {
                    cowPos.x + GetRandomFloatInRange(-20.0f, 20.0f),
                    cowPos.y + GetRandomFloatInRange(-20.0f, 20.0f),
                    cowPos.z
                };
                TASK::TASK_FOLLOW_NAV_MESH_TO_COORD(cow, wanderTarget.x, wanderTarget.y, wanderTarget.z, 0.8f, -1, 0.5f, 0, 0.f);
                continue; // skip normal herding behavior
            }
        }

        if (distToPlayer <= MAX_HERDING_DISTANCE) {
            float offsetX = -2.0f * (i % 5) - normDir.x * (i / 5) * 2.0f;
            float offsetY = -2.0f * (i % 5) - normDir.y * (i / 5) * 2.0f;
            Vector3 target = { herdTarget.x + offsetX, herdTarget.y + offsetY, herdTarget.z };

            TASK::TASK_FOLLOW_NAV_MESH_TO_COORD(cow, target.x, target.y, target.z, herdSpeed, -1, 0.5f, 0, 0.f);

        }
        else {
            TASK::CLEAR_PED_TASKS(cow, true, true);
        }
    }
}

// ================== MAIN LOOP ==================
void main() {
    // Setup blips and prompts
    int sellBlipValentine = CreateBlipForCoords(joaat("BLIP_STYLE_CAMP"), joaat("blip_ambient_herd"), "Sell Cows", sellLocationValentine);
    int buyBlipEmerald = CreateBlipForCoords(joaat("BLIP_STYLE_CAMP"), joaat("blip_ambient_herd"), "Buy Cows", buyLocationEmerald);
    int sellBlipBlackwater = CreateBlipForCoords(joaat("BLIP_STYLE_CAMP"), joaat("blip_ambient_herd"), "Sell Cows", sellLocationBlackwater);
    int buyBlipMcFarlane = CreateBlipForCoords(joaat("BLIP_STYLE_CAMP"), joaat("blip_ambient_herd"), "Buy Cows", buyLocationMcFarlane);
    RegisterPrompt(buyPrompt, "INPUT_CONTEXT_A", "Buy Cows");
    RegisterPrompt(sellPrompt, "INPUT_CONTEXT_X", "Sell Cows");
    RegisterPrompt(SpeedUpHerdPrompt, "INPUT_FRONTEND_UP", "Run Herd"); // Run Herd (D-Pad Up)
    RegisterPrompt(SlowDownHerdPrompt, "INPUT_FRONTEND_RIGHT", "Walk Herd"); // Walk Herd (D-Pad Right)
    RegisterPrompt(stopHerdPrompt, "INPUT_FRONTEND_DOWN", "Stop Herd"); // Stop Herd (D-Pad Down)
    RegisterPrompt(hireCowboyPrompt, "INPUT_CONTEXT_B", "Hire Cowboy");
    RegisterPrompt(dismissCowboyPrompt, "INPUT_CONTEXT_Y", "Dismiss Cowboy");

    Hash cowModel = joaat("A_C_COW");

    while (true) {
        Ped playerPed = PLAYER::PLAYER_PED_ID();
        Vector3 playerPos = GetEntityCoords(playerPed);
        int now = MISC::GET_GAME_TIMER();

        // Make sure daily market modifier updates properly
        UpdateMarketModifier();
        
        // ================= COWBOY SYSTEM MAINTENANCE =================
        static int lastCowboyUpdate = 0;
        if (now - lastCowboyUpdate > 5000) { // Every 5 seconds
            CleanupCowboys();
            HandleCowboyWages();
            lastCowboyUpdate = now;
        }
        // Show dismiss prompt when focusing on cowboys
        bool nearCowboy = false;
        Ped targetCowboy = 0;

        // Check if player is free-aiming
        if (PLAYER::IS_PLAYER_FREE_AIMING(PLAYER::PLAYER_ID())) {
            Ped aimedPed;
            if (PLAYER::GET_ENTITY_PLAYER_IS_FREE_AIMING_AT(PLAYER::PLAYER_ID(), &aimedPed)) {
                // Make sure it's one of your hired cowboys
                for (auto& cowboy : activeCowboys) {
                    if (ENTITY::DOES_ENTITY_EXIST(cowboy.ped) && cowboy.ped == aimedPed) {
                        nearCowboy = true;
                        targetCowboy = cowboy.ped;
                        break;
                    }
                }
            }
        }

        // Fallback: still allow distance-based dismissal (optional)
        if (!nearCowboy && !activeCowboys.empty()) {
            Ped playerPed = PLAYER::PLAYER_PED_ID();
            Vector3 playerPos = GetEntityCoords(playerPed);
            bool playerOnHorse = PED::IS_PED_ON_MOUNT(playerPed);

            for (auto& cowboy : activeCowboys) {
                if (ENTITY::DOES_ENTITY_EXIST(cowboy.ped)) {
                    Vector3 cowboyPos = GetEntityCoords(cowboy.ped);
                    float distance = CalculateDistance(playerPos, cowboyPos);
                    float maxRange = playerOnHorse ? 50.0f : 10.0f;

                    if (distance < maxRange) {
                        nearCowboy = true;
                        targetCowboy = cowboy.ped;
                        break;
                    }
                }
            }
        }

        TogglePrompt(dismissCowboyPrompt, nearCowboy);
        
        // ================= HERDING PROMPTS =================
        // Link the entire prompt group visibility directly to the herdingActive state.
        // The prompts will stay visible as long as herding is active.
        TogglePrompt(SpeedUpHerdPrompt, herdingActive);
        TogglePrompt(SlowDownHerdPrompt, herdingActive);
        TogglePrompt(stopHerdPrompt, herdingActive);

        // Handle prompt actions if herding is active
        if (herdingActive) {
            if (HasPromptHoldModeCompleted(SpeedUpHerdPrompt)) {
                SetHerdSpeed(2.0f);
            }
            if (HasPromptHoldModeCompleted(SlowDownHerdPrompt)) {
                SetHerdSpeed(1.0f);
            }
            if (HasPromptHoldModeCompleted(stopHerdPrompt)) {
                SetHerdSpeed(0.0f);
            }
        }

        // ================= BUY/SELL =================
        // Check buy locations (Emerald Ranch and McFarlane)
        float distBuyEmerald = CalculateDistance(playerPos, buyLocationEmerald);
        float distBuyMcFarlane = CalculateDistance(playerPos, buyLocationMcFarlane);
        bool nearBuyLocation = (distBuyEmerald < 5.0f || distBuyMcFarlane < 5.0f);

        if (nearBuyLocation) {
            TogglePrompt(buyPrompt, true);
            if (HasPromptHoldModeCompleted(buyPrompt)) {
                WAIT(500);
                BuyCow();
            }
        }
        else {
            TogglePrompt(buyPrompt, false);
        }

        // ================= COWBOY MANAGEMENT =================
        // Show hire prompt near towns or buy locations
        bool canHireCowboys = nearBuyLocation && (int)activeCowboys.size() < maxCowboys;
        TogglePrompt(hireCowboyPrompt, canHireCowboys);

        if (HasPromptHoldModeCompleted(hireCowboyPrompt)) {
            WAIT(500);
            HireCowboy();
        }
        if (HasPromptHoldModeCompleted(dismissCowboyPrompt)) {
            DismissCowboy(targetCowboy);
        }

        // Check sell locations (Valentine and Blackwater)
        float distSellValentine = CalculateDistance(playerPos, sellLocationValentine);
        float distSellBlackwater = CalculateDistance(playerPos, sellLocationBlackwater);
        bool nearSellLocation = (distSellValentine < 40.0f || distSellBlackwater < 40.0f);

        TogglePrompt(sellPrompt, nearSellLocation);

        if (HasPromptHoldModeCompleted(sellPrompt)) {
            if (herd.empty()) {
                
            }
            else {
                AutoSellHerd();
                int cowPrice = 2500;
                int numCows = (int)herd.size();
                int finalPrice = cowPrice * numCows;

                // Apply market fluctuation
                finalPrice = (int)(finalPrice * dailyMarketModifier);
                if (dailyMarketModifier > 1.0f) {
                    Notify("Market is strong today, buyers pay more");
                }
                else if (dailyMarketModifier < 1.0f) {
                    Notify("Market is weak today, prices are down");
                }

                // Apply modifiers
                if (ranchReputation > 20) {
                    finalPrice = (int)(finalPrice * 1.1f);
                }
                else if (ranchReputation < -10) {
                    finalPrice = (int)(finalPrice * 0.9f);
                }
                finalPrice += herdingSkill;

                // Update money
                MONEY::_MONEY_INCREMENT_CASH_BALANCE(finalPrice, 3000);

                // Update progression
                ChangeReputation(1);
                ImproveHerding((numCows > 10) ? 1 : 0);
                herd.clear();
            }
        }

        if (IsKeyJustUp(0x55)) { // U // HUD toggle
            showHerdHUD = !showHerdHUD;
            DisplayLeftToast("Debug HUD", showHerdHUD ? "Enabled" : "Disabled", nullptr, nullptr, 1500);
        }

        // ================= HERDING =================
        if (herdingActive && !herd.empty()) {
            herd.erase(std::remove_if(herd.begin(), herd.end(), [](Ped p) {
                return !ENTITY::DOES_ENTITY_EXIST(p) || ENTITY::IS_ENTITY_DEAD(p);
                }), herd.end());

            // Clean up stragglers too
            stragglerCows.erase(std::remove_if(stragglerCows.begin(), stragglerCows.end(),
                [](Ped p) { return !ENTITY::DOES_ENTITY_EXIST(p) || ENTITY::IS_ENTITY_DEAD(p); }),
                stragglerCows.end());

            int currentTime = MISC::GET_GAME_TIMER();
            if (currentTime - lastUpdateTime > updateInterval) {
                UpdateHerdMovement(playerPed);
                UpdateCowboyAI(); // Update AI cowboys
                lastUpdateTime = currentTime;
            }
        }

        // ================= RANDOM EVENTS =================
        if (herdingActive && !herd.empty()) {
            int now = MISC::GET_GAME_TIMER();

            if (now - lastEventTime > eventCooldown) {
                lastEventTime = now;

                if (GetRandomIntInRange(0, 100) < randomEventChance) {
                    TriggerRandomEvent(playerPed);
                }
            }
        }

        // ================= DEBUG VISUALS =================
        if (showHerdHUD) {
            float textY = 0.05f;
            float lineHeight = 0.025f;

            // Main herding info
            DrawTextToScreen("ToggleHud (U) Herding (K): " + std::string(herdingActive ? "ACTIVE" : "INACTIVE"), 0.05f, textY, 0.4f, 255, 255, 255, 255);
            textY += lineHeight;

            DrawTextToScreen("Herd size: " + std::to_string(herd.size()), 0.05f, textY, 0.4f, 255, 255, 255, 255);
            textY += lineHeight;

            DrawTextToScreen("Spawn cows/amount (O): " + std::to_string(spawnedCows.size()), 0.05f, textY, 0.4f, 255, 255, 255, 255);
            textY += lineHeight;

            // Cowboy system info
            DrawTextToScreen("Cowboys hired: " + std::to_string(activeCowboys.size()) + "/" + std::to_string(maxCowboys), 0.05f, textY, 0.4f, 255, 255, 255, 255);
            textY += lineHeight;

            // Show individual cowboy info
            for (size_t i = 0; i < activeCowboys.size() && i < 3; i++) {
                if (ENTITY::DOES_ENTITY_EXIST(activeCowboys[i].ped)) {
                    std::string cowboyInfo = "  " + activeCowboys[i].name + " (Skill:" + std::to_string(activeCowboys[i].skillLevel) +
                        " " + (activeCowboys[i].isWorking ? "WORKING" : "IDLE") + ")";
                    DrawTextToScreen(cowboyInfo, 0.05f, textY, 0.35f, 150, 255, 150, 255);
                    textY += lineHeight;
                }
            }

            // Progression info
            textY += lineHeight * 0.5f; // Small gap
            DrawTextToScreen("Reputation: " + std::to_string(ranchReputation), 0.05f, textY, 0.4f, 255, 255, 255, 255);
            textY += lineHeight;

            DrawTextToScreen("Herding Skill: " + std::to_string(herdingSkill), 0.05f, textY, 0.4f, 255, 255, 255, 255);
            textY += lineHeight;

            // Toggle options
            textY += lineHeight * 0.5f; // Small gap
            DrawTextToScreen("Toggle Rustlers (R): " + std::string(rustlersEnabled ? "ENABLED" : "DISABLED"), 0.05f, textY, 0.4f, 255, 255, 255, 255);
            textY += lineHeight;

            DrawTextToScreen("Toggle Cowboys (C): " + std::string(cowboysEnabled ? "ENABLED" : "DISABLED"), 0.05f, textY, 0.4f, 255, 255, 255, 255);

            // ================= DEBUG KEYS =================
            if (IsKeyJustUp(0x4F)) { //  O // Spawn cow(s)
                WAIT(500); // debounce
                for (int i = 0; i < 10; i++) {
                    Ped cow = CreatePed(cowModel, GetRandomOffsetFromCoords(playerPos, 1.0f, 2.5f), 0, 0);
                    spawnedCows.push_back(cow);
                    herd.push_back(cow);
                    PED::SET_BLOCKING_OF_NON_TEMPORARY_EVENTS(cow, true);
                }
            }
            if (IsKeyJustUp(0x50)) { // P // copy player coords to clipboard / log
                CopyCoords();
            }
            if (IsKeyJustUp(0x4B)) { // K // Toggle Herding 
                herdingActive = !herdingActive;
                if (!herdingActive) {
                    for (Ped p : herd) {
                        if (ENTITY::DOES_ENTITY_EXIST(p)) {
                            TASK::CLEAR_PED_TASKS(p, true, true);
                        }
                    }
                }
            }
            if (IsKeyJustUp(0x52)) { // R key // Toggle rustler events
                rustlersEnabled = !rustlersEnabled;
                if (rustlersEnabled) {
                    Notify("Rustler attacks ENABLED.");
                }
                else {
                    Notify("Rustler attacks DISABLED.");
                }
            }

            if (IsKeyJustUp(0x43)) { // C key // Toggle cowboy system
                cowboysEnabled = !cowboysEnabled;
                if (!cowboysEnabled) {
                    // Dismiss all cowboys
                    for (auto& cowboy : activeCowboys) {
                        if (ENTITY::DOES_ENTITY_EXIST(cowboy.ped)) {
                            DeleteEntity(&cowboy.ped);
                        }
                        if (ENTITY::DOES_ENTITY_EXIST(cowboy.horse)) {
                            DeleteEntity(&cowboy.horse);
                        }
                    }
                    activeCowboys.clear();
                    Notify("Cowboy system DISABLED.");
                }
                else {
                    Notify("Cowboy system ENABLED.");
                }
            }
        }

        if (notifyText != "" && GetGameTimer() - notifyStart < notifyDuration) {
            DrawTextToScreen(notifyText, 0.40f, 0.85f, 0.5f, 213, 225, 224, 255);
        }

        WAIT(0);
    }
}

void ScriptMain() {
    srand(GetTickCount64());
    main();
}