-- This Source Code Form is subject to the terms of the bCDDL, v. 1.1.
-- If a copy of the bCDDL was not distributed with this
-- file, You can obtain one at http://beamng.com/bCDDL-1.1.txt

local M = {}

local helper = require('scenario/scenariohelper')

local running = false
local playerOnCorrectPath = false

local checkForRobberCrash = false
local checkForPoliceDistance = false
local robberCrashTimeout = 0
local robberCrashMaxTimeout = 7

local maxPoliceDamage = 4000
local maxPoliceDistance = 400
local policeDistanceEnable = 50
local robberMinimumSpeed = 1.4 --m/s

local potentialTargets = {'wp_swamp', 'wp_farm', 'wp_sawmill', 'wp_mansion', 'wp_house_shore', 'wp_woods', 'wp_house_woods', 'd_clifftrail_b9'}
local targetName = nil

local function reset()
    helper.setAiMode('robber', 'stopping')
    checkForRobberCrash = false
    checkForPoliceDistance = false
    robberCrashTimeout = 0
    running = false

    playerOnCorrectPath = false
end

local function fail(reason)
    scenario_scenarios.finish({failed = reason})
    reset()
end

local function success()
    local scenario = scenario_scenarios.getScenario()
    if not scenario then return end

    local finalTime = scenario.timer
    local minutes = math.floor(finalTime / 60);
    local seconds = finalTime - (minutes * 60);
    local timeStr = ''
    if minutes > 0 then
        timeStr = string.format("%02.0f:%05.2f", minutes, seconds)
    else
        timeStr = string.format("%0.2f", seconds) .. 's'
    end

    local result = {msg = {txt = 'scenarios.east_coast_usa.bank_robbery.win.msg', context={timeLimit=timeStr}}}
    scenario_scenarios.finish(result)
    reset()
end

-- called when countdown finished
local function onRaceStart()
    reset()
    running = true
    math.randomseed(os.time())
    targetName = potentialTargets[math.random(#potentialTargets)]

    helper.trackVehicle('scenario_player0', 'policeCar')
    helper.trackVehicle('robber', 'robber')
    helper.queueLuaCommandByName('scenario_player0', 'electrics.set_lightbar_signal(2)')
end

-- data { vehicleName = 'string', waypointName = 'string' }
local function onRaceWaypointReached( data )
    local playerVehicleId = be:getPlayerVehicleID(0)
    if data.vehicleId == playerVehicleId and data.waypointName == 'br_wp_start_ai' then
        helper.flashUiMessage('scenarios.east_coast_usa.bank_robbery.onRaceWaypointReached.msg', 2)
        helper.setAiMode('robber', 'flee', 'scenario_player0')
        helper.setAiTarget('robber', targetName)
        helper.setAiAggression('robber', 0.75)
        --helper.queueLuaCommandByName('robber', 'ai.setState({debugMode="speeds"})')
        helper.trackVehicle('robber', 'robber')

    playerOnCorrectPath = true

    end
end

local function onBeamNGTrigger(data)
  if data.triggerName == 'failTrigger1' and data.event == 'exit' and playerOnCorrectPath == false and running == true then -- If the player takes another route to go to the robber
    fail('scenarios.east_coast_usa.bank_robbery.lost.fail.msg')
  end
end

-- called every 250ms. Use "status.setScenarioFailed()" for return failed test
-- data { dTime = number }
local function onRaceTick(raceTickTime)
    if not running then
        return true
    end

    local policeData = map.objects[map.objectNames['policeCar']]
    local robberData = map.objects[map.objectNames['robber']]

    if not policeData or not robberData then
        return
    end

    local distanceToRobber = math.abs((policeData.pos - robberData.pos):length())
    local robberSpeed = robberData.vel:length()

    if not checkForPoliceDistance and distanceToRobber < policeDistanceEnable then
        checkForPoliceDistance = true
    end

    if not checkForRobberCrash and robberSpeed > robberMinimumSpeed and playerOnCorrectPath == true then
        checkForRobberCrash = true
    end

    if checkForPoliceDistance and distanceToRobber > maxPoliceDistance then
        fail('scenarios.east_coast_usa.bank_robbery.distance.fail.msg')
    end

    if policeData.damage > maxPoliceDamage then
        fail('scenarios.east_coast_usa.bank_robbery.damage.fail.msg')
    end

    if checkForRobberCrash and (robberSpeed < robberMinimumSpeed) then
        robberCrashTimeout = robberCrashTimeout + raceTickTime
        if robberCrashTimeout % 1 == 0 then
            local countdown = robberCrashMaxTimeout - math.modf(robberCrashTimeout)
            if countdown > 0 then
                helper.flashUiMessage({txt = "scenarios.east_coast_usa.bank_robbery.countdown", context = {countdown = tostring(countdown)}}, 0.5, true)
            end
        end
    else
        robberCrashTimeout = 0
    end

    if robberCrashTimeout >= robberCrashMaxTimeout then -- wait x seconds
        success()
    end

    local distance = helper.getDistanceBetweenSceneObjects('robber', targetName)
    if distance >= 0 and distance <= 5 then
        fail("scenarios.east_coast_usa.bank_robbery.distancebtwsceneobject.fail.msg")
    end
end

M.onRaceStart = onRaceStart
M.onRaceWaypointReached = onRaceWaypointReached
M.onBeamNGTrigger = onBeamNGTrigger
M.onRaceTick = onRaceTick

return M
