-- This Source Code Form is subject to the terms of the bCDDL, v. 1.1.
-- If a copy of the bCDDL was not distributed with this
-- file, You can obtain one at http://beamng.com/bCDDL-1.1.txt

local M = {}
local luaLoaded = false

-- TWEAKS
local partitionGridSpaceSize = 256.0
local lightGroupNames = {}


-- DATA, DON'T TOUCH
local partitionGridMin
local partitionGridMax
local partitionGridSpacesX
local partitionGridSpacesY
local totalPartitionGridSpaces
local partitionGrid = {}

local lastValue = nil
local lastCameraCell = -1
local allLights = {}

local lightmgr

local sparkParticles = {}

-- Keep track of the last set of lights we enabled so we don't have to loop over every light and disable it each time we change cells
local lastEnabledLightCount = 0
local lastEnabledLightIndices = {}

local function enableLights(xCell, zCell, value)
  -- disable previously enabled lights
  for i=1,lastEnabledLightCount,1 do
    local light = allLights[lastEnabledLightIndices[i]]
    light:setLightEnabled(false)
  end

  lastEnabledLightCount = 0
  for x=math.max(0, xCell - 1), math.min(partitionGridSpacesX - 1, xCell + 1),1 do
    for z=math.max(0, zCell - 1), math.min(partitionGridSpacesY - 1, zCell + 1),1 do
      for _,lightIndex in ipairs(partitionGrid[1 + (z * partitionGridSpacesX) + x]) do
        -- mark enabled
        lastEnabledLightCount = lastEnabledLightCount + 1
        lastEnabledLightIndices[lastEnabledLightCount] = lightIndex

        -- enable light
        local light = allLights[lightIndex]
        light:setLightEnabled(value)
      end
    end
  end
end


local function getCellForPosition(p)
  local xCell = math.floor((p.x - partitionGridMin.x) / partitionGridSpaceSize)
  local yCell = math.floor((p.y - partitionGridMin.y) / partitionGridSpaceSize)
  return xCell, yCell
end

local function getCellIndexForPosition(p)
  local xCell, yCell = getCellForPosition(p)
  return (yCell * partitionGridSpacesX) + xCell
end

local function positionInRange(p)
  return p.x >= partitionGridMin.x and p.x <= partitionGridMax.x and p.y >= partitionGridMin.y and p.y <= partitionGridMax.y
end

local function onClientStartMission()
  luaLoaded = true
  lightmgr = tostring(TorqueScriptLua.getVar("$pref::lightManager"))
  local allObjects = scenetree:getAllObjects()
  for _, name in pairs(allObjects) do
    if string.endswith(name,'dynamic_lights') or string.endswith(name,'dynamic_lights2')  then
     table.insert(lightGroupNames, name)
    end
  end
  -- collect lights
  for _, lightGroupName in pairs(lightGroupNames) do
    local lightGroup = scenetree[lightGroupName]
    if lightGroup then
      for i = 0, lightGroup.obj:getCount(), 1 do
        local id = lightGroup.obj:idAt(i)
        local obj = scenetree.findObjectById(id)
        if obj and obj.obj:isSubClassOf('LightBase') then
          if obj:getDynDataFieldbyName('is_tunnel', 0) == "1" or obj:getName() == "thndr" then
            --do nothing
          else
            table.insert(allLights, obj.obj)
          end
        end
      end
    else
      print("ERROR: lightGroup " .. lightGroupName .. " not found!")
    end
  end

  -- compute grid
  local minX = 99999.0
  local minY = 99999.0
  local maxX = -99999.0
  local maxY = -99999.0

  for i,v in ipairs(allLights) do
    local pos = v:getPosition()
    minX = math.min(pos.x, minX)
    minY = math.min(pos.y, minY)

    maxX = math.max(pos.x, maxX)
    maxY = math.max(pos.y, maxY)
  end

  partitionGridMin = vec3(minX, minY, 0)
  partitionGridMax = vec3(maxX, maxY, 0)
  partitionGridSpacesX = math.ceil((partitionGridMax.x - partitionGridMin.x) / partitionGridSpaceSize)
  partitionGridSpacesY = math.ceil((partitionGridMax.y - partitionGridMin.y) / partitionGridSpaceSize)
  totalPartitionGridSpaces = (partitionGridSpacesX * partitionGridSpacesY)

  -- allocate tables
  for i=1,totalPartitionGridSpaces,1 do
    table.insert(partitionGrid, {})
  end

  for i=1,512,1 do
    table.insert(lastEnabledLightIndices, 0)
  end

  -- place lights in cells
  for i,v in ipairs(allLights) do
    local pos = v:getPosition()
    table.insert(partitionGrid[getCellIndexForPosition(pos) + 1], i)
  end

  -- turn off all lights
  for i,v in ipairs(allLights) do
    v:setLightEnabled(false)
  end

  -- debug
  local largestCell = -1
  local largestCellIndex = 0
  local totalLightsInCells = 0

  for i,c in ipairs(partitionGrid) do
    if #c > largestCell then
      largestCellIndex = i
      largestCell = #c
    end
    totalLightsInCells = totalLightsInCells + #c
  end

  --print("Tot spaces: " .. tostring(totalPartitionGridSpaces))
  --print("X spaces: " .. tostring(partitionGridSpacesX))
  --print("Y spaces: " .. tostring(partitionGridSpacesY))
  --print("Min extents: " .. tostring(partitionGridMin))
  --print("Max extents: " .. tostring(partitionGridMax))
  --print("Worst bucket was " .. tostring(largestCellIndex) .. " containing " .. tostring(largestCell) .. ". Total " .. tostring(totalLightsInCells))
end

local lastActive = nil

local function onUpdate(dtReal)
  if luaLoaded == false then
    extensions.mainLevel.onClientStartMission()
  end
  local tod = scenetree.tod
  if not tod then return end

  local value = (tod.time > 0.21 and tod.time < 0.79)
  local cameraCell = getCellIndexForPosition(core_camera.getPosition())

  if (cameraCell ~= lastCameraCell or lastValue ~= value) and positionInRange(core_camera.getPosition())  then
    local xCell, zCell = getCellForPosition(core_camera.getPosition())
    enableLights(xCell, zCell, value)
  end
  lastCameraCell = cameraCell
  lastValue = value
end

M.onClientStartMission = onClientStartMission
M.onUpdate = onUpdate

return M