-- This Source Code Form is subject to the terms of the bCDDL, v. 1.1.
-- If a copy of the bCDDL was not distributed with this
-- file, You can obtain one at http://beamng.com/bCDDL-1.1.txt

local M = {}
local logTag = 'guineapig'
local playerInstance = 'scenario_player0'
local cargoInstance = 'covet'
local running = false
local cargoDelivered = false
local cargoFellOff = false
local trailerAttached = false
local displayedRouteInfo = false
local displayedZoomInfo = false
local messageDelayTimer = 0
local autoConnectTrailer = true
local attemptedAutoAttach = false
local attemptTimer = 0

local function reset()
  running = false
  cargoDelivered = false
  cargoFellOff = false
  trailerAttached = false
  displayedRouteInfo = false
  displayedZoomInfo = false
  messageDelayTimer = 0
end

local function fail(reason)
  scenario_scenarios.finish({failed = reason})
  reset()
end

local function success(reason)
  scenario_scenarios.finish({msg = reason})
  reset()
end

local function onRaceStart()
  -- log('I', logTag,'onRaceStart called')
  reset()
  scenario_scenarios.trackVehicleMovementAfterDamage(playerInstance, {waitTimerLimit=2})
  running = true

end

local function onRaceResult()
  if cargoDelivered then
    success('scenarios.utah.guineapig.pass.msg')
  else
    if cargoFellOff then
      fail('scenarios.utah.guineapig.cargoFellOff.fail.msg')
    else
      fail('scenarios.utah.guineapig.nocargo.msg')
    end
  end
end

local function onPreRender(dt)
  if not running then
    return
  end

  local playerVehicle = scenetree.findObject(playerInstance)
  local trailer = scenetree.findObject('trailer')
  local cargo = scenetree.findObject(cargoInstance)

  if playerVehicle and trailer and cargo then
    messageDelayTimer = messageDelayTimer + dt
    if not trailerAttached and messageDelayTimer >= 1 then
      ui_message("Attach the trailer to the vehicle using  [action=couplersToggle]", 1, 'GUINEAPIG', nil)
      messageDelayTimer = messageDelayTimer - 1
    end


    local playerPos = playerVehicle:getPosition()
    local trailerPos = trailer:getPosition()
    local trailerBSphereRadius = trailer:getBSphereRadius()
    local cargoPos = cargo:getPosition()
    -- local cargoRadius = cargo:getBSphereRadius()
    local cargoTrailerDistance = (cargoPos - trailerPos):len()
    if trailerAttached and (cargoTrailerDistance > trailerBSphereRadius) then
      if not cargoDelivered then
        cargoFellOff = true
        running = false
        scenario_scenarios.endRace()
      end
    end

    local distanceToTrailer = (playerPos - trailerPos):len()
    -- log('I', logTag, 'distanceToTrailer: '..distanceToTrailer)
    if autoConnectTrailer and not attemptedAutoAttach and not trailerAttached and distanceToTrailer <= 7 then
      log('D', logTag,'Auto attaching couplers...')
      playerVehicle:queueLuaCommand('beamstate.toggleCouplers()')
      attemptedAutoAttach = true
      attemptTimer = 0
    end

    if attemptedAutoAttach then
      attemptTimer = attemptTimer + dt
      if not trailerAttached and attemptTimer >= 0.25 and distanceToTrailer >= 9 then
        log('D', logTag,'Auto deattaching couplters...')
        playerVehicle:queueLuaCommand('beamstate.toggleCouplers()')
        attemptedAutoAttach = false
      end
    end

    -- debugDrawer:drawSphere(vec3(trailerPos), 0.05, ColorF(1.0,1.0,1.0,1.0))
    -- debugDrawer:drawSphere(vec3(trailerPos), trailerBSphereRadius, ColorF(1.0,0.0,0.0,0.4))
    -- debugDrawer:drawSphere(vec3(cargoPos), 0.05, ColorF(1.0,1.0,1.0,1.0))
    -- debugDrawer:drawSphere(vec3(cargoPos), cargoRadius, ColorF(0.0,1.0,0.0,0.4))
  end
end

local function onBeamNGTrigger(data)
  local cargo = scenetree.findObject(cargoInstance)
  -- log('I', logTag,'onBeamNGTrigger called...')
  if data.event == 'enter' then
    if cargo and data.triggerName == 'delivery_zone' then
      if data.subjectID and data.subjectID == cargo:getID() then
        cargoDelivered = true
      end
    end
  end
end

local function onVehicleStoppedMoving(vehicleID, damaged)
  if running then
    local playerVehicleID = scenetree.findObject(playerInstance):getID()
    if vehicleID == playerVehicleID and damaged then
      if not cargoDelivered then
       running = false
       scenario_scenarios.endRace()
      end
    end
  end
end

local function onCouplerAttached(objId1, objId2)
  --log('I', logTag,'onCouplerAttached called... objId1: '..objId1..' objId2: '..objId2)
  Engine.Audio.playOnce('AudioGui','event:>Vehicle>Coupler_Attach_2D')

  trailerAttached = true
  messageDelayTimer = 0
  autoConnectTrailer = false
  attemptedAutoAttach = false
end

local function onCouplerDetached(objId1, objId2)
  --log('I', logTag,'onCouplerDetached called... objId1: '..objId1..' objId2: '..objId2)
  Engine.Audio.playOnce('AudioGui','event:>Vehicle>Coupler_Detach_2D')
  trailerAttached = false
  messageDelayTimer = 1
end

M.onRaceStart = onRaceStart
M.onRaceResult = onRaceResult
M.onBeamNGTrigger = onBeamNGTrigger
M.onVehicleStoppedMoving = onVehicleStoppedMoving
M.onPreRender = onPreRender
M.onCouplerAttached = onCouplerAttached
M.onCouplerDetached = onCouplerDetached

return M