-- This Source Code Form is subject to the terms of the bCDDL, v. 1.1.
-- If a copy of the bCDDL was not distributed with this
-- file, You can obtain one at http://beamng.com/bCDDL-1.1.txt

-- In other words, if you are a modder feel free to reuse this code with above license

local M = {}
M.dependencies = {"ui_imgui"}
local im = ui_imgui
local ffi = require("ffi")

local internal = not shipping_build or not string.match(beamng_windowtitle, "RELEASE")

local TorqueScriptLua = TorqueScriptLua

-- TWEAKS
local partitionGridSpaceSize = 110.0
local lightGroupNames = {"DynamicLights"}

--independent GUI
local showUI = nil
local pos = im.ImVec2(0, 0)
local skipImgui = false

--scenetree
local objects = nil

local tod = nil

-- DATA, DON'T TOUCH
local partitionGridMin
local partitionGridMax
local partitionGridSpacesX
local partitionGridSpacesY
local totalPartitionGridSpaces
local partitionGrid = {}
local luaLoaded = false

local lastValue = nil
local lastCameraCell = -1
local allLights = {}

local lightmgr
local shadowdisable = 0

local debugEnabled = false

local speedtrpgroup = im.ArrayChar(256)
local speedzonegroup = im.ArrayChar(256)
local cubemapgroup = im.ArrayChar(256)

local instanceData = {}

--weather
local isWet = 0

local isThunder = false

--level data
local levelfolder = nil
local levelname =  nil
local tool_version = "2.2"
local appTitle = " CK Map Controller - ".. tool_version .." - ".. "no-init" .." - ".. beamng_arch
local debugTitle = "Debug Controller - ".. tool_version .." - ".. "no-init" .." - ".. beamng_arch

local toolWindowName = 'map_controller'

-- Keep track of the last set of lights we enabled so we don't have to loop over every light and disable it each time we change cells
local lastEnabledLightCount = 0
local lastEnabledLightIndices = {}

local function enableLights(xCell, zCell, value, table)
  -- disable previously enabled lights
  for i=1,lastEnabledLightCount,1 do
    local light = allLights[lastEnabledLightIndices[i]]
    if light and scenetree.findObjectById(light:getID()) then light:setLightEnabled(false) else log('E', "enableLights", "Light " .. light:getName() .. " is not existing in mission!") end
  end

  lastEnabledLightCount = 0
  for x=math.max(0, xCell - 1), math.min(partitionGridSpacesX - 1, xCell + 1),1 do
    for z=math.max(0, zCell - 1), math.min(partitionGridSpacesY - 1, zCell + 1),1 do
      for _,lightIndex in ipairs(partitionGrid[1 + (z * partitionGridSpacesX) + x]) do
        -- mark enabled
        lastEnabledLightCount = lastEnabledLightCount + 1
        lastEnabledLightIndices[lastEnabledLightCount] = lightIndex

        -- enable light
        local light = allLights[lightIndex]
        if table then
          if tableContains(table, scenetree.findObjectById(light:getField('parentGroup',0)):getName()) then
            if light and scenetree.findObjectById(light:getID()) then light:setLightEnabled(value) else log('E', "enableLights", "Light " .. light:getName() .. " is not existing in mission!") end
          end
        else
          if light and scenetree.findObjectById(light:getID()) then light:setLightEnabled(value) else log('E', "enableLights", "Light " .. light:getName() .. " is not existing in mission!") end
        end
      end
    end
  end
end

local function getCellForPosition(p)
  local xCell = math.floor((p.x - partitionGridMin.x) / partitionGridSpaceSize)
  local yCell = math.floor((p.y - partitionGridMin.y) / partitionGridSpaceSize)
  return xCell, yCell
end

local function getCellIndexForPosition(p)
  local xCell, yCell = getCellForPosition(p)
  return (yCell * partitionGridSpacesX) + xCell
end

local function positionInRange(p)
  return p.x >= partitionGridMin.x and p.x <= partitionGridMax.x and p.y >= partitionGridMin.y and p.y <= partitionGridMax.y
end

local function onParticleActivity(group, value)
  if not group then
    log('E', "onParticleActivity", "Group is missing")
  else
    for i = 0, group.obj:getCount(), 1 do
      local id = group.obj:idAt(i)
      local particleEmitter = scenetree.findObjectById(id)
      particleEmitter.active = value
    end
  end
end

local function setAllLightsEnabled(group, value)
  for i = 0, group.obj:getCount(), 1 do
      local id = group.obj:idAt(i)
      local obj = scenetree.findObjectById(id)
      if obj and obj.obj:isSubClassOf('LightBase') then
          obj.obj:setLightEnabled( value )
      end
  end
end

local function getInstanceData()
  local instanceData = {}
  local materials = {}
  local allObjects = scenetree:getAllObjects()
  if allObjects then
    for k,name in pairs(allObjects) do
      local object = scenetree.findObject(name)
      if object and object.___type == "class<Material>" then
        if object:getField('instanceEmissive', 0) == "1" then
          if name ~= "SygnalizacjaSwiatloDzial" then materials[name] = true end
        end
      end
    end
    for k,name in pairs(allObjects) do
      local object = scenetree.findObject(name)
      if object and object:isSubClassOf("TSStatic") then
        if object:getDynDataFieldbyName('is_tunnel', 0) == "1" then
          -- do nothing
        else
          local matNames = object:getMaterialNames()
          for k,v in pairs(matNames) do
            if materials[v] == true then
              local objId = object:getId()
              local instColor = object:getField('instanceColor', 0)
              instanceData[objId] = instColor
            end
          end
        end
      end
    end
    for k,v in pairs(instanceData) do
      local instance = scenetree.findObjectById(k)
      if instance then
        instance:setField('instanceColor', 0, "Black")
        instance:updateInstanceRenderData()
      end
    end
  end
  return instanceData
end

local function dynamicSpeedDisplayReset(targetName)
  if targetName then
    local target = scenetree.findObject(targetName)
    if target then
      if target.aboveLimitWarn then
        local aboveLimittxt = scenetree.findObject(target.aboveLimitWarn)
        if aboveLimittxt then
          if aboveLimittxt.hidden ~= true then aboveLimittxt.hidden = true end
        end
      end
      if target.belowLimitWarn then
        local belowLimittxt = scenetree.findObject(target.belowLimitWarn)
        if belowLimittxt then
          if belowLimittxt.hidden ~= true then belowLimittxt.hidden = true end
        end
      end
      if target.aboveLimitTiersGroup then
        local tiersGroup = scenetree.findObject(target.aboveLimitTiersGroup)
        for i = 0, tiersGroup.obj:getCount(), 1 do
          local id = tiersGroup.obj:idAt(i)
          local obj = scenetree.findObjectById(id)
          if obj and obj.obj:isSubClassOf("TSStatic") and obj.name then
            if obj.hidden ~= true then obj.hidden = true end
          end
        end
      end
      if target.displayGroup then
        local displayGroup = scenetree.findObject(target.displayGroup)
        for i = 0, displayGroup.obj:getCount(), 1 do
          local id = displayGroup.obj:idAt(i)
          local obj = scenetree.findObjectById(id)
          if obj and obj.obj:isSubClassOf("TSStatic") and obj.name then
            if obj.hidden ~= true then obj.hidden = true end
          end
        end
      end
    end
  end
end

--simple dynamic level cubemap experiment
local cubemapname

local function setCubemap(cubemapname)
  if cubemapname then
    local levelinf = scenetree.findObject("theLevelInfo")
    if levelinf then
      local tod = scenetree.tod
      log("I", "setCubemap", "Enabling cubemap " ..cubemapname)
      levelinf:setField('globalEnviromentMap', 0, cubemapname)
      levelinf:postApply()
      if tod then
        core_environment.setTimeOfDay(tod)
      end
    else
      log("W", "setCubemap", "Enabling fallback cubemap " ..cubemapname)
      setConsoleVariable("$defaultLevelEnviromentMap", cubemapname)
    end
  end
end

local function onClientStartMission()
  print("DM light cull onClientStartMission")
  luaLoaded = true
  -- collect lights
  for _, lightGroupName in pairs(lightGroupNames) do
    local lightGroup = scenetree[lightGroupName]
    if lightGroup then
      for i = 0, lightGroup.obj:getCount(), 1 do
        local id = lightGroup.obj:idAt(i)
        local obj = scenetree.findObjectById(id)
        if obj and obj.obj:isSubClassOf('LightBase') then
          table.insert(allLights, obj.obj)
        end
      end
    else
      print("ERROR: lightGroup " .. lightGroupName .. " not found!")
    end
  end

  -- compute grid
  local minX = 99999.0
  local minY = 99999.0
  local maxX = -99999.0
  local maxY = -99999.0

  for i,v in ipairs(allLights) do
    local pos = v:getPosition()
    minX = math.min(pos.x, minX)
    minY = math.min(pos.y, minY)

    maxX = math.max(pos.x, maxX)
    maxY = math.max(pos.y, maxY)
  end

  partitionGridMin = vec3(minX, minY, 0)
  partitionGridMax = vec3(maxX, maxY, 0)
  partitionGridSpacesX = math.ceil((partitionGridMax.x - partitionGridMin.x) / partitionGridSpaceSize)
  partitionGridSpacesY = math.ceil((partitionGridMax.y - partitionGridMin.y) / partitionGridSpaceSize)
  totalPartitionGridSpaces = (partitionGridSpacesX * partitionGridSpacesY)

  -- allocate tables
  for i=1,totalPartitionGridSpaces,1 do
    table.insert(partitionGrid, {})
  end

  for i=1,512,1 do
    table.insert(lastEnabledLightIndices, 0)
  end

  -- place lights in cells
  for i,v in ipairs(allLights) do
    local pos = v:getPosition()
    table.insert(partitionGrid[getCellIndexForPosition(pos) + 1], i)
  end

  -- turn off all lights
  for i,v in ipairs(allLights) do
    v:setLightEnabled(false)
  end

  -- debug
  local largestCell = -1
  local largestCellIndex = 0
  local totalLightsInCells = 0

  for i,c in ipairs(partitionGrid) do
    if #c > largestCell then
      largestCellIndex = i
      largestCell = #c
    end
    totalLightsInCells = totalLightsInCells + #c
  end

  print("Tot spaces: " .. tostring(totalPartitionGridSpaces))
  print("X spaces: " .. tostring(partitionGridSpacesX))
  print("Y spaces: " .. tostring(partitionGridSpacesY))
  print("Min extents: " .. tostring(partitionGridMin))
  print("Max extents: " .. tostring(partitionGridMax))
  print("Worst bucket was " .. tostring(largestCellIndex) .. " containing " .. tostring(largestCell) .. ". Total " .. tostring(totalLightsInCells))

  if not TorqueScriptLua then
    TorqueScriptLua = TorqueScript
  end
  log('I', 'onClientStartMission', 'Getting current level path' )
  levelname = getCurrentLevelIdentifier()
  --print(levelname)
  log('I', 'onClientStartMission', 'Level is loaded' )
  levelfolder = ("/levels/" .. levelname .. "/")
  appTitle = " CK Map Controller - ".. tool_version .." - ".. levelname .." - ".. beamng_arch
  debugTitle = "Debug Controller - ".. tool_version .." - ".. levelname .." - ".. beamng_arch
  if TorqueScriptLua then
    lightmgr = tostring(TorqueScriptLua.getVar("$pref::lightManager"))
    log('I', 'onClientStartMission', lightmgr )
    shadowdisable = tonumber(TorqueScriptLua.getVar("$pref::Shadows::disable"))
    log('I','onClientStartMission', shadowdisable )
    instanceData = getInstanceData()
  end
  --set initial state
  if scenetree.findObject('Krzyz_animacja') then
    scenetree.findObject('Krzyz_animacja'):playAnim('krzyz_wraca')
  end
  setAllLightsEnabled(scenetree.Pochodnie_emittery_swiatla, true)
  onParticleActivity(scenetree.Pochodnie_emittery_swiatla, true)
  setAllLightsEnabled(scenetree.spotKrzyz, false)
  local allObjects = scenetree:getAllObjects()
  if allObjects then
    for k,name in pairs(allObjects) do
      if string.startswith(name, "tablica_trigger_") then
        dynamicSpeedDisplayReset(name)
      end
    end
  end
  setCubemap("cubemap_polish_roads_remaster_reflection")
end

--get scene tree all objects
local function getSimObjects(fileName)
  local ret = {}
  local objs = scenetree.getAllObjects()
  --log('E', '', '# objects existing: ' .. tostring(#scenetree.getAllObjects()))
  for _, objName in ipairs(objs) do
    local o = scenetree.findObject(objName)
    if o and o.getFileName then
      if o:getFileName() == fileName then
        table.insert(ret, o)
      end
    end
  end
  return ret
  --log('E', '', '# objects left: ' .. tostring(#scenetree.getAllObjects()))
end

local function getFontList()
  local fonts = {}
  for i = 0, im.IoFontsGetCount() - 1 do
    table.insert(fonts,
    {
      name = ffi.string(im.IoFontsGetName(i))
    })
  end
  return fonts
end

local function onUiChangedState(newUIState, prevUIState)
  if newUIState:sub(1, 4) == 'menu' then
    skipImgui = false
  else
    skipImgui = true
  end
end

--Hide Elements of map

local function setHiddenRec(object, hidden)
  object.hidden = hidden
  if object:isSubClassOf("SimSet") then
    for i=0, object:size() - 1 do
      setHiddenRec(object:at(i), hidden)
    end
  end
end

local function hideGroup(groupName, hidden)
  if not groupName then
    log('E', "", "Group is missing, please contact level author")
    guihooks.trigger('Message', {ttl = 10, msg = 'Group is missing, please contact level author', icon = "goat"})
  else
    local group = scenetree.findObject(groupName)
    if group then setHiddenRec(group, hidden) end
  end
end

local function onClientPostStartMission()
  if TorqueScriptLua then
    log('I','onClientPostStartMission', 'Hiding forest group' )
    hideGroup("godraypr", true)
  end
end

--change material data
local function setMaterialProperty(material, property, layer, value)
  --guihooks.trigger('menuHide')
  --guihooks.trigger('app:waiting', true)
  if type(material) == "string" then
    material = scenetree.findObject(material)
  end

  if material.___type == "class<Material>" then
    material:setField(property, layer, value)
    --we do not want to flush materials
    --material:flush()
    --print("reloading material")
    material:reload()
    --guihooks.trigger('app:waiting', false) -- shows the loading icon
    return true
  else
    log('E', "", "Given object is not a material.")
    return false
  end
end

local function getFadeScreenData(preview, sub, desc)
  local level_name = "Polish Roads"
  local preview = preview or nil
  if level_name == nil then
    if levelname == nil then level_name = "no_init" else level_name = levelname end
  end
  return {image = preview, title = level_name, subtitle = sub, text = desc}
end

local rainAmount = 0
local isUsingSkybox = false
--DynamicWeather
local function setWeather(fogscaleval, colorval, sunval, ambientval, fogamnt, fogHeight, group, rainamnt, vol, useSkybox, skyMat)
    local tod = scenetree.tod
    core_environment.setFogScaleGradientFile(fogscaleval)
    core_environment.setColorizeGradientFile(colorval)
    core_environment.setSunScaleGradientFile(sunval)
    core_environment.setAmbientScaleGradientFile(ambientval)
    core_environment.setFogDensity(fogamnt)
    core_environment.setFogAtmosphereHeight(fogHeight)
    core_environment.setNightFogGradientFile(fogscaleval)
    --core_environment.setPrecipitation(rainamnt)
    for i = 0, group.obj:getCount(), 1 do
      local id = group.obj:idAt(i)
      local precipitation = scenetree.findObjectById(id)
      local sfx
      if isThunder == true then
        sfx = scenetree.findObject("thndr_rain")
        local nightSkybox = scenetree.findObject("sunsky")
        nightSkybox.nightCubemap = "cubemap_cloudy-night"
        nightSkybox:postApply()
      else
        local nightSkybox = scenetree.findObject("sunsky")
        nightSkybox.nightCubemap = "TachoCubemap"
        nightSkybox:postApply()
        sfx = scenetree.findObjectById(id)
      end
      local skyBox = scenetree.findObject("SkyBox_1")
      if useSkybox == true then
        skyBox.Material = skyMat
        isUsingSkybox = true
        skyBox:postApply()
        skyBox.hidden = false
      else
        skyBox.hidden = true
        isUsingSkybox = false
      end
      precipitation.numDrops = rainamnt
      rainAmount = rainamnt
      sfx.volume = vol
      sfx:postApply()
    end
    core_environment.setTimeOfDay(tod)
    guihooks.trigger('Message', {ttl = 10, msg = 'Weather Changed', icon = "goat"})
end

--DYNAMIC MATERIALS SYSTEM - 2022 Car_Killer
--DAY/NIGHT Section
local function enabledisableEmissionGlow(isEmissive)
  if isEmissive == 1 then
    for k,v in pairs(instanceData) do
      local instance = scenetree.findObjectById(k)
      if instance then
        -- for now we will skip colors for objects to avoid weird issues on saving
        --instance:setField('instanceColor', 0, v)
        instance:setField('instanceColor', 0, "White")
        instance:updateInstanceRenderData()
      end
    end
  else
    for k,v in pairs(instanceData) do
      local instance = scenetree.findObjectById(k)
      if instance then
        instance:setField('instanceColor', 0, "Black")
        instance:updateInstanceRenderData()
      end
    end
  end
end

--WET/DRY Section
local function setDryWetMaterials(roughnessVal, groundVal)
  log('I', "setDryWetMaterials", "Switching Materials...")
  if levelname == nil then
    log('E', "setDryWetMaterials", "No init, please restart the game")
    guihooks.trigger('Message', {ttl = 10, msg = 'No init, please restart the game', icon = "goat"})
  else
    local materialFiles = FS:findFiles(levelfolder, "road.materials.json", -1, true, false)
    log('D', 'setDryWetMaterials', dumps(materialFiles))
    for _, fn in ipairs(materialFiles) do
      local dir, basefilename, ext = path.splitWithoutExt(fn)

      if string.find(fn, 'materials.cs$') then
        TorqueScript.exec(fn)
        objects = getSimObjects(fn)
      elseif string.find(fn, 'materials.json$') then
        loadJsonMaterialsFile(fn)
        objects = getSimObjects(fn)
      end

      if not tableIsEmpty(objects) then
        log('I', 'setDryWetMaterials', 'parsing all materials file: ' .. tostring(fn))

        for _, obj in ipairs(objects) do
          log('I', 'setDryWetMaterials', ' * ' .. tostring(obj:getClassName()) .. ' - ' .. tostring(obj:getName()) .. ' - Roughness: ' .. tostring(obj:getField('roughnessFactor', 0)) )
          log('I', 'setDryWetMaterials', ' * ' .. tostring(obj:getClassName()) .. ' - ' .. tostring(obj:getName()) .. ' - GroundType: ' .. tostring(obj:getField('groundType', 0)) )
          setMaterialProperty(obj:getName(), 'roughnessFactor', 0, roughnessVal)
          setMaterialProperty(obj:getName(), 'groundType', 0, groundVal)
        end

      end
    end
    be:reloadCollision()
    log('I', "setDryWetMaterials", "Switching Done!")
  end
end

local thndrcnt = 0
local delay = 0
local isPlayedA = false
local isPlayedB = false
local isPlayedC = false
local isPlayedD = false
local realTime = 0
local rndcont = 0

local randomoffset1 = math.random(-3000, 3000)
local randomoffset2 = math.random(-3000, 3000)
local randomoffset3 = math.random(150, 300)

local function onWeatherThunder(dtSim, dtReal)
  local paused = dtSim < 0.00001
  if not paused then
    local playerWPos
    realTime = realTime + dtReal
    thndrcnt = thndrcnt + dtSim
    if realTime >= 5 then
      rndcont = math.random(5, 60)
      if math.random(1, 8) == 4 then
        randomoffset1 = math.random(-500, 500)
        randomoffset2 = math.random(-500, 500)
        randomoffset3 = math.random(1, 200)
      else
        randomoffset1 = math.random(-3000, 3000)
        randomoffset2 = math.random(-3000, 3000)
        randomoffset3 = math.random(1, 200)
      end
      realTime = 0
    end
    if thndrcnt >= rndcont then
      --FIX FUCKIN MATH THERE, THERE IS A BETTER WAY TO DO THIS
      -- done :)
      local veh = be:getPlayerVehicle(0)
      local sounddelay = 0
      local dir = levelfolder.."/art/sound/"
      local sndTable = {dir.."Thunder_a.mp3",dir.."thunder_b.mp3",dir.."thunder_c.mp3",dir.."thunder_d.mp3"}
      if veh then
        playerWPos = vec3(veh:getPosition())
      else
        playerWPos = vec3(core_camera.getPosition())
      end
      local oldplayerPos = playerWPos
      playerWPos = playerWPos + vec3(randomoffset1, randomoffset2, randomoffset3)
      sounddelay = (oldplayerPos - playerWPos):length()
      sounddelay = sounddelay / 331 --sound speed in m/s
      if sounddelay < 0 then sounddelay = -sounddelay end
      local thunderbolt = scenetree.findObject("thunder_obj")
      local vfx = scenetree.findObject("thndr")
      thunderbolt:setPosition(playerWPos)
      vfx.radius = scenetree.theLevelInfo.visibleDistance / 2
      vfx:setPosition(playerWPos)
      if isPlayedA ~= true and randomoffset2 then
        hideGroup("thunder", false)
        isPlayedA = true
        delay = 0
      end
      if delay >= math.random(0.05, 0.3) and isPlayedA == true and isPlayedB ~= true then
        hideGroup("thunder", true)
        delay = 0
        isPlayedB = true
      end
      if delay >= math.random(0.05, 0.1) and isPlayedB == true and isPlayedC ~= true then
        hideGroup("thunder", false)
        delay = 0
        isPlayedC = true
      end
      if delay >= math.random(0.05, 0.3) and isPlayedC == true and isPlayedD ~= true then
        hideGroup("thunder", true)
        delay = 0
        isPlayedD = true
      end
      if delay >= sounddelay and isPlayedC == true then
        Engine.Audio.playOnce('AudioEnvironment', sndTable[math.random(1, 4)])
        thndrcnt = 0
        delay = 0
        isPlayedA = false
        isPlayedB = false
        isPlayedC = false
        isPlayedD = false
      end
    end
  end
end

--dynamic speedtrap

local dynamicSpeedDisplayData = {isActive = false, cachedTriggers = {}, subjectData = {}}

local function dynamicSpeedDisplayUpdate(triggerName, subjectID, event)
  for k,v in pairs(dynamicSpeedDisplayData.subjectData) do
    if scenetree.findObjectById(k).hidden == true or scenetree.findObjectById(k).isPakred == true then
      dynamicSpeedDisplayData.subjectData[k] = nil
    end
  end
  if event == "enter" then
    dynamicSpeedDisplayData.isActive = true
    dynamicSpeedDisplayData.subjectData[subjectID] = {triggerName = triggerName}
    dynamicSpeedDisplayData.cachedTriggers[triggerName] = true
  end
  if event == "exit" and dynamicSpeedDisplayData.subjectData[subjectID] then
    dynamicSpeedDisplayData.subjectData[subjectID] = nil
    local tempTriggers = {}
    for k,v in pairs(dynamicSpeedDisplayData.subjectData) do
      tempTriggers[v.triggerName] = true
    end
    for k,v in pairs(dynamicSpeedDisplayData.cachedTriggers) do
      if not tempTriggers[k] == true then
        dynamicSpeedDisplayReset(k)
      end
    end
  end
end

local cachedDisplays = {}

local function getDisplayNumbers(groupName)
  local displayGroup = scenetree.findObject(groupName)
  local displayTab = groupName:match("tab(%d+)")
  if not cachedDisplays[displayTab] then
    cachedDisplays[displayTab] = {}
    for i = 0, displayGroup.obj:getCount(), 1 do
      local id = displayGroup.obj:idAt(i)
      local obj = scenetree.findObjectById(id)
      if obj and obj.obj:isSubClassOf("TSStatic") and obj.name then
        local name = obj.name
        local row = name:match("row(%d+)")
        local number = name:match("row%d+_(%d+)$")
        local tab = name:match("tab(%d+)")
        if not cachedDisplays[tab][row] then cachedDisplays[tab][row] = {} end
        cachedDisplays[tab][row][number] = id
      end
    end
  end
end

local cachedTiers = {}

local function getDisplayTiers(groupName)
  local tiersGroup = scenetree.findObject(groupName)
  local tiersTab = groupName:match("tab(%d+)")
  if not cachedTiers[tiersTab] then
    cachedTiers[tiersTab] = {}
    for i = 0, tiersGroup.obj:getCount(), 1 do
      local id = tiersGroup.obj:idAt(i)
      local obj = scenetree.findObjectById(id)
      if obj and obj.obj:isSubClassOf("TSStatic") and obj.name then
        local name = obj.name
        local limit = name:match("aboveSpeed_(%d+)$")
        local tab = name:match("tablica_trigger_(%d+)")
        if not cachedTiers[tab][limit] then cachedTiers[tab][limit] = {} end
        cachedTiers[tab][limit] = id
      end
    end
  end
end

local function dynamicSpeedDisplayLevel(data)
  if data then
    for k,v in pairs(data) do
      local trigger = scenetree.findObject(k)
      if trigger then
        if trigger.speedLimit then
          local speedLimit = tonumber(trigger.speedLimit)
          if trigger.aboveLimitWarn then
            local aboveLimittxt = scenetree.findObject(trigger.aboveLimitWarn)
            if aboveLimittxt then
              if v > (speedLimit + 1) then
                if aboveLimittxt.hidden ~= false then aboveLimittxt.hidden = false end
              else
                if aboveLimittxt.hidden ~= true then aboveLimittxt.hidden = true end
              end
            end
          end
          if trigger.aboveLimitTiersGroup then
            local tiersTab = trigger.aboveLimitTiersGroup:match("tab(%d+)")
            if not cachedTiers[tiersTab] then getDisplayTiers(trigger.aboveLimitTiersGroup) end
            local speed = math.floor(v)
            if cachedTiers[tiersTab] then
              local selectedTier = nil
              for k,v in pairs(cachedTiers[tiersTab]) do
                local tier = tonumber(k)
                if speed > tier + trigger.speedLimit then
                  if selectedTier == nil or tier >= selectedTier then
                    selectedTier = tier
                  end
                end
              end
              for k,v in pairs(cachedTiers[tiersTab]) do
                if k == tostring(selectedTier) then
                  local targetObj = scenetree.findObjectById(v)
                  if targetObj then
                    if targetObj.hidden ~= false then targetObj.hidden = false end
                  end
                else
                  local targetObj = scenetree.findObjectById(v)
                  if targetObj then
                    if targetObj.hidden ~= true then targetObj.hidden = true end
                  end
                end
              end
            end
          end
          if trigger.belowLimitWarn then
            local belowLimittxt = scenetree.findObject(trigger.belowLimitWarn)
            if belowLimittxt then
              if v <= (speedLimit + 1) then
                if belowLimittxt.hidden ~= false then belowLimittxt.hidden = false end
              else
                if belowLimittxt.hidden ~= true then belowLimittxt.hidden = true end
              end
            end
          end
        end
        if trigger.displayGroup then
          local displayTab = trigger.displayGroup:match("tab(%d+)")
          if not cachedDisplays[displayTab] then getDisplayNumbers(trigger.displayGroup) end
          local speed = math.floor(v)
          local maxSpeed = 199
          if trigger.maxSpeed then maxSpeed = tonumber(trigger.maxSpeed) end
          if speed > maxSpeed then speed = maxSpeed end
          local speedString = tostring(speed)
          local digits = {}
          for i = 1, #speedString do
            table.insert(digits, tonumber(speedString:sub(i, i)))
          end
          local digitsAmnt = table.getn(digits)
          if digitsAmnt == 1 then
            if cachedDisplays[displayTab]["3"] and cachedDisplays[displayTab]["3"][tostring(digits[1])] then
              local target = cachedDisplays[displayTab]["3"][tostring(digits[1])]
              local targetObj = scenetree.findObjectById(target)
              if targetObj then
                if targetObj.hidden ~= false then targetObj.hidden = false end
                for k,v in pairs(cachedDisplays[displayTab]) do
                  for k,v in pairs(v) do
                    if v ~= target then
                      local disableTarget = scenetree.findObjectById(v)
                      if disableTarget then
                        if disableTarget.hidden ~= true then disableTarget.hidden = true end
                      end
                    end
                  end
                end
              end
            end
          end
          if digitsAmnt == 2 then
            if cachedDisplays[displayTab]["2"] and cachedDisplays[displayTab]["2"][tostring(digits[1])] and cachedDisplays[displayTab]["3"] and cachedDisplays[displayTab]["3"][tostring(digits[1])] then
              local target_1 = cachedDisplays[displayTab]["2"][tostring(digits[1])]
              local target_2 = cachedDisplays[displayTab]["3"][tostring(digits[2])]
              local targetObj_1 = scenetree.findObjectById(target_1)
              local targetObj_2 = scenetree.findObjectById(target_2)
              if targetObj_1 and targetObj_2 then
                if targetObj_1.hidden ~= false then targetObj_1.hidden = false end
                if targetObj_2.hidden ~= false then targetObj_2.hidden = false end
                for k,v in pairs(cachedDisplays[displayTab]) do
                  for k,v in pairs(v) do
                    if v ~= target_1 and v ~= target_2 then
                      local disableTarget = scenetree.findObjectById(v)
                      if disableTarget then
                        if disableTarget.hidden ~= true then disableTarget.hidden = true end
                      end
                    end
                  end
                end
              end
            end
          end
          if digitsAmnt == 3 then
            if cachedDisplays[displayTab]["1"] and cachedDisplays[displayTab]["1"][tostring(digits[1])] and cachedDisplays[displayTab]["2"] and cachedDisplays[displayTab]["2"][tostring(digits[1])] and cachedDisplays[displayTab]["3"] and cachedDisplays[displayTab]["3"][tostring(digits[1])] then
              local target_1 = cachedDisplays[displayTab]["1"][tostring(digits[1])]
              local target_2 = cachedDisplays[displayTab]["2"][tostring(digits[2])]
              local target_3 = cachedDisplays[displayTab]["3"][tostring(digits[3])]
              local targetObj_1 = scenetree.findObjectById(target_1)
              local targetObj_2 = scenetree.findObjectById(target_2)
              local targetObj_3 = scenetree.findObjectById(target_3)
              if targetObj_1 and targetObj_2 and targetObj_3 then
                if targetObj_1.hidden ~= false then targetObj_1.hidden = false end
                if targetObj_2.hidden ~= false then targetObj_2.hidden = false end
                if targetObj_3.hidden ~= false then targetObj_3.hidden = false end
                for k,v in pairs(cachedDisplays[displayTab]) do
                  for k,v in pairs(v) do
                    if v ~= target_1 and v ~= target_2 and v ~= target_3 then
                      local disableTarget = scenetree.findObjectById(v)
                      if disableTarget then
                        if disableTarget.hidden ~= true then disableTarget.hidden = true end
                      end
                    end
                  end
                end
              end
            end
          end
        end
      end
    end
  end
end

local function dynamicSpeedDisplayCalculate(data)
  local mapLocale = 'metric'
  local processedData = {}
  if data and data.subjectData then
    local count = 0
    for k,v in pairs(data.subjectData) do
      local dir = vec3()
      local veh = be:getObjectByID(k)
      -- to do, maybe add cef notification support?
      -- local user = veh:getDynDataFieldbyName("licenseText", 0)
      local speedVec = vec3(veh:getVelocity(veh:getRefNodeId()) )
      local trigggerRot = quat(scenetree.findObject(v.triggerName):getRotation())
      dir:setRotate(trigggerRot, vec3(1,0,0))
      local speed = dir:dot(speedVec)
      if speed < 0 then speed = speed*-1 end
      local speedConverted = speed*3.6
      count = count + 1
      if mapLocale ~= 'metric' then speedConverted = speed*2.23694 end
      processedData[v.triggerName] = speedConverted
    end
    if count == 0 then
      data.isActive = false
      dynamicSpeedDisplayData.isActive = false
    end
    if not tableIsEmpty(processedData) then dynamicSpeedDisplayLevel(processedData) end
  end
end

local bunkierTable = {BunkierTrigger_1played = false, BunkierTrigger_2played = false, BunkierTrigger_3played = false,
BunkierTrigger_51played = false, BunkierTrigger_52played = false, BunkierTrigger_6played = false, BunkierTrigger_7played = false}

local function onBeamNGTrigger(data)
  if data.triggerName == "BunkierTrigger_1" and data.event == "enter" and bunkierTable.BunkierTrigger_1played == false then
    local veh = be:getPlayerVehicleID(0)
    if data.subjectID == veh then
      Engine.Audio.playOnce('AudioEnvironment', '/levels/polish_roads_v2/SketchUpCK/Buner/Ambienty/nr_1.mp3')
      bunkierTable.BunkierTrigger_1played = true
    end
  end
  if data.triggerName == "BunkierTrigger_2" and data.event == "enter" and bunkierTable.BunkierTrigger_2played == false then
    local veh = be:getPlayerVehicleID(0)
    if data.subjectID == veh then
      Engine.Audio.playOnce('AudioEnvironment', '/levels/polish_roads_v2/SketchUpCK/Buner/Ambienty/nr_2.mp3')
      bunkierTable.BunkierTrigger_2played = true
    end
  end
  if data.triggerName == "BunkierTrigger_3" and data.event == "enter" and bunkierTable.BunkierTrigger_3played == false then
    local veh = be:getPlayerVehicleID(0)
    if data.subjectID == veh then
      Engine.Audio.playOnce('AudioEnvironment', '/levels/polish_roads_v2/SketchUpCK/Buner/Ambienty/nr_3.mp3')
      bunkierTable.BunkierTrigger_3played = true
    end
  end
  if data.triggerName == "BunkierTrigger_51" and data.event == "enter" and bunkierTable.BunkierTrigger_51played == false then
    local veh = be:getPlayerVehicleID(0)
    if data.subjectID == veh then
      Engine.Audio.playOnce('AudioEnvironment', '/levels/polish_roads_v2/SketchUpCK/Buner/Ambienty/nr_4.mp3')
      bunkierTable.BunkierTrigger_51played = true
    end
  end
  if data.triggerName == "BunkierTrigger_52" and data.event == "enter" and bunkierTable.BunkierTrigger_52played == false then
    local veh = be:getPlayerVehicleID(0)
    if data.subjectID == veh then
      Engine.Audio.playOnce('AudioEnvironment', '/levels/polish_roads_v2/SketchUpCK/Buner/Ambienty/nr_5_2.mp3')
      bunkierTable.BunkierTrigger_52played = true
    end
  end
  if data.triggerName == "BunkierTrigger_6" and data.event == "enter" and bunkierTable.BunkierTrigger_6played == false then
    local veh = be:getPlayerVehicleID(0)
    if data.subjectID == veh then
      Engine.Audio.playOnce('AudioEnvironment', '/levels/polish_roads_v2/SketchUpCK/Buner/Ambienty/nr_6.mp3')
      bunkierTable.BunkierTrigger_6played = true
    end
  end
  if data.triggerName == "BunkierTrigger_7" and data.event == "enter" and bunkierTable.BunkierTrigger_7played == false then
    local veh = be:getPlayerVehicleID(0)
    if data.subjectID == veh then
      local obj = scenetree.findObject("Krzyz_animacja")
      if obj and scenetree.Pochodnie_emittery_swiatla and scenetree.spotKrzyz then
        setAllLightsEnabled(scenetree.Pochodnie_emittery_swiatla, false)
        onParticleActivity(scenetree.Pochodnie_emittery_swiatla, false)
        setAllLightsEnabled(scenetree.spotKrzyz, true)
        bunkierTable.BunkierTrigger_7played = true
        obj:playAnim('krzyz_upada', false)
        Engine.Audio.playOnce('AudioEnvironment', '/levels/polish_roads_v2/SketchUpCK/Buner/Ambienty/ambient_gdy_krzyz_bedzie_spadal.mp3')
      else
        print("Object is missing")
      end
    end
  end
  if data.triggerName == "BunkierTrigger_cubemap" then
    local veh = be:getPlayerVehicleID(0)
    if data.subjectID == veh then
      if data.event == "enter" then
        setCubemap('cubemap_polish_roads_bunkier_reflection')
      end
      if data.event == "exit" then
        setCubemap('cubemap_polish_roads_remaster_reflection')
      end
    end
  end
  if data.triggerName and string.startswith(data.triggerName, "tablica_trigger_") then
    --sleeping and parked vehicles needs to be filtered
    local subject = scenetree.findObjectById(data.subjectID)
    if subject and subject.hidden ~= true  and data.event == "enter" then
      if subject.isPakred ~= true then
        dynamicSpeedDisplayUpdate(data.triggerName, data.subjectID, data.event)
      end
    elseif data.event == "exit" then
      dynamicSpeedDisplayUpdate(data.triggerName, data.subjectID, data.event)
    end
  end
end

local function renderImguiDebug()
  im.Begin(debugTitle, showUI, im.WindowFlags_AlwaysAutoResize)
  im.Text("Needs World Editor to run")
  if editor and editor.isEditorActive and editor.isEditorActive() then
    im.PushItemWidth(im.GetContentRegionAvailWidth())
    im.Text("Speedtraps scenetree group")
    if im.InputText("##speedtrpgroup", speedtrpgroup) then
    end
    im.Text("Speedzones scenetree group")
    if im.InputText("##speedzonegroup", speedzonegroup) then
    end
    im.Text("Cubemaps scenetree group")
    if im.InputText("##cubemapgroup", cubemapgroup) then
    end
    im.PopItemWidth()
    if im.Button("Generate waypoints data") then
      local speedtrpgroup = ffi.string(speedtrpgroup)
      local trapgroup = scenetree[speedtrpgroup]

      local speedzonegroup = ffi.string(speedzonegroup)
      local zonegroup = scenetree[speedzonegroup]

      local cubemapgroup = ffi.string(cubemapgroup)
      local cubegroup = scenetree[cubemapgroup]

      local trapwaypoints = {}
      local zonewaypoints = {}
      local cubemapwaypoints = {}

      if trapgroup and speedtrpgroup then
        for i = 0, trapgroup.obj:getCount(), 1 do
          local id = trapgroup.obj:idAt(i)
          local wpobj = scenetree.findObjectById(id)
          local wp = wpobj:getName()
          if wpobj:isSubClassOf("BeamNGTrigger") then
            table.insert(trapwaypoints, wp)
          end
        end
      end

      if zonegroup and speedzonegroup then
        for i = 0, zonegroup.obj:getCount(), 1 do
          local id = zonegroup.obj:idAt(i)
          local wpobj = scenetree.findObjectById(id)
          local wp = wpobj:getName()
          if wpobj:isSubClassOf("BeamNGTrigger") then
            table.insert(zonewaypoints, wp)
          end
        end
      end

      if cubegroup and cubemapgroup then
        for i = 0, cubegroup.obj:getCount(), 1 do
          local id = cubegroup.obj:idAt(i)
          local wpobj = scenetree.findObjectById(id)
          local wp = wpobj:getName()
          if wpobj:isSubClassOf("BeamNGTrigger") then
            table.insert(cubemapwaypoints, wp)
          end
        end
      end

        local jsonData = {}
        jsonData.levelname = levelname
        jsonData.directory = levelfolder
        jsonData.speedtraps = {}
        jsonData.speedzones = {}
        jsonData.cubemaps = {}
        jsonData.version = tool_version
        for k,v in pairs(trapwaypoints) do
          jsonData.speedtraps[v] = {}
          jsonData.speedtraps[v].triggerName = v
          jsonData.speedtraps[v].name = "Replace with name"
          jsonData.speedtraps[v].speedLimit = 60
        end
        for k,v in pairs(zonewaypoints) do
          jsonData.speedzones[v] = {}
          jsonData.speedzones[v].triggerName = v
          jsonData.speedzones[v].name = "Replace with name"
          jsonData.speedzones[v].speedLimit = 60
        end
        for k,v in pairs(cubemapwaypoints) do
          jsonData.cubemaps[v] = {}
          jsonData.cubemaps[v].triggerName = v
          jsonData.cubemaps[v].cubemapname = "cubemap_"..levelname.."_reflection"
        end
        dump(jsonData)
        extensions.editor_fileDialog.saveFile(
          function(data)
            local dir, filename, ext = path.splitWithoutExt(data.filepath, true)
            local saveData = jsonData
            saveData.name = filename
            jsonWriteFile(data.filepath, saveData, true)
            log("I","saveData","Saved Waypoint Data to " .. data.filepath)
          end,
          {{"Waypoint Data",".waypoint_data.json"}}, false, 'levels/'..levelname)
        end
      else
        if im.Button("Enable all features") then
          editor.toggleActive()
        end
      end
      if im.Button("Enable Emissive") then
        enabledisableEmissionGlow(1)
      end
      if im.Button("Add tunnel triggers") then
        local count = 0
        local tunnelZone = scenetree.Zones
        local missionGroup = scenetree.MissionGroup
        if tunnelZone then
          for i = 0, tunnelZone.obj:getCount(), 1 do
            local id = tunnelZone.obj:idAt(i)
            local obj = scenetree.findObjectById(id)
            if obj and obj:getClassName() == "Zone" then
              count = count + 1
              local position = obj:getPosition()
              local rotation = obj:getRotation()
              local scale = obj:getScale()
              local createTrigger
              createTrigger = createObject('BeamNGTrigger')
              createTrigger:setPosRot(position.x, position.y, position.z, rotation.x, rotation.y, rotation.z, rotation.w)
              createTrigger:setScale(scale)
              createTrigger:registerObject("TunnelTrigger_"..count)
              missionGroup:addObject(createTrigger)
            end
          end
        end
      end
      if im.Button("Light generator") then
        local count = 0
        local AllObjects = scenetree.getAllObjects()
        local missionGroup = scenetree.MissionGroup
        --[[if AllObjects and missionGroup then
          for k,name in pairs(AllObjects) do
            local object = scenetree.findObject(name)
            if object:isSubClassOf("TSStatic") then
              if string.endswith(object:getModelFile(), '8A_66_60_BA.dae') or string.endswith(object:getModelFile(), '71_5E_5C_AF.dae') or string.endswith(object:getModelFile(), 'DD_91_E0_0C.dae') then
                local position = object:getWorldBoxCenter()
                local createLight
                print(position)
                count = count + 1
                createLight = createObject('PointLight')
                createLight:setPosition(position)
                createLight:registerObject("StreetLightH_"..count)
                missionGroup:addObject(createLight)
              end
            end
          end
        end]]--
      end
      if im.Button("Light optimizer") then
        local lights = {}
        local count = 0
        local AllObjects = scenetree.getAllObjects()
        if AllObjects then
          for k,name in pairs(AllObjects) do
            local object = scenetree.findObject(name)
            if object:isSubClassOf("LightBase") then
              local position = object:getPosition()
              if position then
                local posX = tonumber(string.format("%.1f", position.x))
                local posY =  tonumber(string.format("%.1f", position.y))
                local newPos = tostring(vec3(posX, posY, 0))
                if lights[newPos] == true then
                  print(object:getName())
                  object:deleteObject()
                end
                lights[newPos] = true
              end
            end
          end
        end
      end
      if im.Button("Get lights amount") then
        local count = 0
        local AllObjects = scenetree.getAllObjects()
        if AllObjects then
          for k,name in pairs(AllObjects) do
            local object = scenetree.findObject(name)
            if object:isSubClassOf("LightBase") then
              count = count +1
            end
          end
          print('There is '..count..' lights in the level')
        end
      end
      if im.Button("Slap Delay Generator") then
        local positions = {}
        local count = 0
        local AllObjects = scenetree.getAllObjects()
        local missionGroup = scenetree.MissionGroup
        if AllObjects and missionGroup then
          for k,name in pairs(AllObjects) do
            local object = scenetree.findObject(name)
            if object:isSubClassOf("SpotLight") then
              local position = object:getPosition()
              if not tableIsEmpty(positions) then
                for k,v in pairs(positions) do if (v - position):length() < 40  then goto skipObject end end
              end
              local createDelay
              count = count + 1
              createDelay = createObject('SFXEmitter')
              createDelay:setPosition(position)
              createDelay:setField('track',0, 'event:>Special>Slap Delays>Delay_10')
              createDelay:registerObject("SlapDelay10_"..count)
              missionGroup:addObject(createDelay)
              table.insert(positions, position)
            end
            ::skipObject::
          end
        end
        print("Done. Generated: "..count.." objects")
      end
    im.End()
end

local function renderImguiMain()
  im.Begin(appTitle, showUI, im.WindowFlags_AlwaysAutoResize)
  if TorqueScriptLua then
    if levelname == nil then
      im.Text("No init, please restart the game")
      im.Text("Some features are not available")
      im.Text("")
      if im.Button("Fix level lua without restarting") then
        extensions.mainLevel.onClientStartMission()
      end
      im.Text("")
    else
      im.Text("Change weather")
      --fog, color, sun, ambient, fog density, group, rain, volume of rain
      --Road roughness and ground type
      if im.Button("Set Clear (Default)") then
        isThunder = false
        setWeather(levelfolder .. "/art/skies/sky_gradients/default/gradient_fog.png", levelfolder .. "/art/skies/sky_gradients/default/gradient_colorize.png", levelfolder .. "/art/skies/sky_gradients/default/gradient_sunscale.png", levelfolder .. "/art/skies/sky_gradients/default/gradient_ambient.png", 0.001, 1200, scenetree.Sky, 0, 0, false, nil)
        if isWet == 1 then
          setDryWetMaterials(1, "ASPHALT")
          hideGroup("ck_rainroad", true)
          fixDecalRoad(scenetree.ck_rainroad)
          isWet = 0
          elseif isWet == 0 then
        end
        hideGroup("thunder", true)
      end

      im.SameLine()
      if im.Button("Set Wet (Clear)") then
        isThunder = false
        setWeather(levelfolder .. "/art/skies/sky_gradients/default/gradient_fog.png", levelfolder .. "/art/skies/sky_gradients/default/gradient_colorize.png", levelfolder .. "/art/skies/sky_gradients/default/gradient_sunscale.png", levelfolder .. "/art/skies/sky_gradients/default/gradient_ambient.png", 0.001, 1200, scenetree.Sky, 0, 0, false, nil)
        if isWet == 0 then
          setDryWetMaterials(0.4, "ASPHALT_WET")
          hideGroup("ck_rainroad", false)
          fixDecalRoad(scenetree.ck_rainroad)
          isWet = 1
          elseif isWet == 1 then
        end
        hideGroup("thunder", true)
      end

      if im.Button("Set Overcast") then
        isThunder = false
        setWeather(levelfolder .. "/art/skies/sky_gradients/overcast/gradient_fog.png", levelfolder .. "/art/skies/sky_gradients/overcast/gradient_colorize.png", levelfolder .. "/art/skies/sky_gradients/overcast/gradient_sunscale.png", levelfolder .. "/art/skies/sky_gradients/overcast/gradient_ambient.png", 0.003, 1200, scenetree.Sky, 0, 0, false, nil)
        if isWet == 1 then
          setDryWetMaterials(1, "ASPHALT")
          hideGroup("ck_rainroad", true)
          fixDecalRoad(scenetree.ck_rainroad)
          isWet = 0
          elseif isWet == 0 then
        end
        hideGroup("thunder", true)
      end

      im.SameLine()
      if im.Button("Set Foggy") then
        isThunder = false
        setWeather(levelfolder .. "/art/skies/sky_gradients/foggy/gradient_fog.png", levelfolder .. "/art/skies/sky_gradients/foggy/gradient_colorize.png", levelfolder .. "/art/skies/sky_gradients/foggy/gradient_sunscale.png", levelfolder .. "/art/skies/sky_gradients/foggy/gradient_ambient.png", 0.010, 0, scenetree.Sky, 0, 0, false, nil)
        if isWet == 1 then
          setDryWetMaterials(1, "ASPHALT")
          hideGroup("ck_rainroad", true)
          fixDecalRoad(scenetree.ck_rainroad)
          isWet = 0
          elseif isWet == 0 then
        end
        hideGroup("thunder", true)
      end
      if im.Button("Set Rain") then
        isThunder = false
        setWeather(levelfolder .. "/art/skies/sky_gradients/overcast/gradient_fog.png", levelfolder .. "/art/skies/sky_gradients/overcast/gradient_colorize.png", levelfolder .. "/art/skies/sky_gradients/overcast/gradient_sunscale.png", levelfolder .. "/art/skies/sky_gradients/overcast/gradient_ambient.png", 0.005, 0, scenetree.Sky, 2048, 0.6, false, nil)
        if isWet == 0 then
          setDryWetMaterials(0.4, "ASPHALT_WET")
          hideGroup("ck_rainroad", false)
          fixDecalRoad(scenetree.ck_rainroad)
          isWet = 1
          elseif isWet == 1 then
        end
        hideGroup("thunder", true)
      end
      im.SameLine()
      if im.Button("Set Thunder") then
        isThunder = true
        setWeather(levelfolder .. "/art/skies/sky_gradients/thunder/gradient_fog.png", levelfolder .. "/art/skies/sky_gradients/thunder/gradient_colorize.png", levelfolder .. "/art/skies/sky_gradients/thunder/gradient_sunscale.png", levelfolder .. "/art/skies/sky_gradients/thunder/gradient_ambient.png", 0.015, 0, scenetree.Sky, 2048*1.5, 1, true, "cloudy-original")
        if isWet == 0 then
          setDryWetMaterials(0.4, "ASPHALT_WET")
          hideGroup("ck_rainroad", false)
          fixDecalRoad(scenetree.ck_rainroad)
          isWet = 1
          elseif isWet == 1 then
        end
      end
      im.Separator()
    end
    if im.TreeNode1("Advanced settings") then
      im.Separator()
      local dbgEnabled = im.BoolPtr(debugEnabled)
      if im.Checkbox("Enable Debug", dbgEnabled) then
        debugEnabled = dbgEnabled[0]
      end
      im.TreePop()
    end
  end
  im.End()
end

local cubevalue = false

local dispDelay = 0

local function onUpdate(dtReal, dtSim)
  if luaLoaded == false then
    extensions.mainLevel.onClientStartMission()
  end

  if gameplay_traffic and gameplay_traffic.getNumOfTraffic() > 0 then
    if gameplay_traffic.getTrafficVars() and gameplay_traffic.getTrafficVars().baseAggression and gameplay_traffic.getTrafficVars().baseAggression ~= 0.6 then
      gameplay_traffic.setTrafficVars({baseAggression = 0.6})
    end
  end

  if not TorqueScriptLua then
    TorqueScriptLua = TorqueScript
  end

  tod = scenetree.tod

  local dont_showUI
  if freeroam_bigMapMode then
    dont_showUI = freeroam_bigMapMode.bigMapActive() == true or photoModeOpen
  end

  if not dont_showUI and skipImgui == false then
    --imgui app
   --renderImguiMain()
    if debugEnabled == true then
      --renderImguiDebug()
    end
  end
  dispDelay = dispDelay + dtSim
  if dynamicSpeedDisplayData.isActive == true and dispDelay > 0.5 then
    dynamicSpeedDisplayCalculate(dynamicSpeedDisplayData)
    dispDelay = 0
  end

  delay = delay + dtSim

  --check for TOD
  if not tod then return end
  local value = (tod.time > 0.25 and tod.time < 0.745)
  local cameraCell = getCellIndexForPosition(core_camera.getPosition())

  if (cameraCell ~= lastCameraCell or lastValue ~= value) and positionInRange(core_camera.getPosition()) then
    local xCell, zCell = getCellForPosition(core_camera.getPosition())
    enableLights(xCell, zCell, value)
  end

  lastCameraCell = cameraCell
  lastValue = value


  if tod.time > 0.25 and tod.time < 0.745 then
    if cubevalue == false then
      enabledisableEmissionGlow(1)
      cubemapname = "cubemap_polish_roads_night_reflection"
      setCubemap(cubemapname)
      print("night")
      cubevalue = true
      hideGroup("godraypr", false)
    end
    value = true
  else
    if cubevalue == true then
      enabledisableEmissionGlow(0)
      cubemapname = "cubemap_polish_roads_remaster_reflection"
      setCubemap(cubemapname)
      print("day")
      cubevalue = false
      hideGroup("godraypr", true)
    end
  end
end

local function onEditorActivated()
  log('I', "onEditorActivated", "Restoring default map state to avoid saving broken data")
  enabledisableEmissionGlow(1)
  hideGroup("godraypr", true)
  core_environment.setTimeOfDay(tod)
end

local function onEditorDeactivated()
  log('I', "onEditorDeactivated", "Restoring dynamic map state")
  if tod.time > 0.25 and tod.time < 0.745 then
    enabledisableEmissionGlow(1)
    print("night")
    cubevalue = true
    hideGroup("godraypr", false)
  else
    enabledisableEmissionGlow(0)
    print("day")
    cubevalue = false
    hideGroup("godraypr", true)
  end
  core_environment.setTimeOfDay(tod)
end


local function onEditorInitialized()
  editor.addWindowMenuItem("Map Controller", onWindowMenuItem, {groupMenuName = 'Car_Killer Addons'})
  editor.registerWindow(toolWindowName, im.ImVec2(500, 200))
end


M.onClientStartMission = onClientStartMission
M.onClientPostStartMission = onClientPostStartMission
M.onUiChangedState = onUiChangedState
M.onBeamNGTrigger = onBeamNGTrigger
M.onUpdate = onUpdate
M.onEditorActivated = onEditorActivated
M.onEditorDeactivated = onEditorDeactivated
M.onEditorInitialized = onEditorInitialized

return M